<?php

declare(strict_types=1);

namespace Drupal\crm;

/**
 * Interface for the relationship statistics service.
 */
interface RelationshipStatisticsServiceInterface {

  /**
   * Increments the count for a relationship type key on a contact.
   *
   * If the type key doesn't exist, creates a new entry with count=1.
   *
   * @param \Drupal\crm\CrmContactInterface $contact
   *   The contact entity.
   * @param string $type_key
   *   The relationship type key (e.g., "friends" or "parent_child:a").
   */
  public function increment(CrmContactInterface $contact, string $type_key): void;

  /**
   * Decrements the count for a relationship type key on a contact.
   *
   * If the count reaches 0, the entry is removed from the field.
   *
   * @param \Drupal\crm\CrmContactInterface $contact
   *   The contact entity.
   * @param string $type_key
   *   The relationship type key (e.g., "friends" or "parent_child:a").
   */
  public function decrement(CrmContactInterface $contact, string $type_key): void;

  /**
   * Recalculates all relationship statistics for a single contact.
   *
   * Queries all active relationships for the contact and rebuilds the
   * statistics field from scratch.
   *
   * @param \Drupal\crm\CrmContactInterface $contact
   *   The contact entity.
   */
  public function recalculateForContact(CrmContactInterface $contact): void;

  /**
   * Recalculates relationship statistics for all contacts.
   *
   * Processes all contacts in batches.
   *
   * @param int $batch_size
   *   The number of contacts to process per batch.
   *
   * @return int
   *   The total number of contacts processed.
   */
  public function recalculateAll(int $batch_size = 100): int;

  /**
   * Gets the type key for a relationship and position.
   *
   * For symmetrical relationship, return the machine name.
   * For asymmetrical relationship, return "machine_name:a" or "machine_name:b".
   *
   * @param \Drupal\crm\CrmRelationshipInterface $relationship
   *   The relationship entity.
   * @param string $position
   *   The position ('a' or 'b').
   *
   * @return string
   *   The type key.
   */
  public function getTypeKey(CrmRelationshipInterface $relationship, string $position): string;

}
