<?php

namespace Drupal\crm\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\crm\CrmContactInterface;
use Drupal\crm\CrmUserContactMappingSyncRelationInterface;
use Drupal\user\UserInterface;

/**
 * User contact mapping sync relation service.
 *
 * @package Drupal\crm
 */
class CrmUserContactMappingSyncRelation implements CrmUserContactMappingSyncRelationInterface {

  /**
   * Relation Storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $relationStorage;

  /**
   * Entity Storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $contactStorage;

  /**
   * Logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * Current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * Rules.
   *
   * @var \Drupal\crm\Service\CrmUserSyncRules
   */
  protected $rules;

  /**
   * Settings.
   *
   * @var \Drupal\Core\Config\Config
   */
  protected $settings;

  /**
   * EntityTypeManager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a CrmCoreUserSyncRelation object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   Logger channel.
   * @param \Drupal\user\AccountProxyInterface $current_user
   *   The current user.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    LoggerChannelInterface $logger,
    AccountProxyInterface $current_user,
  ) {
    $this->relationStorage = $entity_type_manager->getStorage('crm_user_contact_mapping');
    $this->contactStorage = $entity_type_manager->getStorage('crm_contact');
    $this->logger = $logger;
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public function getContactIdFromUserId($user_id) {
    $contact_id = NULL;

    $rids = $this->relationStorage->getQuery()
      ->accessCheck(TRUE)
      ->condition('user', $user_id)
      ->range(0, 1)
      ->execute();

    if (!empty($rids)) {
      $relation_id = reset($rids);
      $relation = $this->relationStorage->load($relation_id);
      $contact_id = $relation->getContactId();
    }

    return $contact_id;
  }

  /**
   * {@inheritdoc}
   */
  public function getUserIdFromContactId($contact_id) {
    $user_id = NULL;

    $rids = $this->relationStorage->getQuery()
      ->accessCheck(TRUE)
      ->condition('crm_contact', $contact_id)
      ->range(0, 1)
      ->execute();

    if (!empty($rids)) {
      $relation_id = reset($rids);
      $relation = $this->relationStorage->load($relation_id);
      $user_id = $relation->getUserId();
    }

    return $user_id;
  }

  /**
   * {@inheritdoc}
   */
  public function getRelationIdFromUserId($user_id) {
    $rids = $this->relationStorage->getQuery()
      ->accessCheck(TRUE)
      ->condition('user', $user_id)
      ->range(0, 1)
      ->execute();

    if (!empty($rids)) {
      return reset($rids);
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getRelationIdFromContactId($contact_id) {
    $rids = $this->relationStorage->getQuery()
      ->accessCheck(TRUE)
      ->condition('crm_contact', $contact_id)
      ->range(0, 1)
      ->execute();

    if (!empty($rids)) {
      return reset($rids);
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function relate(UserInterface $account, ?CrmContactInterface $person = NULL) {
    // No contact and $account->crm_core_no_auto_sync => no sync.
    if (empty($person) && !empty($account->crm_core_no_auto_sync)) {
      return NULL;
    }

    if (empty($person)) {
      if ($this->getContactIdFromUserId($account->id())) {
        // Account already has related contact.
        return NULL;
      }

      $contact_type = $this->rules->getContactType($account);
      if (!$contact_type) {
        // No rules configured on this type.
        return NULL;
      }

      $type = $this->entityTypeManager
        ->getStorage('crm_contact_type')
        ->load($contact_type);
      $fields = $type->getPrimaryFields();

      if ($this->settings->get('auto_sync_user_relate') && isset($fields['email']) && !empty($fields['email'])) {
        $matches = $this->contactStorage->loadByProperties([
          $fields['email'] => $account->getEmail(),
          'type' => $contact_type,
        ]);
        if (count($matches) === 1) {
          $person = reset($matches);
        }
      }

      if (empty($person)) {
        $person = $this->contactStorage->create(['type' => $contact_type]);
        $person->setOwnerId($this->currentUser->id());
        // For now we just add the name.
        $person->name->given = $account->getAccountName();

        if (isset($fields['email']) && !empty($fields['email'])) {
          $person->set($fields['email'], $account->getEmail());
        }
        $person->save();
      }
    }

    // Check if contact can be synchronized to a contact.
    if (!$this->rules->valid($account, $person)) {
      return NULL;
    }

    // Check if crm_core_user_sync relation exists for any of endpoint.
    if ($this->getContactIdFromUserId($account->id()) ||
      $this->getUserIdFromContactId($person->id())) {
      return NULL;
    }
    $storage = $this->relationStorage->getStorage('crm_user_contact_mapping');
    $relation = $storage->create();
    $relation->setUser($account);
    $relation->setContact($person);
    $relation->save();

    $this->logger->notice('User @user @uid has been synchronized to the contact @contact_id, relation @rid has been created.', [
      '@user' => $account->getDisplayName(),
      '@uid' => $account->id(),
      '@contact_id' => $person->id(),
      '@rid' => $relation->id(),
    ]);

    return $person;
  }

}
