<?php

declare(strict_types=1);

namespace Drupal\crm\Service;

use Drupal\Core\Cache\CacheTagsInvalidatorInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\crm\CrmContactInterface;
use Drupal\crm\CrmRelationshipInterface;
use Drupal\crm\RelationshipStatisticsServiceInterface;

/**
 * Service for managing relationship statistics on contacts.
 *
 * Statistics are updated directly in the database without saving the contact
 * entity, similar to how Drupal's comment module handles comment statistics.
 * This avoids creating new revisions when statistics change.
 */
class RelationshipStatisticsService implements RelationshipStatisticsServiceInterface {

  /**
   * The base table for the relationship statistics field.
   */
  protected const BASE_TABLE = 'crm_contact__relationship_statistics';

  /**
   * The revision table for the relationship statistics field.
   */
  protected const REVISION_TABLE = 'crm_contact_revision__relationship_statistics';

  /**
   * Constructs a RelationshipStatisticsService object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Cache\CacheTagsInvalidatorInterface $cacheTagsInvalidator
   *   The cache tags invalidator.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected Connection $database,
    protected CacheTagsInvalidatorInterface $cacheTagsInvalidator,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function increment(CrmContactInterface $contact, string $type_key): void {
    $this->mergeStatistic($contact, $type_key, 1);
  }

  /**
   * {@inheritdoc}
   */
  public function decrement(CrmContactInterface $contact, string $type_key): void {
    $this->mergeStatistic($contact, $type_key, -1);
  }

  /**
   * Merges a statistic value (increment or decrement).
   *
   * Updates the database directly without saving the contact entity.
   *
   * @param \Drupal\crm\CrmContactInterface $contact
   *   The contact entity.
   * @param string $type_key
   *   The relationship type key.
   * @param int $delta
   *   The amount to change (positive or negative).
   */
  protected function mergeStatistic(CrmContactInterface $contact, string $type_key, int $delta): void {
    $contact_id = (int) $contact->id();
    if (!$contact_id) {
      return;
    }

    // Get contact info needed for database operations.
    $contact_storage = $this->entityTypeManager->getStorage('crm_contact');
    $contact_storage->resetCache([$contact_id]);
    $contact = $contact_storage->load($contact_id);

    if (!$contact || !$contact->hasField('relationship_statistics')) {
      return;
    }

    $revision_id = (int) $contact->getRevisionId();
    $bundle = $contact->bundle();

    // Query current statistics from the database.
    $current_statistics = $this->loadStatisticsFromDatabase($contact_id);

    // Update the statistics.
    $found = FALSE;
    foreach ($current_statistics as $key => &$item) {
      if ($item['value'] === $type_key) {
        $item['count'] = (int) $item['count'] + $delta;
        $found = TRUE;

        // Remove entry if count reaches 0 or below.
        if ($item['count'] <= 0) {
          unset($current_statistics[$key]);
        }
        break;
      }
    }

    // If not found and incrementing, add new entry.
    if (!$found && $delta > 0) {
      $current_statistics[] = [
        'value' => $type_key,
        'count' => $delta,
      ];
    }

    // Re-index the array.
    $current_statistics = array_values($current_statistics);

    // Write updated statistics to database.
    $this->writeStatisticsToDatabase($contact_id, $revision_id, $bundle, $current_statistics);

    // Reset entity static cache so subsequent loads get fresh data.
    $contact_storage->resetCache([$contact_id]);

    // Invalidate render cache tags.
    $this->cacheTagsInvalidator->invalidateTags(['crm_contact:' . $contact_id]);
  }

  /**
   * {@inheritdoc}
   */
  public function recalculateForContact(CrmContactInterface $contact): void {
    if (!$contact->hasField('relationship_statistics')) {
      return;
    }

    $contact_id = (int) $contact->id();
    if (!$contact_id) {
      return;
    }

    $revision_id = (int) $contact->getRevisionId();
    $bundle = $contact->bundle();

    // Query all active relationships for this contact.
    $relationship_storage = $this->entityTypeManager->getStorage('crm_relationship');
    $query = $relationship_storage->getQuery()
      ->condition('status', 1)
      ->condition('contacts', $contact_id)
      ->accessCheck(FALSE);

    $relationship_ids = $query->execute();

    if (empty($relationship_ids)) {
      // Clear all statistics if no relationships exist.
      $this->writeStatisticsToDatabase($contact_id, $revision_id, $bundle, []);
      $this->entityTypeManager->getStorage('crm_contact')->resetCache([$contact_id]);
      $this->cacheTagsInvalidator->invalidateTags(['crm_contact:' . $contact_id]);
      return;
    }

    $relationships = $relationship_storage->loadMultiple($relationship_ids);
    $statistics = [];

    foreach ($relationships as $relationship) {
      // Determine which position(s) this contact occupies.
      $contacts_field = $relationship->get('contacts')->getValue();
      $contact_a_id = $contacts_field[0]['target_id'] ?? NULL;
      $contact_b_id = $contacts_field[1]['target_id'] ?? NULL;

      if ($contact_a_id == $contact_id) {
        $type_key = $this->getTypeKey($relationship, 'a');
        $statistics[$type_key] = ($statistics[$type_key] ?? 0) + 1;
      }

      if ($contact_b_id == $contact_id) {
        $type_key = $this->getTypeKey($relationship, 'b');
        $statistics[$type_key] = ($statistics[$type_key] ?? 0) + 1;
      }
    }

    // Convert to field values format.
    $field_values = [];
    foreach ($statistics as $type_key => $count) {
      $field_values[] = [
        'value' => $type_key,
        'count' => $count,
      ];
    }

    // Write to database directly.
    $this->writeStatisticsToDatabase($contact_id, $revision_id, $bundle, $field_values);

    // Reset entity static cache so subsequent loads get fresh data.
    $this->entityTypeManager->getStorage('crm_contact')->resetCache([$contact_id]);

    // Invalidate render cache tags.
    $this->cacheTagsInvalidator->invalidateTags(['crm_contact:' . $contact_id]);
  }

  /**
   * {@inheritdoc}
   */
  public function recalculateAll(int $batch_size = 100): int {
    $contact_storage = $this->entityTypeManager->getStorage('crm_contact');
    $total_processed = 0;
    $last_id = 0;

    do {
      $query = $contact_storage->getQuery()
        ->condition('id', $last_id, '>')
        ->sort('id', 'ASC')
        ->range(0, $batch_size)
        ->accessCheck(FALSE);

      $contact_ids = $query->execute();

      if (empty($contact_ids)) {
        break;
      }

      $contacts = $contact_storage->loadMultiple($contact_ids);

      foreach ($contacts as $contact) {
        $this->recalculateForContact($contact);
        $last_id = $contact->id();
        $total_processed++;
      }

      // Clear entity cache to prevent memory issues.
      $contact_storage->resetCache($contact_ids);

    } while (!empty($contact_ids));

    return $total_processed;
  }

  /**
   * {@inheritdoc}
   */
  public function getTypeKey(CrmRelationshipInterface $relationship, string $position): string {
    $bundle = $relationship->bundle();
    $relationship_type = $this->entityTypeManager
      ->getStorage('crm_relationship_type')
      ->load($bundle);

    if (!$relationship_type) {
      return $bundle;
    }

    // Check if the relationship type is asymmetric.
    $is_asymmetric = (bool) $relationship_type->get('asymmetric');

    if ($is_asymmetric) {
      return $bundle . ':' . $position;
    }

    return $bundle;
  }

  /**
   * Loads current statistics from the database.
   *
   * @param int $contact_id
   *   The contact entity ID.
   *
   * @return array
   *   An array of statistics with 'value' and 'count' keys.
   */
  protected function loadStatisticsFromDatabase(int $contact_id): array {
    $result = $this->database->select(self::BASE_TABLE, 's')
      ->fields('s', ['relationship_statistics_value', 'relationship_statistics_count'])
      ->condition('entity_id', $contact_id)
      ->orderBy('delta')
      ->execute();

    $statistics = [];
    foreach ($result as $row) {
      $statistics[] = [
        'value' => $row->relationship_statistics_value,
        'count' => (int) $row->relationship_statistics_count,
      ];
    }

    return $statistics;
  }

  /**
   * Writes statistics directly to the database.
   *
   * Updates both the base table and the revision table for the current
   * revision. Uses a transaction to ensure data integrity.
   *
   * @param int $contact_id
   *   The contact entity ID.
   * @param int $revision_id
   *   The contact revision ID.
   * @param string $bundle
   *   The contact bundle.
   * @param array $statistics
   *   An array of statistics with 'value' and 'count' keys.
   */
  protected function writeStatisticsToDatabase(int $contact_id, int $revision_id, string $bundle, array $statistics): void {
    $transaction = $this->database->startTransaction();

    try {
      // Delete existing rows from base table.
      $this->database->delete(self::BASE_TABLE)
        ->condition('entity_id', $contact_id)
        ->execute();

      // Delete existing rows from revision table for current revision.
      $this->database->delete(self::REVISION_TABLE)
        ->condition('entity_id', $contact_id)
        ->condition('revision_id', $revision_id)
        ->execute();

      // Insert new rows.
      foreach ($statistics as $delta => $item) {
        $fields = [
          'bundle' => $bundle,
          'deleted' => 0,
          'entity_id' => $contact_id,
          'revision_id' => $revision_id,
          'langcode' => 'und',
          'delta' => $delta,
          'relationship_statistics_value' => $item['value'],
          'relationship_statistics_count' => $item['count'],
        ];

        // Insert into base table.
        $this->database->insert(self::BASE_TABLE)
          ->fields($fields)
          ->execute();

        // Insert into revision table.
        $this->database->insert(self::REVISION_TABLE)
          ->fields($fields)
          ->execute();
      }
    }
    catch (\Exception $e) {
      $transaction->rollBack();
      throw $e;
    }
  }

}
