<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Functional\Form;

use Drupal\Tests\BrowserTestBase;
use Drupal\crm\Entity\ContactDetailType;

/**
 * Comprehensive functional tests for the ContactDetailTypeForm.
 *
 * @group crm
 * @covers \Drupal\crm\Form\ContactDetailTypeForm
 */
class ContactDetailTypeFormTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'crm',
    'datetime',
    'name',
    'telephone',
    'address',
  ];

  /**
   * A user with CRM administration permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * A user without CRM administration permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $normalUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->adminUser = $this->createUser([
      'administer crm',
    ]);

    $this->normalUser = $this->createUser([
      'view any crm_contact',
    ]);
  }

  /**
   * Tests access to the contact detail type forms.
   */
  public function testContactDetailTypeFormAccess(): void {
    // Test that anonymous users cannot access the form.
    $this->drupalGet('admin/structure/crm/contact-detail-types/add');
    $this->assertSession()->statusCodeEquals(403);

    // Test that normal users cannot access the form.
    $this->drupalLogin($this->normalUser);
    $this->drupalGet('admin/structure/crm/contact-detail-types/add');
    $this->assertSession()->statusCodeEquals(403);

    // Test that admin users can access the form.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/structure/crm/contact-detail-types/add');
    $this->assertSession()->statusCodeEquals(200);
  }

  /**
   * Tests creating a new contact detail type with minimal data.
   */
  public function testCreateContactDetailTypeMinimal(): void {
    $this->drupalLogin($this->adminUser);

    // Navigate to the add form.
    $this->drupalGet('admin/structure/crm/contact-detail-types/add');
    $this->assertSession()->statusCodeEquals(200);

    // Check that required form elements are present.
    $this->assertSession()->fieldExists('label');
    $this->assertSession()->fieldExists('id');
    $this->assertSession()->fieldExists('description');

    // Submit the form with minimal data.
    $edit = [
      'label' => 'Test Contact Detail Type',
      'id' => 'test_contact_detail_type',
    ];
    $this->submitForm($edit, 'Save CRM contact detail type');

    // Check that we were redirected to the collection page.
    $this->assertSession()->addressEquals('admin/structure/crm/contact-detail-types');

    // Check for success message.
    $this->assertSession()->pageTextContains('The CRM contact detail type Test Contact Detail Type has been added.');

    // Verify the entity was created.
    $contact_detail_type = ContactDetailType::load('test_contact_detail_type');
    $this->assertNotNull($contact_detail_type);
    $this->assertEquals('Test Contact Detail Type', $contact_detail_type->label());
    $this->assertEquals('', $contact_detail_type->getDescription());
  }

  /**
   * Tests creating a new contact detail type with full data.
   */
  public function testCreateContactDetailTypeComplete(): void {
    $this->drupalLogin($this->adminUser);

    $this->drupalGet('admin/structure/crm/contact-detail-types/add');

    $edit = [
      'label' => 'Complete Contact Detail Type',
      'id' => 'complete_contact_detail_type',
      'description' => 'This is a complete contact detail type with description.',
    ];
    $this->submitForm($edit, 'Save CRM contact detail type');

    $this->assertSession()->addressEquals('admin/structure/crm/contact-detail-types');
    $this->assertSession()->pageTextContains('The CRM contact detail type Complete Contact Detail Type has been added.');

    // Verify the entity was created with all data.
    $contact_detail_type = ContactDetailType::load('complete_contact_detail_type');
    $this->assertNotNull($contact_detail_type);
    $this->assertEquals('Complete Contact Detail Type', $contact_detail_type->label());
    $this->assertEquals('This is a complete contact detail type with description.', $contact_detail_type->getDescription());
  }

  /**
   * Tests editing an existing contact detail type.
   */
  public function testEditContactDetailType(): void {
    $this->drupalLogin($this->adminUser);

    // Create a contact detail type to edit.
    $contact_detail_type = ContactDetailType::create([
      'id' => 'edit_test_type',
      'label' => 'Original Label',
      'description' => 'Original description.',
    ]);
    $contact_detail_type->save();

    // Navigate to the edit form.
    $this->drupalGet('admin/structure/crm/contact-detail-types/manage/edit_test_type');
    $this->assertSession()->statusCodeEquals(200);

    // Check that the form is pre-populated.
    $this->assertSession()->fieldValueEquals('label', 'Original Label');
    $this->assertSession()->fieldValueEquals('id', 'edit_test_type');
    $this->assertSession()->fieldValueEquals('description', 'Original description.');

    // Check that the page title is set correctly.
    $this->assertSession()->pageTextContains('Edit Original Label CRM contact detail type');

    // Update the contact detail type.
    $edit = [
      'label' => 'Updated Label',
      'description' => 'Updated description.',
    ];
    $this->submitForm($edit, 'Save CRM contact detail type');

    $this->assertSession()->addressEquals('admin/structure/crm/contact-detail-types');
    $this->assertSession()->pageTextContains('The CRM contact detail type Updated Label has been updated.');

    // Verify the entity was updated.
    $updated_entity = ContactDetailType::load('edit_test_type');
    $this->assertEquals('Updated Label', $updated_entity->label());
    $this->assertEquals('Updated description.', $updated_entity->getDescription());
  }

  /**
   * Tests form validation for required fields.
   */
  public function testFormValidationRequiredFields(): void {
    $this->drupalLogin($this->adminUser);

    $this->drupalGet('admin/structure/crm/contact-detail-types/add');

    // Submit form without required label field.
    $edit = [
      'id' => 'test_validation',
      'description' => 'Test description.',
    ];
    $this->submitForm($edit, 'Save CRM contact detail type');

    // Should stay on the same page with validation error.
    $this->assertSession()->addressEquals('admin/structure/crm/contact-detail-types/add');
    $this->assertSession()->pageTextContains('Label field is required.');
  }

  /**
   * Tests form validation for machine name constraints.
   */
  public function testFormValidationMachineName(): void {
    $this->drupalLogin($this->adminUser);

    $this->drupalGet('admin/structure/crm/contact-detail-types/add');

    // Test with invalid machine name characters.
    $edit = [
      'label' => 'Test Type',
      'id' => 'invalid-machine-name!',
      'description' => 'Test description.',
    ];
    $this->submitForm($edit, 'Save CRM contact detail type');

    // Should stay on the same page with validation error.
    $this->assertSession()->addressEquals('admin/structure/crm/contact-detail-types/add');
    $this->assertSession()->pageTextContains('The machine-readable name must contain only lowercase letters, numbers, and underscores.');
  }

  /**
   * Tests form validation for duplicate machine name.
   */
  public function testFormValidationDuplicateMachineName(): void {
    $this->drupalLogin($this->adminUser);

    // Create an existing contact detail type.
    $existing_type = ContactDetailType::create([
      'id' => 'duplicate_test',
      'label' => 'Existing Type',
    ]);
    $existing_type->save();

    $this->drupalGet('admin/structure/crm/contact-detail-types/add');

    // Try to create another with the same machine name.
    $edit = [
      'label' => 'Duplicate Test Type',
      'id' => 'duplicate_test',
      'description' => 'This should fail.',
    ];
    $this->submitForm($edit, 'Save CRM contact detail type');

    // Should stay on the same page with validation error.
    $this->assertSession()->addressEquals('admin/structure/crm/contact-detail-types/add');
    $this->assertSession()->pageTextContains('The machine-readable name is already in use. It must be unique.');
  }

  /**
   * Tests the machine name field configuration.
   */
  public function testMachineNameFieldConfiguration(): void {
    $this->drupalLogin($this->adminUser);

    $this->drupalGet('admin/structure/crm/contact-detail-types/add');

    // Test that the machine name field exists and has proper attributes.
    $machine_name_field = $this->assertSession()->fieldExists('id');
    $this->assertEquals('32', $machine_name_field->getAttribute('maxlength'));

    // Test that we can manually enter a machine name.
    $edit = [
      'label' => 'Manual Machine Name Test',
      'id' => 'manual_machine_name',
      'description' => 'Testing manual machine name entry.',
    ];
    $this->submitForm($edit, 'Save CRM contact detail type');

    // Should be successful.
    $this->assertSession()->addressEquals('admin/structure/crm/contact-detail-types');
    $this->assertSession()->pageTextContains('The CRM contact detail type Manual Machine Name Test has been added.');
  }

  /**
   * Tests navigation to and from the contact detail type forms.
   */
  public function testFormNavigation(): void {
    $this->drupalLogin($this->adminUser);

    // Start from the collection page.
    $this->drupalGet('admin/structure/crm/contact-detail-types');
    $this->assertSession()->statusCodeEquals(200);

    // Navigate to the add form (either by link or direct URL).
    $this->drupalGet('admin/structure/crm/contact-detail-types/add');
    $this->assertSession()->statusCodeEquals(200);

    // Create a contact detail type.
    $edit = [
      'label' => 'Navigation Test Type',
      'id' => 'navigation_test_type',
      'description' => 'Testing navigation.',
    ];
    $this->submitForm($edit, 'Save CRM contact detail type');

    // Should be redirected back to collection page.
    $this->assertSession()->addressEquals('admin/structure/crm/contact-detail-types');

    // The new type should be listed.
    $this->assertSession()->pageTextContains('Navigation Test Type');

    // Navigate directly to the edit form.
    $this->drupalGet('admin/structure/crm/contact-detail-types/manage/navigation_test_type');
    $this->assertSession()->statusCodeEquals(200);

    // Form should be pre-populated.
    $this->assertSession()->fieldValueEquals('label', 'Navigation Test Type');
    $this->assertSession()->fieldValueEquals('description', 'Testing navigation.');
  }

  /**
   * Tests form behavior with special characters in fields.
   */
  public function testFormWithSpecialCharacters(): void {
    $this->drupalLogin($this->adminUser);

    $this->drupalGet('admin/structure/crm/contact-detail-types/add');

    // Test with special characters in label and description.
    $edit = [
      'label' => 'Special & Characters "Type"',
      'id' => 'special_characters_type',
      'description' => 'Description with special characters: & < > " \' and unicode: é ñ ü',
    ];
    $this->submitForm($edit, 'Save CRM contact detail type');

    $this->assertSession()->addressEquals('admin/structure/crm/contact-detail-types');
    $this->assertSession()->pageTextContains('The CRM contact detail type Special & Characters "Type" has been added.');

    // Verify the entity was created with special characters preserved.
    $contact_detail_type = ContactDetailType::load('special_characters_type');
    $this->assertNotNull($contact_detail_type);
    $this->assertEquals('Special & Characters "Type"', $contact_detail_type->label());
    $this->assertEquals('Description with special characters: & < > " \' and unicode: é ñ ü', $contact_detail_type->getDescription());
  }

  /**
   * Tests maximum length constraints on form fields.
   */
  public function testFormFieldLengthConstraints(): void {
    $this->drupalLogin($this->adminUser);

    $this->drupalGet('admin/structure/crm/contact-detail-types/add');

    // Test with very long machine name (exceeding 32 character limit).
    $long_machine_name = str_repeat('a', 35);
    $edit = [
      'label' => 'Long Machine Name Test',
      'id' => $long_machine_name,
      'description' => 'Testing length constraints.',
    ];
    $this->submitForm($edit, 'Save CRM contact detail type');

    // Should stay on the same page with validation error.
    $this->assertSession()->addressEquals('admin/structure/crm/contact-detail-types/add');
    $this->assertSession()->pageTextContains('cannot be longer than 32 characters');
  }

  /**
   * Tests the delete functionality.
   */
  public function testDeleteContactDetailType(): void {
    $this->drupalLogin($this->adminUser);

    // Create a contact detail type to delete.
    $contact_detail_type = ContactDetailType::create([
      'id' => 'delete_test_type',
      'label' => 'Delete Test Type',
      'description' => 'This will be deleted.',
    ]);
    $contact_detail_type->save();

    // Navigate directly to the delete form.
    $this->drupalGet('admin/structure/crm/contact-detail-types/manage/delete_test_type/delete');
    $this->assertSession()->statusCodeEquals(200);

    // Should show the confirmation page.
    $this->assertSession()->pageTextContains('Are you sure you want to delete the crm contact detail type Delete Test Type?');

    // Confirm deletion.
    $this->submitForm([], 'Delete');

    // Should be redirected to collection page.
    $this->assertSession()->addressEquals('admin/structure/crm/contact-detail-types');
    $this->assertSession()->pageTextContains('deleted');

    // Verify the entity was deleted.
    $deleted_entity = ContactDetailType::load('delete_test_type');
    $this->assertNull($deleted_entity);
  }

  /**
   * Tests editing access to existing contact detail types.
   */
  public function testEditContactDetailTypeAccess(): void {
    // Create a contact detail type.
    $contact_detail_type = ContactDetailType::create([
      'id' => 'access_test_type',
      'label' => 'Access Test Type',
    ]);
    $contact_detail_type->save();

    // Test that anonymous users cannot access the edit form.
    $this->drupalGet('admin/structure/crm/contact-detail-types/manage/access_test_type');
    $this->assertSession()->statusCodeEquals(403);

    // Test that normal users cannot access the edit form.
    $this->drupalLogin($this->normalUser);
    $this->drupalGet('admin/structure/crm/contact-detail-types/manage/access_test_type');
    $this->assertSession()->statusCodeEquals(403);

    // Test that admin users can access the edit form.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/structure/crm/contact-detail-types/manage/access_test_type');
    $this->assertSession()->statusCodeEquals(200);
  }

  /**
   * Tests the form structure and field properties.
   */
  public function testFormStructureAndFields(): void {
    $this->drupalLogin($this->adminUser);

    $this->drupalGet('admin/structure/crm/contact-detail-types/add');

    // Test label field properties.
    $label_field = $this->assertSession()->fieldExists('label');
    $this->assertTrue($label_field->hasAttribute('required'));

    // Test machine name field properties.
    $id_field = $this->assertSession()->fieldExists('id');
    $this->assertEquals('32', $id_field->getAttribute('maxlength'));

    // Test description field.
    $description_field = $this->assertSession()->fieldExists('description');
    $this->assertEquals('textarea', $description_field->getTagName());

    // Test form actions.
    $this->assertSession()->buttonExists('Save CRM contact detail type');
  }

  /**
   * Tests form behavior when editing with pre-existing contact details.
   */
  public function testEditFormWithExistingContactDetails(): void {
    $this->drupalLogin($this->adminUser);

    // Create a contact detail type.
    $contact_detail_type = ContactDetailType::create([
      'id' => 'in_use_type',
      'label' => 'In Use Type',
      'description' => 'This type is in use.',
    ]);
    $contact_detail_type->save();

    // Navigate to edit form.
    $this->drupalGet('admin/structure/crm/contact-detail-types/manage/in_use_type');
    $this->assertSession()->statusCodeEquals(200);

    // The machine name field should be disabled for existing entities.
    $id_field = $this->assertSession()->fieldExists('id');
    $this->assertTrue($id_field->hasAttribute('disabled') || $id_field->hasAttribute('readonly'));

    // Update the label and description.
    $edit = [
      'label' => 'Updated In Use Type',
      'description' => 'Updated description for in use type.',
    ];
    $this->submitForm($edit, 'Save CRM contact detail type');

    $this->assertSession()->addressEquals('admin/structure/crm/contact-detail-types');
    $this->assertSession()->pageTextContains('The CRM contact detail type Updated In Use Type has been updated.');

    // Verify the entity was updated but ID remained the same.
    $updated_entity = ContactDetailType::load('in_use_type');
    $this->assertEquals('Updated In Use Type', $updated_entity->label());
    $this->assertEquals('Updated description for in use type.', $updated_entity->getDescription());
  }

  /**
   * Tests form validation with empty label.
   */
  public function testFormValidationEmptyLabel(): void {
    $this->drupalLogin($this->adminUser);

    $this->drupalGet('admin/structure/crm/contact-detail-types/add');

    // Submit form with empty label.
    $edit = [
      'label' => '',
      'id' => 'empty_label_test',
      'description' => 'Testing empty label validation.',
    ];
    $this->submitForm($edit, 'Save CRM contact detail type');

    // Should stay on the form with validation error.
    $this->assertSession()->addressEquals('admin/structure/crm/contact-detail-types/add');
    $this->assertSession()->pageTextContains('Label field is required.');

    // Verify no entity was created.
    $contact_detail_type = ContactDetailType::load('empty_label_test');
    $this->assertNull($contact_detail_type);
  }

  /**
   * Tests that the collection page shows created contact detail types.
   */
  public function testCollectionPageShowsCreatedTypes(): void {
    $this->drupalLogin($this->adminUser);

    // Create multiple contact detail types.
    $types = [
      ['id' => 'type_one', 'label' => 'Type One'],
      ['id' => 'type_two', 'label' => 'Type Two'],
      ['id' => 'type_three', 'label' => 'Type Three'],
    ];

    foreach ($types as $type_data) {
      $contact_detail_type = ContactDetailType::create($type_data);
      $contact_detail_type->save();
    }

    // Visit the collection page.
    $this->drupalGet('admin/structure/crm/contact-detail-types');
    $this->assertSession()->statusCodeEquals(200);

    // All types should be listed.
    foreach ($types as $type_data) {
      $this->assertSession()->pageTextContains($type_data['label']);
    }

    // Should show the types in the listing.
    $this->assertSession()->pageTextContains('Contact Detail types');

    // Check that we can navigate to manage individual types.
    foreach ($types as $type_data) {
      $this->drupalGet('admin/structure/crm/contact-detail-types/manage/' . $type_data['id']);
      $this->assertSession()->statusCodeEquals(200);
    }
  }

}
