<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Functional\Form;

use Drupal\Tests\BrowserTestBase;

/**
 * Comprehensive functional tests for the ContactTypeForm.
 *
 * @group crm
 * @covers \Drupal\crm\Form\ContactTypeForm
 */
class ContactTypeFormTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'crm',
    'datetime',
    'name',
    'telephone',
    'address',
  ];

  /**
   * A user with CRM administration permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * A user without CRM administration permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $normalUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->adminUser = $this->createUser([
      'administer crm',
    ]);

    $this->normalUser = $this->createUser([
      'view any crm_contact',
    ]);
  }

  /**
   * Tests access to the contact type forms.
   */
  public function testContactTypeFormAccess(): void {
    // Test that anonymous users cannot access the form.
    $this->drupalGet('admin/structure/crm/contact-types/add');
    $this->assertSession()->statusCodeEquals(403);

    // Test that normal users cannot access the form.
    $this->drupalLogin($this->normalUser);
    $this->drupalGet('admin/structure/crm/contact-types/add');
    $this->assertSession()->statusCodeEquals(403);

    // Test that admin users can access the form.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/structure/crm/contact-types/add');
    $this->assertSession()->statusCodeEquals(200);
  }

  /**
   * Tests creating a new contact type with minimal data.
   */
  public function testCreateContactTypeMinimal(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/structure/crm/contact-types/add');

    // Verify form elements are present.
    $this->assertSession()->fieldExists('label');
    $this->assertSession()->fieldExists('id');
    $this->assertSession()->fieldExists('description');
    $this->assertSession()->buttonExists('Save contact type');

    // Submit form with minimal data.
    $edit = [
      'label' => 'Test Contact Type',
      'id' => 'test_contact_type',
    ];
    $this->submitForm($edit, 'Save contact type');

    // Verify success message.
    $this->assertSession()->pageTextContains('The contact type Test Contact Type has been added.');

    // Verify entity was created.
    $contact_type = \Drupal::entityTypeManager()
      ->getStorage('crm_contact_type')
      ->load('test_contact_type');

    $this->assertNotNull($contact_type);
    $this->assertEquals('Test Contact Type', $contact_type->label());
    $this->assertEquals('test_contact_type', $contact_type->id());
  }

  /**
   * Tests creating a contact type with all fields populated.
   */
  public function testCreateContactTypeComplete(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/structure/crm/contact-types/add');

    // Verify additional settings elements are present.
    $this->assertSession()->fieldExists('start_date_label');
    $this->assertSession()->fieldExists('start_date_description');
    $this->assertSession()->fieldExists('end_date_label');
    $this->assertSession()->fieldExists('end_date_description');

    // Submit form with complete data.
    $edit = [
      'label' => 'Complete Contact Type',
      'id' => 'complete_contact_type',
      'description' => 'A complete contact type for testing',
      'start_date_label' => 'Membership Start Date',
      'start_date_description' => 'Date when membership begins',
      'end_date_label' => 'Membership End Date',
      'end_date_description' => 'Date when membership expires',
    ];
    $this->submitForm($edit, 'Save contact type');

    // Verify success message.
    $this->assertSession()->pageTextContains('The contact type Complete Contact Type has been added.');

    // Verify entity was created with all data.
    $contact_type = \Drupal::entityTypeManager()
      ->getStorage('crm_contact_type')
      ->load('complete_contact_type');

    $this->assertNotNull($contact_type);
    $this->assertEquals('Complete Contact Type', $contact_type->label());
    $this->assertEquals('complete_contact_type', $contact_type->id());
    $this->assertEquals('A complete contact type for testing', $contact_type->get('description'));

    $date_config = $contact_type->get('date');
    $this->assertEquals('Membership Start Date', $date_config['start_date']['label']);
    $this->assertEquals('Date when membership begins', $date_config['start_date']['description']);
    $this->assertEquals('Membership End Date', $date_config['end_date']['label']);
    $this->assertEquals('Date when membership expires', $date_config['end_date']['description']);
  }

  /**
   * Tests form validation.
   */
  public function testContactTypeFormValidation(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/structure/crm/contact-types/add');

    // Test submitting empty form.
    $this->submitForm([], 'Save contact type');
    $this->assertSession()->pageTextContains('Label field is required.');

    // Test submitting with only label.
    $edit = ['label' => 'Test Label', 'id' => 'test_label'];
    $this->submitForm($edit, 'Save contact type');
    // Should succeed as ID should be auto-generated.
    $this->assertSession()->pageTextContains('The contact type Test Label has been added.');

    // Test duplicate machine name.
    $this->drupalGet('admin/structure/crm/contact-types/add');
    $edit = [
      'label' => 'Another Test',
    // Same as auto-generated from previous test.
      'id' => 'test_label',
    ];
    $this->submitForm($edit, 'Save contact type');
    $this->assertSession()->pageTextContains('The machine-readable name is already in use');
  }

  /**
   * Tests editing an existing contact type.
   */
  public function testEditContactType(): void {
    // Create a contact type to edit.
    $contact_type = \Drupal::entityTypeManager()
      ->getStorage('crm_contact_type')
      ->create([
        'id' => 'editable_type',
        'label' => 'Editable Type',
        'description' => 'Original description',
        'date' => [
          'start_date' => [
            'label' => 'Original Start',
            'description' => 'Original start description',
          ],
          'end_date' => [
            'label' => 'Original End',
            'description' => 'Original end description',
          ],
        ],
      ]);
    $contact_type->save();

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/structure/crm/contact-types/manage/editable_type');

    $this->assertSession()->fieldValueEquals('label', 'Editable Type');
    $this->assertSession()->fieldValueEquals('id', 'editable_type');
    $this->assertSession()->fieldValueEquals('description', 'Original description');
    $this->assertSession()->fieldValueEquals('start_date_label', 'Original Start');
    $this->assertSession()->fieldValueEquals('start_date_description', 'Original start description');
    $this->assertSession()->fieldValueEquals('end_date_label', 'Original End');
    $this->assertSession()->fieldValueEquals('end_date_description', 'Original end description');

    // Edit the contact type.
    $edit = [
      'label' => 'Updated Type',
      'description' => 'Updated description',
      'start_date_label' => 'Updated Start',
      'start_date_description' => 'Updated start description',
      'end_date_label' => 'Updated End',
      'end_date_description' => 'Updated end description',
    ];
    $this->submitForm($edit, 'Save contact type');

    // Verify update message.
    $this->assertSession()->pageTextContains('The contact type Updated Type has been updated.');

    // Verify entity was updated.
    \Drupal::entityTypeManager()->getStorage('crm_contact_type')->resetCache();
    $updated_contact_type = \Drupal::entityTypeManager()
      ->getStorage('crm_contact_type')
      ->load('editable_type');

    $this->assertEquals('Updated Type', $updated_contact_type->label());
    $this->assertEquals('Updated description', $updated_contact_type->get('description'));

    $date_config = $updated_contact_type->get('date');
    $this->assertEquals('Updated Start', $date_config['start_date']['label']);
    $this->assertEquals('Updated start description', $date_config['start_date']['description']);
    $this->assertEquals('Updated End', $date_config['end_date']['label']);
    $this->assertEquals('Updated end description', $date_config['end_date']['description']);
  }

  /**
   * Tests the vertical tabs functionality.
   */
  public function testVerticalTabsInterface(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/structure/crm/contact-types/add');

    // Verify vertical tabs container exists
    // $this->assertSession()->elementExists('css', '.vertical-tabs');.
    // Verify start date and end date sections are in vertical tabs.
    $this->assertSession()->elementExists('css', 'details[data-drupal-selector="edit-start-date"]');
    $this->assertSession()->elementExists('css', 'details[data-drupal-selector="edit-end-date"]');

    // Verify start date section.
    $start_date_section = $this->getSession()->getPage()->find('css', 'details[data-drupal-selector="edit-start-date"]');
    $this->assertStringContainsString('Start Date', $start_date_section->getText());

    // Verify end date section.
    $end_date_section = $this->getSession()->getPage()->find('css', 'details[data-drupal-selector="edit-end-date"]');
    $this->assertStringContainsString('End Date', $end_date_section->getText());
  }

  /**
   * Tests machine name generation.
   */
  public function testMachineNameGeneration(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/structure/crm/contact-types/add');

    // Enter a label and verify machine name is auto-generated.
    $this->getSession()->getPage()->fillField('label', 'Special Contact Type!');
    $this->getSession()->getPage()->pressButton('Save contact type');

  }

  /**
   * Tests the delete functionality.
   */
  public function testDeleteContactType(): void {
    // Create a contact type to delete.
    $contact_type = \Drupal::entityTypeManager()
      ->getStorage('crm_contact_type')
      ->create([
        'id' => 'deletable_type',
        'label' => 'Deletable Type',
        'date' => [
          'start_date' => [
            'label' => '',
            'description' => '',
          ],
          'end_date' => [
            'label' => '',
            'description' => '',
          ],
        ],
      ]);
    $contact_type->save();

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/structure/crm/contact-types/manage/deletable_type/delete');
    $this->assertSession()->statusCodeEquals(200);

    // Verify delete confirmation page.
    $this->assertSession()->pageTextContains('Are you sure you want to delete the contact type Deletable Type?');
    $this->assertSession()->buttonExists('Delete');

    // Confirm deletion.
    $this->submitForm([], 'Delete');
    $this->assertSession()->pageTextContains('The contact type Deletable Type has been deleted.');

    // Verify entity was deleted.
    $deleted_contact_type = \Drupal::entityTypeManager()
      ->getStorage('crm_contact_type')
      ->load('deletable_type');
    $this->assertNull($deleted_contact_type);
  }

  /**
   * Tests redirects after form submission.
   */
  public function testFormRedirects(): void {
    $this->drupalLogin($this->adminUser);

    // Test redirect after creating new contact type.
    $this->drupalGet('admin/structure/crm/contact-types/add');
    $edit = [
      'label' => 'Redirect Test',
      'id' => 'redirect_test',
    ];
    $this->submitForm($edit, 'Save contact type');

    // Should redirect to collection page.
    $this->assertSession()->addressEquals('admin/structure/crm/contact-types');

    // Test redirect after editing contact type.
    $this->drupalGet('admin/structure/crm/contact-types/manage/redirect_test');
    $edit = ['label' => 'Updated Redirect Test'];
    $this->submitForm($edit, 'Save contact type');

    // Should redirect to collection page.
    $this->assertSession()->addressEquals('admin/structure/crm/contact-types');
  }

  /**
   * Tests form with JavaScript disabled scenarios.
   */
  public function testFormWithoutJavaScript(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/structure/crm/contact-types/add');

    // Test form functionality without relying on JavaScript.
    $edit = [
      'label' => 'No JS Test',
      'id' => 'no_js_test',
      'description' => 'Testing without JavaScript',
      'start_date_label' => 'Start',
      'start_date_description' => 'Start desc',
      'end_date_label' => 'End',
      'end_date_description' => 'End desc',
    ];

    $this->submitForm($edit, 'Save contact type');
    $this->assertSession()->pageTextContains('The contact type No JS Test has been added.');

    // Verify all data was saved correctly.
    $contact_type = \Drupal::entityTypeManager()
      ->getStorage('crm_contact_type')
      ->load('no_js_test');

    $this->assertNotNull($contact_type);
    $this->assertEquals('No JS Test', $contact_type->label());
    $this->assertEquals('Testing without JavaScript', $contact_type->get('description'));

    $date_config = $contact_type->get('date');
    $this->assertEquals('Start', $date_config['start_date']['label']);
    $this->assertEquals('Start desc', $date_config['start_date']['description']);
    $this->assertEquals('End', $date_config['end_date']['label']);
    $this->assertEquals('End desc', $date_config['end_date']['description']);
  }

}
