<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Functional\Form;

use Drupal\Tests\BrowserTestBase;
use Drupal\crm\Entity\DetailType;
use Drupal\crm\Entity\ContactDetailType;

/**
 * Comprehensive functional tests for the DetailTypeForm.
 *
 * @group crm
 * @covers \Drupal\crm\Form\DetailTypeForm
 */
class DetailTypeFormTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'crm',
    'datetime',
    'name',
    'telephone',
    'address',
  ];

  /**
   * A user with CRM administration permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * A user without CRM administration permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $normalUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->adminUser = $this->createUser([
      'administer crm',
    ]);

    $this->normalUser = $this->createUser([
      'view any crm_contact',
    ]);

  }

  /**
   * Tests access to the detail type forms.
   */
  public function testDetailTypeFormAccess(): void {
    // Test that anonymous users cannot access the form.
    $this->drupalGet('admin/structure/crm_detail_type/add');
    $this->assertSession()->statusCodeEquals(403);

    // Test that normal users cannot access the form.
    $this->drupalLogin($this->normalUser);
    $this->drupalGet('admin/structure/crm_detail_type/add');
    $this->assertSession()->statusCodeEquals(403);

    // Test that admin users can access the form.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/structure/crm_detail_type/add');
    $this->assertSession()->statusCodeEquals(200);
  }

  /**
   * Tests creating a new detail type with minimal data.
   */
  public function testCreateDetailTypeMinimal(): void {
    $this->drupalLogin($this->adminUser);

    // Navigate to the add form.
    $this->drupalGet('admin/structure/crm_detail_type/add');
    $this->assertSession()->statusCodeEquals(200);

    // Check that required form elements are present.
    $this->assertSession()->fieldExists('label');
    $this->assertSession()->fieldExists('id');
    $this->assertSession()->fieldExists('status');
    $this->assertSession()->fieldExists('description');
    $this->assertSession()->fieldExists('bundles[address]');
    $this->assertSession()->fieldExists('bundles[email]');
    $this->assertSession()->fieldExists('bundles[telephone]');
    $this->assertSession()->fieldExists('negate');

    // Submit the form with minimal data.
    $edit = [
      'label' => 'Test Detail Type',
      'id' => 'test_detail_type',
      'status' => TRUE,
    ];
    $this->submitForm($edit, 'Save');

    // Check that we were redirected to the collection page.
    $this->assertSession()->addressEquals('admin/structure/crm-detail-type');

    // Check that the success message is displayed.
    $this->assertSession()->pageTextContains('Created new detail type Test Detail Type.');

    // Verify the entity was created.
    $detail_type = DetailType::load('test_detail_type');
    $this->assertNotNull($detail_type);
    $this->assertEquals('Test Detail Type', $detail_type->label());
    $this->assertEquals('test_detail_type', $detail_type->id());
    $this->assertTrue($detail_type->status());
    $this->assertEquals('', $detail_type->get('description'));
    $this->assertEquals([], $detail_type->get('bundles'));
    $this->assertFalse($detail_type->get('negate'));
  }

  /**
   * Tests creating a new detail type with complete data.
   */
  public function testCreateDetailTypeComplete(): void {
    $this->drupalLogin($this->adminUser);

    $this->drupalGet('admin/structure/crm_detail_type/add');

    $edit = [
      'label' => 'Work Contact',
      'id' => 'work_contact',
      'status' => TRUE,
      'description' => 'Work-related contact information for business purposes',
      'bundles[address]' => 'address',
      'bundles[email]' => 'email',
      'bundles[telephone]' => FALSE,
      'negate' => FALSE,
    ];
    $this->submitForm($edit, 'Save');

    $this->assertSession()->addressEquals('admin/structure/crm-detail-type');
    $this->assertSession()->pageTextContains('Created new detail type Work Contact.');

    // Verify the entity was created with all data.
    $detail_type = DetailType::load('work_contact');
    $this->assertNotNull($detail_type);
    $this->assertEquals('Work Contact', $detail_type->label());
    $this->assertEquals('work_contact', $detail_type->id());
    $this->assertTrue($detail_type->status());
    $this->assertEquals('Work-related contact information for business purposes', $detail_type->get('description'));
    $this->assertEquals(['address', 'email'], $detail_type->get('bundles'));
    $this->assertFalse($detail_type->get('negate'));
  }

  /**
   * Tests creating a detail type with negate option.
   */
  public function testCreateDetailTypeWithNegate(): void {
    $this->drupalLogin($this->adminUser);

    $this->drupalGet('admin/structure/crm_detail_type/add');

    $edit = [
      'label' => 'Non-Address',
      'id' => 'non_address',
      'description' => 'Applies to all bundles except address',
      'bundles[address]' => 'address',
      'bundles[email]' => FALSE,
      'bundles[telephone]' => FALSE,
      'negate' => TRUE,
    ];
    $this->submitForm($edit, 'Save');

    $this->assertSession()->pageTextContains('Created new detail type Non-Address.');

    $detail_type = DetailType::load('non_address');
    $this->assertNotNull($detail_type);
    $this->assertEquals(['address'], $detail_type->get('bundles'));
    $this->assertTrue($detail_type->get('negate'));
  }

  /**
   * Tests editing an existing detail type.
   */
  public function testEditDetailType(): void {
    // Create a detail type to edit.
    $detail_type = DetailType::create([
      'id' => 'functional_personal_test',
      'label' => 'Personal',
      'description' => 'Personal contact information',
      'bundles' => ['email'],
      'negate' => FALSE,
    ]);
    $detail_type->save();

    $this->drupalLogin($this->adminUser);

    // Navigate to the edit form.
    $this->drupalGet('admin/structure/crm-detail-type/functional_personal_test');
    $this->assertSession()->statusCodeEquals(200);

    // Check that the form is pre-populated with existing values.
    $this->assertSession()->fieldValueEquals('label', 'Personal');
    $this->assertSession()->fieldValueEquals('id', 'functional_personal_test');
    $this->assertSession()->fieldValueEquals('description', 'Personal contact information');
    $this->assertSession()->checkboxChecked('bundles[email]');
    $this->assertSession()->checkboxNotChecked('bundles[address]');
    $this->assertSession()->checkboxNotChecked('bundles[telephone]');
    $this->assertSession()->checkboxNotChecked('negate');

    // Check that the ID field is disabled.
    $this->assertSession()->fieldDisabled('id');

    // Update the detail type.
    $edit = [
      'label' => 'Personal Information',
      'description' => 'Updated personal contact information',
      'bundles[address]' => 'address',
      'bundles[email]' => 'email',
      'bundles[telephone]' => FALSE,
      'negate' => TRUE,
    ];
    $this->submitForm($edit, 'Save');

    $this->assertSession()->addressEquals('admin/structure/crm-detail-type');
    $this->assertSession()->pageTextContains('Updated detail type Personal Information.');

    // Verify the entity was updated.
    $updated_detail_type = DetailType::load('functional_personal_test');
    $this->assertNotNull($updated_detail_type);
    $this->assertEquals('Personal Information', $updated_detail_type->label());
    $this->assertEquals('Updated personal contact information', $updated_detail_type->get('description'));
    $this->assertEquals(['address', 'email'], $updated_detail_type->get('bundles'));
    $this->assertTrue($updated_detail_type->get('negate'));
  }

  /**
   * Tests form validation for required fields.
   */
  public function testFormValidationRequiredFields(): void {
    $this->drupalLogin($this->adminUser);

    $this->drupalGet('admin/structure/crm_detail_type/add');

    // Submit form without required label.
    $edit = [
      'label' => '',
      'id' => 'test_empty',
    ];
    $this->submitForm($edit, 'Save');

    // Should stay on the same page and show validation error.
    $this->assertSession()->addressEquals('admin/structure/crm_detail_type/add');
    $this->assertSession()->pageTextContains('Label field is required.');
  }

  /**
   * Tests machine name validation and uniqueness.
   */
  public function testMachineNameValidation(): void {
    // Create an existing detail type.
    DetailType::create([
      'id' => 'functional_existing_test',
      'label' => 'Existing Type',
      'bundles' => [],
    ])->save();

    $this->drupalLogin($this->adminUser);

    $this->drupalGet('admin/structure/crm_detail_type/add');

    // Try to create a detail type with duplicate machine name.
    $edit = [
      'label' => 'Duplicate Type',
      'id' => 'functional_existing_test',
    ];
    $this->submitForm($edit, 'Save');

    // Should show validation error for duplicate machine name.
    $this->assertSession()->addressEquals('admin/structure/crm_detail_type/add');
    $this->assertSession()->pageTextContains('The machine-readable name is already in use. It must be unique.');
  }

  /**
   * Tests disabled status functionality.
   */
  public function testDetailTypeStatusToggle(): void {
    $this->drupalLogin($this->adminUser);

    $this->drupalGet('admin/structure/crm_detail_type/add');

    // Create a disabled detail type.
    $edit = [
      'label' => 'Disabled Type',
      'id' => 'disabled_type',
      'status' => FALSE,
      'description' => 'This type is disabled',
    ];
    $this->submitForm($edit, 'Save');

    $this->assertSession()->pageTextContains('Created new detail type Disabled Type.');

    // Verify the entity was created as disabled.
    $detail_type = DetailType::load('disabled_type');
    $this->assertNotNull($detail_type);
    $this->assertFalse($detail_type->status());

    // Edit to enable it.
    $this->drupalGet('admin/structure/crm-detail-type/disabled_type');
    $edit = [
      'status' => TRUE,
    ];
    $this->submitForm($edit, 'Save');

    $this->assertSession()->pageTextContains('Updated detail type Disabled Type.');

    // Verify it's now enabled.
    $updated_detail_type = DetailType::load('disabled_type');
    $this->assertTrue($updated_detail_type->status());
  }

  /**
   * Tests complex bundle selection scenarios.
   */
  public function testComplexBundleSelection(): void {
    $this->drupalLogin($this->adminUser);

    $this->drupalGet('admin/structure/crm_detail_type/add');

    // Test selecting multiple bundles.
    $edit = [
      'label' => 'Multi Bundle Type',
      'id' => 'multi_bundle',
      'bundles[address]' => 'address',
      'bundles[email]' => 'email',
      'bundles[telephone]' => 'telephone',
      'negate' => FALSE,
    ];
    $this->submitForm($edit, 'Save');

    $detail_type = DetailType::load('multi_bundle');
    $this->assertEquals(['address', 'email', 'telephone'], $detail_type->get('bundles'));
    $this->assertFalse($detail_type->get('negate'));

    // Test with negate option - should apply to all bundles except selected.
    $this->drupalGet('admin/structure/crm_detail_type/add');
    $edit = [
      'label' => 'Negated Type',
      'id' => 'negated_type',
      'bundles[address]' => 'address',
      'bundles[email]' => FALSE,
      'bundles[telephone]' => FALSE,
      'negate' => TRUE,
    ];
    $this->submitForm($edit, 'Save');

    $negated_type = DetailType::load('negated_type');
    $this->assertEquals(['address'], $negated_type->get('bundles'));
    $this->assertTrue($negated_type->get('negate'));
  }

  /**
   * Tests form workflow from creation to editing to deletion.
   */
  public function testDetailTypeWorkflow(): void {
    $this->drupalLogin($this->adminUser);

    // Step 1: Create a new detail type.
    $this->drupalGet('admin/structure/crm_detail_type/add');
    $edit = [
      'label' => 'Workflow Test Type',
      'id' => 'workflow_test',
      'description' => 'Initial description',
      'bundles[email]' => 'email',
    ];
    $this->submitForm($edit, 'Save');

    $this->assertSession()->pageTextContains('Created new detail type Workflow Test Type.');

    // Step 2: Verify it appears in the collection list.
    $this->assertSession()->addressEquals('admin/structure/crm-detail-type');
    $this->assertSession()->pageTextContains('Workflow Test Type');

    // Step 3: Edit the detail type.
    $this->drupalGet('admin/structure/crm-detail-type/workflow_test');
    $this->assertSession()->statusCodeEquals(200);

    // Verify form is pre-populated.
    $this->assertSession()->fieldValueEquals('label', 'Workflow Test Type');
    $this->assertSession()->fieldValueEquals('description', 'Initial description');
    $this->assertSession()->checkboxChecked('bundles[email]');

    // Update the detail type.
    $edit = [
      'label' => 'Updated Workflow Type',
      'description' => 'Updated description',
      'bundles[address]' => 'address',
      'bundles[email]' => 'email',
      'negate' => TRUE,
    ];
    $this->submitForm($edit, 'Save');

    $this->assertSession()->pageTextContains('Updated detail type Updated Workflow Type.');

    // Step 4: Verify the changes were saved.
    $updated_type = DetailType::load('workflow_test');
    $this->assertEquals('Updated Workflow Type', $updated_type->label());
    $this->assertEquals('Updated description', $updated_type->get('description'));
    $bundles = $updated_type->get('bundles');
    sort($bundles);
    $this->assertEquals(['address', 'email'], $bundles);
    $this->assertTrue($updated_type->get('negate'));
  }

  /**
   * Tests form behavior with no available contact detail types.
   */
  public function testFormWithNoContactDetailTypes(): void {
    // Delete all contact detail types to test edge case.
    $contact_detail_types = ContactDetailType::loadMultiple();
    foreach ($contact_detail_types as $type) {
      $type->delete();
    }

    $this->drupalLogin($this->adminUser);

    $this->drupalGet('admin/structure/crm_detail_type/add');
    $this->assertSession()->statusCodeEquals(200);

    // Bundles field should still exist but have no options.
    $this->assertSession()->fieldNotExists('bundles');

    // Should still be able to create a detail type.
    $edit = [
      'label' => 'No Bundles Type',
      'id' => 'no_bundles',
      'description' => 'Type with no available bundles',
    ];
    $this->submitForm($edit, 'Save');

    $this->assertSession()->pageTextContains('Created new detail type No Bundles Type.');

    $detail_type = DetailType::load('no_bundles');
    $this->assertNotNull($detail_type);
    $this->assertEquals([], $detail_type->get('bundles'));
  }

  /**
   * Tests form validation errors display properly.
   */
  public function testFormValidationErrorDisplay(): void {
    $this->drupalLogin($this->adminUser);

    $this->drupalGet('admin/structure/crm_detail_type/add');

    // Submit form with invalid data.
    $edit = [
    // Empty required field.
      'label' => '',
    // Invalid machine name characters.
      'id' => 'test-with-dashes',
    ];
    $this->submitForm($edit, 'Save');

    // Should stay on form page and show validation errors.
    $this->assertSession()->addressEquals('admin/structure/crm_detail_type/add');
    $this->assertSession()->pageTextContains('Label field is required.');
  }

  /**
   * Tests the machine name auto-generation functionality.
   */
  public function testMachineNameAutoGeneration(): void {
    $this->drupalLogin($this->adminUser);

    $this->drupalGet('admin/structure/crm_detail_type/add');

    // Type in the label field and verify machine name is auto-generated.
    $this->getSession()->getPage()->fillField('label', 'Auto Generated Name');

    // Check that the machine name field is present.
    $this->assertSession()->fieldExists('id');

    // The exact auto-generation behavior depends on JavaScript, but we can test
    // that the machine name field exists and accepts manual input.
    $edit = [
      'label' => 'Manual Machine Name',
      'id' => 'manual_machine_name',
    ];
    $this->submitForm($edit, 'Save');

    $this->assertSession()->pageTextContains('Created new detail type Manual Machine Name.');

    $detail_type = DetailType::load('manual_machine_name');
    $this->assertNotNull($detail_type);
    $this->assertEquals('manual_machine_name', $detail_type->id());
  }

  /**
   * Tests form access for edit operations.
   */
  public function testEditFormAccess(): void {
    // Create a detail type to edit.
    DetailType::create([
      'id' => 'functional_edit_test',
      'label' => 'Test Edit Type',
      'bundles' => [],
    ])->save();

    // Test that anonymous users cannot access edit form.
    $this->drupalGet('admin/structure/crm-detail-type/functional_edit_test');
    $this->assertSession()->statusCodeEquals(403);

    // Test that normal users cannot access edit form.
    $this->drupalLogin($this->normalUser);
    $this->drupalGet('admin/structure/crm-detail-type/functional_edit_test');
    $this->assertSession()->statusCodeEquals(403);

    // Test that admin users can access edit form.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/structure/crm-detail-type/functional_edit_test');
    $this->assertSession()->statusCodeEquals(200);

    // Verify it's the edit form (ID field should be disabled).
    $this->assertSession()->fieldDisabled('id');
  }

  /**
   * Tests that bundles are properly filtered on save.
   */
  public function testBundleFiltering(): void {
    $this->drupalLogin($this->adminUser);

    $this->drupalGet('admin/structure/crm_detail_type/add');

    // Submit with mixed selected/unselected bundles.
    $edit = [
      'label' => 'Bundle Filter Test',
      'id' => 'bundle_filter_test',
      'bundles[address]' => 'address',
    // This should be filtered out.
      'bundles[email]' => FALSE,
      'bundles[telephone]' => 'telephone',
    ];
    $this->submitForm($edit, 'Save');

    $detail_type = DetailType::load('bundle_filter_test');
    $this->assertNotNull($detail_type);

    // Should only contain the selected bundles, with falsy values filtered out.
    $bundles = $detail_type->get('bundles');
    $this->assertContains('address', $bundles);
    $this->assertContains('telephone', $bundles);
    $this->assertNotContains('email', $bundles);
    $this->assertCount(2, $bundles);
  }

}
