<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Functional\Form;

use Drupal\Tests\BrowserTestBase;
use Drupal\crm\Entity\RelationshipType;

/**
 * Comprehensive functional tests for the RelationshipTypeForm.
 *
 * @group crm
 * @covers \Drupal\crm\Form\RelationshipTypeForm
 */
class RelationshipTypeFormTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'crm',
    'datetime',
    'inline_entity_form',
    'primary_entity_reference',
    'name',
    'telephone',
    'address',
    'views',
  ];

  /**
   * A user with CRM administration permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * A user without CRM administration permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $normalUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->adminUser = $this->createUser([
      'administer crm',
    ]);

    $this->normalUser = $this->createUser([
      'view any crm_contact',
    ]);
  }

  /**
   * Tests access to the relationship type forms.
   */
  public function testRelationshipTypeFormAccess(): void {
    // Test that anonymous users cannot access the form.
    $this->drupalGet('admin/structure/crm/relationship_types/add');
    $this->assertSession()->statusCodeEquals(403);

    // Test that normal users cannot access the form.
    $this->drupalLogin($this->normalUser);
    $this->drupalGet('admin/structure/crm/relationship_types/add');
    $this->assertSession()->statusCodeEquals(403);

    // Test that admin users can access the form.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/structure/crm/relationship_types/add');
    $this->assertSession()->statusCodeEquals(200);
  }

  /**
   * Tests creating a new asymmetric relationship type.
   */
  public function testCreateAsymmetricRelationshipType(): void {
    $this->drupalLogin($this->adminUser);

    // Navigate to the add form.
    $this->drupalGet('admin/structure/crm/relationship_types/add');
    $this->assertSession()->statusCodeEquals(200);

    // Check that required form elements are present.
    $this->assertSession()->fieldExists('label');
    $this->assertSession()->fieldExists('id');
    $this->assertSession()->fieldExists('description');
    $this->assertSession()->fieldExists('asymmetric');
    $this->assertSession()->fieldExists('label_a');
    $this->assertSession()->fieldExists('contact_type_a[]');
    $this->assertSession()->fieldExists('label_b');
    $this->assertSession()->fieldExists('contact_type_b[]');

    // Submit the form to create an asymmetric relationship.
    $edit = [
      'label' => 'Manager-Employee',
      'id' => 'func_manager_employee',
      'description' => 'Manager to employee relationship',
      'asymmetric' => TRUE,
      'label_a' => 'Manager',
      'contact_type_a[]' => ['person'],
      'label_b' => 'Employee',
      'contact_type_b[]' => ['person'],
    ];
    $this->submitForm($edit, 'Save relationship type');

    // Check that we were redirected to the collection page.
    $this->assertSession()->addressEquals('admin/structure/crm/relationship_types');

    // Check that the success message is displayed.
    $this->assertSession()->pageTextContains('The crm relationship type Manager-Employee has been added.');

    // Verify the entity was created.
    $relationship_type = RelationshipType::load('func_manager_employee');
    $this->assertNotNull($relationship_type);
    $this->assertEquals('Manager-Employee', $relationship_type->label());
    $this->assertEquals('func_manager_employee', $relationship_type->id());
    $this->assertEquals('Manager to employee relationship', $relationship_type->get('description'));
    $this->assertTrue($relationship_type->get('asymmetric'));
    $this->assertEquals('Manager', $relationship_type->get('label_a'));
    $this->assertEquals(['person'], $relationship_type->get('contact_type_a'));
    $this->assertEquals('Employee', $relationship_type->get('label_b'));
    $this->assertEquals(['person'], $relationship_type->get('contact_type_b'));
  }

  /**
   * Tests creating a new symmetric relationship type.
   */
  public function testCreateSymmetricRelationshipType(): void {
    $this->drupalLogin($this->adminUser);

    $this->drupalGet('admin/structure/crm/relationship_types/add');

    // Submit the form to create a symmetric relationship.
    // Note: We need to provide label_b and contact_type_b values even for
    // symmetric  because they are marked as required fields, even though they
    // will be overwritten by the form's save method.
    $edit = [
      'label' => 'Sibling',
      'id' => 'functional_sibling_test',
      'description' => 'Brother or sister relationship',
      'asymmetric' => FALSE,
      'label_a' => 'Sibling',
      'contact_type_a[]' => ['person'],
      // These values are required by the form but will be ignored for
      // symmetric relationships.
      'label_b' => 'Sibling',
      'contact_type_b[]' => ['person'],
    ];
    $this->submitForm($edit, 'Save relationship type');

    $this->assertSession()->addressEquals('admin/structure/crm/relationship_types');
    $this->assertSession()->pageTextContains('The crm relationship type Sibling has been added.');

    // Verify the entity was created with symmetric values.
    $relationship_type = RelationshipType::load('functional_sibling_test');
    $this->assertNotNull($relationship_type);
    $this->assertEquals('Sibling', $relationship_type->label());
    $this->assertFalse($relationship_type->get('asymmetric'));
    $this->assertEquals('Sibling', $relationship_type->get('label_a'));
    $this->assertEquals(['person'], $relationship_type->get('contact_type_a'));
    // These should match label_a and contact_type_a due to symmetric logic.
    $this->assertEquals('Sibling', $relationship_type->get('label_b'));
    $this->assertEquals(['person'], $relationship_type->get('contact_type_b'));
  }

  /**
   * Tests editing an existing relationship type.
   */
  public function testEditRelationshipType(): void {
    // Create a relationship type to edit.
    $relationship_type = RelationshipType::create([
      'id' => 'functional_edit_test',
      'label' => 'Original Type',
      'description' => 'Original description',
      'asymmetric' => TRUE,
      'label_a' => 'Original A',
      'contact_type_a' => ['person'],
      'label_b' => 'Original B',
      'contact_type_b' => ['organization'],
    ]);
    $relationship_type->save();

    $this->drupalLogin($this->adminUser);

    // Navigate to the edit form.
    $this->drupalGet('admin/structure/crm/relationship_types/manage/functional_edit_test');
    $this->assertSession()->statusCodeEquals(200);

    // Check that the form is pre-populated with existing values.
    $this->assertSession()->fieldValueEquals('label', 'Original Type');
    $this->assertSession()->fieldValueEquals('id', 'functional_edit_test');
    $this->assertSession()->fieldValueEquals('description', 'Original description');
    $this->assertSession()->checkboxChecked('asymmetric');
    $this->assertSession()->fieldValueEquals('label_a', 'Original A');
    $this->assertSession()->elementExists('css', 'select[name="contact_type_a[]"] option[value="person"][selected]');
    $this->assertSession()->fieldValueEquals('label_b', 'Original B');
    $this->assertSession()->elementExists('css', 'select[name="contact_type_b[]"] option[value="organization"][selected]');

    // Update the relationship type.
    $edit = [
      'label' => 'Updated Type',
      'description' => 'Updated description',
      'asymmetric' => FALSE,
      'label_a' => 'Updated A',
      'contact_type_a[]' => ['household'],
    ];
    $this->submitForm($edit, 'Save relationship type');

    $this->assertSession()->addressEquals('admin/structure/crm/relationship_types');
    $this->assertSession()->pageTextContains('The crm relationship type Updated Type has been updated.');

    // Verify the entity was updated.
    $updated_relationship_type = RelationshipType::load('functional_edit_test');
    $this->assertNotNull($updated_relationship_type);
    $this->assertEquals('Updated Type', $updated_relationship_type->label());
    $this->assertEquals('Updated description', $updated_relationship_type->get('description'));
    $this->assertFalse($updated_relationship_type->get('asymmetric'));
    $this->assertEquals('Updated A', $updated_relationship_type->get('label_a'));
    $this->assertEquals(['household'], $updated_relationship_type->get('contact_type_a'));
    // These should be automatically set due to symmetric logic.
    $this->assertEquals('Updated A', $updated_relationship_type->get('label_b'));
    $this->assertEquals(['household'], $updated_relationship_type->get('contact_type_b'));
  }

  /**
   * Tests form validation for required fields.
   */
  public function testFormValidationRequiredFields(): void {
    $this->drupalLogin($this->adminUser);

    $this->drupalGet('admin/structure/crm/relationship_types/add');

    // Submit form without required fields.
    // Note: We don't include contact_type_a[] in the edit array to leave it
    // unselected, which will trigger the required field validation.
    $edit = [
      'label' => '',
      'id' => 'test_empty',
      'label_a' => '',
    ];
    $this->submitForm($edit, 'Save relationship type');

    // Should stay on the same page and show validation errors.
    $this->assertSession()->addressEquals('admin/structure/crm/relationship_types/add');
    $this->assertSession()->pageTextContains('Label field is required.');
    $this->assertSession()->pageTextContains('Contact A label field is required.');
    $this->assertSession()->pageTextContains('Contact A type field is required.');
  }

  /**
   * Tests machine name validation and uniqueness.
   */
  public function testMachineNameValidation(): void {
    // Create an existing relationship type.
    RelationshipType::create([
      'id' => 'functional_existing_test',
      'label' => 'Existing Type',
      'label_a' => 'Test A',
      'contact_type_a' => ['person'],
      'contact_type_b' => ['person'],
    ])->save();

    $this->drupalLogin($this->adminUser);

    $this->drupalGet('admin/structure/crm/relationship_types/add');

    // Try to create a relationship type with duplicate machine name.
    $edit = [
      'label' => 'Duplicate Type',
      'id' => 'functional_existing_test',
      'label_a' => 'Test A',
      'contact_type_a[]' => ['person'],
    ];
    $this->submitForm($edit, 'Save relationship type');

    // Should show validation error for duplicate machine name.
    $this->assertSession()->addressEquals('admin/structure/crm/relationship_types/add');
    $this->assertSession()->pageTextContains('The machine-readable name is already in use. It must be unique.');
  }

  /**
   * Tests the asymmetric checkbox behavior.
   */
  public function testAsymmetricCheckboxBehavior(): void {
    $this->drupalLogin($this->adminUser);

    $this->drupalGet('admin/structure/crm/relationship_types/add');

    // Test creating an asymmetric relationship.
    $edit = [
      'label' => 'Parent-Child',
      'id' => 'parent_child',
      'asymmetric' => TRUE,
      'label_a' => 'Parent',
      'contact_type_a[]' => ['person'],
      'label_b' => 'Child',
      'contact_type_b[]' => ['person'],
    ];
    $this->submitForm($edit, 'Save relationship type');

    $relationship_type = RelationshipType::load('parent_child');
    $this->assertTrue($relationship_type->get('asymmetric'));
    $this->assertEquals('Parent', $relationship_type->get('label_a'));
    $this->assertEquals('Child', $relationship_type->get('label_b'));

    // Test creating a symmetric relationship.
    $this->drupalGet('admin/structure/crm/relationship_types/add');
    $edit = [
      'label' => 'Friend',
      'id' => 'friend',
      'asymmetric' => FALSE,
      'label_a' => 'Friend',
      'contact_type_a[]' => ['person'],
      // These should be ignored.
      'label_b' => 'Different',
      'contact_type_b[]' => ['organization'],
    ];
    $this->submitForm($edit, 'Save relationship type');

    $friend_type = RelationshipType::load('friend');
    $this->assertFalse($friend_type->get('asymmetric'));
    $this->assertEquals('Friend', $friend_type->get('label_a'));
    $this->assertEquals('Friend', $friend_type->get('label_b'));
    $this->assertEquals(['person'], $friend_type->get('contact_type_a'));
    $this->assertEquals(['person'], $friend_type->get('contact_type_b'));
  }

  /**
   * Tests contact type dropdown options.
   */
  public function testContactTypeDropdownOptions(): void {
    $this->drupalLogin($this->adminUser);

    $this->drupalGet('admin/structure/crm/relationship_types/add');

    // Check that contact type dropdowns have the expected options.
    $this->assertSession()->optionExists('contact_type_a[]', 'person');
    $this->assertSession()->optionExists('contact_type_a[]', 'household');
    $this->assertSession()->optionExists('contact_type_a[]', 'organization');

    $this->assertSession()->optionExists('contact_type_b[]', 'person');
    $this->assertSession()->optionExists('contact_type_b[]', 'household');
    $this->assertSession()->optionExists('contact_type_b[]', 'organization');
  }

  /**
   * Tests form workflow from creation to editing.
   */
  public function testRelationshipTypeWorkflow(): void {
    $this->drupalLogin($this->adminUser);

    // Step 1: Create a new relationship type.
    $this->drupalGet('admin/structure/crm/relationship_types/add');
    $edit = [
      'label' => 'Workflow Test Type',
      'id' => 'workflow_test',
      'description' => 'Initial description',
      'asymmetric' => TRUE,
      'label_a' => 'Supervisor',
      'contact_type_a[]' => ['person'],
      'label_b' => 'Supervised',
      'contact_type_b[]' => ['person'],
    ];
    $this->submitForm($edit, 'Save relationship type');

    $this->assertSession()->pageTextContains('The crm relationship type Workflow Test Type has been added.');

    // Step 2: Verify it appears in the collection list.
    $this->assertSession()->addressEquals('admin/structure/crm/relationship_types');
    $this->assertSession()->pageTextContains('Workflow Test Type');

    // Step 3: Edit the relationship type.
    $this->drupalGet('admin/structure/crm/relationship_types/manage/workflow_test');
    $this->assertSession()->statusCodeEquals(200);

    // Verify form is pre-populated.
    $this->assertSession()->fieldValueEquals('label', 'Workflow Test Type');
    $this->assertSession()->fieldValueEquals('description', 'Initial description');
    $this->assertSession()->checkboxChecked('asymmetric');
    $this->assertSession()->fieldValueEquals('label_a', 'Supervisor');
    $this->assertSession()->elementExists('css', 'select[name="contact_type_a[]"] option[value="person"][selected]');
    $this->assertSession()->fieldValueEquals('label_b', 'Supervised');
    $this->assertSession()->elementExists('css', 'select[name="contact_type_b[]"] option[value="person"][selected]');

    // Update the relationship type to be symmetric.
    $edit = [
      'label' => 'Updated Workflow Type',
      'description' => 'Updated description',
      'asymmetric' => FALSE,
      'label_a' => 'Partner',
      'contact_type_a[]' => ['organization'],
    ];
    $this->submitForm($edit, 'Save relationship type');

    $this->assertSession()->pageTextContains('The crm relationship type Updated Workflow Type has been updated.');

    // Step 4: Verify the changes were saved.
    $updated_type = RelationshipType::load('workflow_test');
    $this->assertEquals('Updated Workflow Type', $updated_type->label());
    $this->assertEquals('Updated description', $updated_type->get('description'));
    $this->assertFalse($updated_type->get('asymmetric'));
    $this->assertEquals('Partner', $updated_type->get('label_a'));
    $this->assertEquals(['organization'], $updated_type->get('contact_type_a'));
    // These should be automatically set due to symmetric logic.
    $this->assertEquals('Partner', $updated_type->get('label_b'));
    $this->assertEquals(['organization'], $updated_type->get('contact_type_b'));
  }

  /**
   * Tests form validation errors display properly.
   */
  public function testFormValidationErrorDisplay(): void {
    $this->drupalLogin($this->adminUser);

    $this->drupalGet('admin/structure/crm/relationship_types/add');

    // Submit form with invalid data.
    // Note: We don't include contact_type_a[] in the edit array to leave it
    // unselected, which will trigger the required field validation.
    $edit = [
      // Empty required field.
      'label' => '',
      // Invalid machine name characters.
      'id' => 'test-with-dashes',
      'label_a' => '',
    ];
    $this->submitForm($edit, 'Save relationship type');

    // Should stay on form page and show validation errors.
    $this->assertSession()->addressEquals('admin/structure/crm/relationship_types/add');
    $this->assertSession()->pageTextContains('Label field is required.');
    $this->assertSession()->pageTextContains('Contact A label field is required.');
    $this->assertSession()->pageTextContains('Contact A type field is required.');
  }

  /**
   * Tests the machine name auto-generation functionality.
   */
  public function testMachineNameAutoGeneration(): void {
    $this->drupalLogin($this->adminUser);

    $this->drupalGet('admin/structure/crm/relationship_types/add');

    // Check that the machine name field is present.
    $this->assertSession()->fieldExists('id');

    // Test manual machine name input.
    $edit = [
      'label' => 'Manual Machine Name',
      'id' => 'func_manual_machine_name',
      'asymmetric' => FALSE,
      'label_a' => 'Test A',
      'contact_type_a[]' => ['person'],
      'label_b' => 'Test A',
      'contact_type_b[]' => ['person'],
    ];
    $this->submitForm($edit, 'Save relationship type');

    $this->assertSession()->pageTextContains('The crm relationship type Manual Machine Name has been added.');

    $relationship_type = RelationshipType::load('func_manual_machine_name');
    $this->assertNotNull($relationship_type);
    $this->assertEquals('func_manual_machine_name', $relationship_type->id());
  }

  /**
   * Tests form access for edit operations.
   */
  public function testEditFormAccess(): void {
    // Create a relationship type to edit.
    RelationshipType::create([
      'id' => 'functional_edit_access_test',
      'label' => 'Test Edit Type',
      'label_a' => 'Test A',
      'contact_type_a' => ['person'],
      'contact_type_b' => ['person'],
      'asymmetric' => FALSE,
    ])->save();

    // Test that anonymous users cannot access edit form.
    $this->drupalGet('admin/structure/crm/relationship_types/manage/functional_edit_access_test');
    $this->assertSession()->statusCodeEquals(403);

    // Test that normal users cannot access edit form.
    $this->drupalLogin($this->normalUser);
    $this->drupalGet('admin/structure/crm/relationship_types/manage/functional_edit_access_test');
    $this->assertSession()->statusCodeEquals(403);

    // Test that admin users can access edit form.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/structure/crm/relationship_types/manage/functional_edit_access_test');
    $this->assertSession()->statusCodeEquals(200);

    // Verify it's the edit form by checking the page title.
    $this->assertSession()->pageTextContains('Edit Test Edit Type crm relationship type');
  }

  /**
   * Tests form field visibility based on asymmetric checkbox.
   */
  public function testFieldVisibilityStates(): void {
    $this->drupalLogin($this->adminUser);

    $this->drupalGet('admin/structure/crm/relationship_types/add');

    // By default, asymmetric should be checked (value = 1 for new entities).
    $this->assertSession()->checkboxChecked('asymmetric');

    // The label_b and contact_type_b fields should be visible when asymmetric
    // is checked.
    // Note: We can't easily test JavaScript-based visibility in functional
    // tests, but we can verify the fields exist and have the proper states
    // configuration.
    $this->assertSession()->fieldExists('label_b');
    $this->assertSession()->fieldExists('contact_type_b[]');
  }

  /**
   * Tests creating relationship types with different contact type combinations.
   */
  public function testContactTypeCombinations(): void {
    $this->drupalLogin($this->adminUser);

    // Test person to organization relationship.
    $this->drupalGet('admin/structure/crm/relationship_types/add');
    $edit = [
      'label' => 'Employee-Company',
      'id' => 'func_employee_company',
      'asymmetric' => TRUE,
      'label_a' => 'Employee',
      'contact_type_a[]' => ['person'],
      'label_b' => 'Company',
      'contact_type_b[]' => ['organization'],
    ];
    $this->submitForm($edit, 'Save relationship type');

    $relationship_type = RelationshipType::load('func_employee_company');
    $this->assertEquals(['person'], $relationship_type->get('contact_type_a'));
    $this->assertEquals(['organization'], $relationship_type->get('contact_type_b'));

    // Test household to person relationship.
    $this->drupalGet('admin/structure/crm/relationship_types/add');
    $edit = [
      'label' => 'Household-Member',
      'id' => 'func_household_member',
      'asymmetric' => TRUE,
      'label_a' => 'Household',
      'contact_type_a[]' => ['household'],
      'label_b' => 'Member',
      'contact_type_b[]' => ['person'],
    ];
    $this->submitForm($edit, 'Save relationship type');

    $household_relationship = RelationshipType::load('func_household_member');
    $this->assertNotNull($household_relationship, 'household_member relationship type should exist');

    // Debug the actual values if assertion fails.
    $actual_contact_type_a = $household_relationship->get('contact_type_a');
    $actual_contact_type_b = $household_relationship->get('contact_type_b');

    $this->assertEquals(['household'], $actual_contact_type_a,
      "Expected contact_type_a to be ['household'], got " . var_export($actual_contact_type_a, TRUE));
    $this->assertEquals(['person'], $actual_contact_type_b,
      "Expected contact_type_b to be ['person'], got " . var_export($actual_contact_type_b, TRUE));
  }

  /**
   * Tests form with minimal required data.
   */
  public function testCreateRelationshipTypeMinimal(): void {
    $this->drupalLogin($this->adminUser);

    $this->drupalGet('admin/structure/crm/relationship_types/add');

    // Submit the form with only required fields.
    // Note: label_b and contact_type_b are required fields
    // even when asymmetric is FALSE.
    $edit = [
      'label' => 'Minimal Type',
      'id' => 'func_minimal_type',
      'asymmetric' => FALSE,
      'label_a' => 'Contact A',
      'contact_type_a[]' => ['person'],
      'label_b' => 'Contact A',
      'contact_type_b[]' => ['person'],
    ];
    $this->submitForm($edit, 'Save relationship type');

    $this->assertSession()->pageTextContains('The crm relationship type Minimal Type has been added.');

    // Verify the entity was created.
    $relationship_type = RelationshipType::load('func_minimal_type');
    $this->assertNotNull($relationship_type);
    $this->assertEquals('Minimal Type', $relationship_type->label());
    $this->assertEquals('func_minimal_type', $relationship_type->id());
    $this->assertEquals('', $relationship_type->get('description'));
    // Should default to symmetric (asymmetric = FALSE) if not specified.
    $this->assertEquals('Contact A', $relationship_type->get('label_a'));
    $this->assertEquals(['person'], $relationship_type->get('contact_type_a'));
  }

  /**
   * Tests the form title for edit operations.
   */
  public function testEditFormTitle(): void {
    RelationshipType::create([
      'id' => 'title_test',
      'label' => 'Title Test Type',
      'label_a' => 'Test A',
      'contact_type_a' => ['person'],
      'contact_type_b' => ['person'],
    ])->save();

    $this->drupalLogin($this->adminUser);

    $this->drupalGet('admin/structure/crm/relationship_types/manage/title_test');

    // Check that the edit form has the proper title.
    $this->assertSession()->pageTextContains('Edit Title Test Type crm relationship type');
  }

  /**
   * Tests creating a relationship type with limits.
   */
  public function testCreateRelationshipTypeWithLimits(): void {
    $this->drupalLogin($this->adminUser);

    $this->drupalGet('admin/structure/crm/relationship_types/add');
    $this->assertSession()->statusCodeEquals(200);

    // Check that limit form elements are present.
    $this->assertSession()->fieldExists('limit_a');
    $this->assertSession()->fieldExists('limit_b');
    $this->assertSession()->fieldExists('limit_active_only');

    // Submit the form with limit values.
    $edit = [
      'label' => 'Parent-Child Limited',
      'id' => 'parent_child_limited',
      'description' => 'Parent-child relationship with limits',
      'asymmetric' => TRUE,
      'label_a' => 'Parent',
      'contact_type_a[]' => ['person'],
      'label_b' => 'Child',
      'contact_type_b[]' => ['person'],
      'limit_a' => '',
      'limit_b' => '2',
      'limit_active_only' => TRUE,
    ];
    $this->submitForm($edit, 'Save relationship type');

    // Check that we were redirected to the collection page.
    $this->assertSession()->addressEquals('admin/structure/crm/relationship_types');
    $this->assertSession()->pageTextContains('The crm relationship type Parent-Child Limited has been added.');

    // Verify the entity was created with limit values.
    $relationship_type = RelationshipType::load('parent_child_limited');
    $this->assertNotNull($relationship_type);
    $this->assertNull($relationship_type->getLimitA());
    $this->assertEquals(2, $relationship_type->getLimitB());
    $this->assertTrue($relationship_type->isLimitActiveOnly());
  }

  /**
   * Tests that symmetric relationship types copy limit_a to limit_b.
   */
  public function testSymmetricRelationshipTypeLimits(): void {
    $this->drupalLogin($this->adminUser);

    $this->drupalGet('admin/structure/crm/relationship_types/add');

    $edit = [
      'label' => 'Partner Limited',
      'id' => 'partner_limited',
      'asymmetric' => FALSE,
      'label_a' => 'Partner',
      'contact_type_a[]' => ['person'],
      'label_b' => 'Partner',
      'contact_type_b[]' => ['person'],
      'limit_a' => '1',
      'limit_active_only' => FALSE,
    ];
    $this->submitForm($edit, 'Save relationship type');

    $this->assertSession()->pageTextContains('The crm relationship type Partner Limited has been added.');

    // Verify limit_b was set to match limit_a for symmetric relationships.
    $relationship_type = RelationshipType::load('partner_limited');
    $this->assertNotNull($relationship_type);
    $this->assertEquals(1, $relationship_type->getLimitA());
    $this->assertEquals(1, $relationship_type->getLimitB());
    $this->assertFalse($relationship_type->isLimitActiveOnly());
  }

  /**
   * Tests editing a relationship type's limit values.
   */
  public function testEditRelationshipTypeLimits(): void {
    // Create a relationship type with limits.
    $relationship_type = RelationshipType::create([
      'id' => 'limit_edit_test',
      'label' => 'Limit Edit Test',
      'asymmetric' => TRUE,
      'label_a' => 'Test A',
      'contact_type_a' => ['person'],
      'label_b' => 'Test B',
      'contact_type_b' => ['person'],
      'limit_a' => 5,
      'limit_b' => 3,
      'limit_active_only' => FALSE,
    ]);
    $relationship_type->save();

    $this->drupalLogin($this->adminUser);

    $this->drupalGet('admin/structure/crm/relationship_types/manage/limit_edit_test');
    $this->assertSession()->statusCodeEquals(200);

    // Check that the form is pre-populated with existing limit values.
    $this->assertSession()->fieldValueEquals('limit_a', '5');
    $this->assertSession()->fieldValueEquals('limit_b', '3');
    $this->assertSession()->checkboxNotChecked('limit_active_only');

    // Update the limits.
    $edit = [
      'limit_a' => '10',
      'limit_b' => '',
      'limit_active_only' => TRUE,
    ];
    $this->submitForm($edit, 'Save relationship type');

    $this->assertSession()->pageTextContains('The crm relationship type Limit Edit Test has been updated.');

    // Verify the entity was updated.
    $updated_type = RelationshipType::load('limit_edit_test');
    $this->assertEquals(10, $updated_type->getLimitA());
    $this->assertNull($updated_type->getLimitB());
    $this->assertTrue($updated_type->isLimitActiveOnly());
  }

}
