<?php

namespace Drupal\Tests\crm\Functional\Plugin\Menu\LocalTask;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Tests\crm\Functional\ContactTestBase;

/**
 * Tests the Comments local task plugin.
 *
 * @group crm
 */
class CommentsTest extends ContactTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['crm', 'comment', 'block'];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'claro';

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Install comment field configuration for CRM contacts.
    // Since comment fields are now optional, we need to explicitly create them
    // only if they don't already exist.
    $entity_type_manager = \Drupal::entityTypeManager();

    // Create comment type if it doesn't exist.
    $comment_type_storage = $entity_type_manager->getStorage('comment_type');
    if (!$comment_type_storage->load('crm_contact')) {
      $comment_type = $comment_type_storage->create([
        'id' => 'crm_contact',
        'label' => 'Contact Comment',
        'target_entity_type_id' => 'crm_contact',
        'description' => 'Comment type for CRM contacts',
      ]);
      $comment_type->save();
    }

    // Create field storage if it doesn't exist.
    $field_storage_storage = $entity_type_manager->getStorage('field_storage_config');
    $field_storage = $field_storage_storage->load('crm_contact.comment');
    if (!$field_storage) {
      $field_storage = $field_storage_storage->create([
        'field_name' => 'comment',
        'entity_type' => 'crm_contact',
        'type' => 'comment',
        'settings' => [
          'comment_type' => 'crm_contact',
        ],
      ]);
      $field_storage->save();
    }

    // Create field instance for organization bundle if it doesn't exist.
    $field_config_storage = $entity_type_manager->getStorage('field_config');
    if (!$field_config_storage->load('crm_contact.organization.comment')) {
      $field = $field_config_storage->create([
        'field_storage' => $field_storage,
        'bundle' => 'organization',
        'label' => 'Comment',
        'default_value' => [
          [
            'status' => 2,
            'cid' => 0,
            'last_comment_timestamp' => 0,
            'last_comment_name' => NULL,
            'last_comment_uid' => 0,
            'comment_count' => 0,
          ],
        ],
      ]);
      $field->save();
    }

    // Rebuild the container to ensure the comment lazy builder
    // service is available.
    $this->rebuildContainer();
  }

  /**
   * Test that the local task title displays 'Comments (3)'.
   */
  public function testLocalTaskTitle() {
    // Create a new CRM contact entity.
    $contact = $this->createContact();

    // Verify the contact has the comment field.
    $this->assertTrue($contact->hasField('comment'), 'Contact entity has comment field');

    // Create some comments on the contact entity.
    $this->createComment($contact, 'Comment 1');
    $this->createComment($contact, 'Comment 2');
    $this->createComment($contact, 'Comment 3');

    $user = $this->drupalCreateUser([
      'view any crm_contact',
      'access comments',
    ]);

    $this->drupalLogin($user);

    // Visit the canonical contact page where local tasks are displayed.
    $this->drupalGet('/crm/contact/' . $contact->id());

    // Check that the local task tab shows the correct count.
    $this->assertSession()->pageTextContains('Comments (3)');
  }

  /**
   * Test local task title displays 'Comment (1)'.
   */
  public function testLocalTaskTitleSingle() {
    // Create a new CRM contact entity.
    $contact = $this->createContact();

    // Create some comments on the contact entity.
    $this->createComment($contact, 'Comment 1');

    $user = $this->drupalCreateUser([
      'view any crm_contact',
      'access comments',
    ]);

    $this->drupalLogin($user);

    // Visit the canonical contact page where local tasks are displayed.
    $this->drupalGet('/crm/contact/' . $contact->id());

    // Check that the local task tab shows the correct count.
    $this->assertSession()->pageTextContains('Comment (1)');
  }

  /**
   * Test that the local task title displays 'Comments (0)'.
   */
  public function testLocalTaskTitleNone() {
    // Create a new CRM contact entity.
    $contact = $this->createContact();

    $user = $this->drupalCreateUser([
      'view any crm_contact',
      'access comments',
    ]);

    $this->drupalLogin($user);

    // Visit the canonical contact page where local tasks are displayed.
    $this->drupalGet('/crm/contact/' . $contact->id());

    // Check that the local task tab shows the correct count.
    $this->assertSession()->pageTextContains('Comments (0)');
  }

  /**
   * Helper method to create a new CRM contact entity.
   *
   * @return \Drupal\Core\Entity\EntityInterface
   *   The newly created entity.
   */
  protected function createContact() {
    $values = [
      'name' => $this->randomString(),
      'bundle' => 'organization',
    ];
    $contact = $this->container
      ->get('entity_type.manager')
      ->getStorage('crm_contact')
      ->create($values);
    $contact->save();

    return $contact;
  }

  /**
   * Helper method to create a new comment on a CRM contact entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $contact
   *   The CRM contact entity.
   * @param string $text
   *   The comment text.
   */
  protected function createComment(EntityInterface $contact, $text) {
    $values = [
      'entity_type' => 'crm_contact',
      'entity_id' => $contact->id(),
      'comment_type' => 'crm_contact',
      'subject' => $text,
      'field_name' => 'comment',
      'status' => 1,
    ];
    $comment = $this->container->get('entity_type.manager')
      ->getStorage('comment')
      ->create($values);
    $comment->save();

  }

}
