<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Functional\Plugin\Search;

use Drupal\Tests\crm\Functional\ContactTestBase;

/**
 * Tests the contact search plugin.
 *
 * @group crm
 */
class ContactSearchTest extends ContactTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'crm',
    'datetime',
    'search',
    'block',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * A user with search permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $searchUser;

  /**
   * An admin user.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * The search index service.
   *
   * @var \Drupal\search\SearchIndexInterface
   */
  protected $searchIndex;

  /**
   * The initial contact count before tests.
   *
   * @var int
   */
  protected int $initialContactCount;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->searchIndex = $this->container->get('search.index');

    // Create the search page entity.
    $this->container->get('entity_type.manager')
      ->getStorage('search_page')
      ->create([
        'id'     => 'crm_contact_search',
        'label'  => 'Contacts',
        'path'   => 'contact_search',
        'plugin' => 'crm_contact_search',
      ])
      ->save();

    // Rebuild the router to register the new search page route.
    $this->container->get('router.builder')->rebuild();

    // Create users with different permissions.
    $this->searchUser = $this->drupalCreateUser([
      'search content',
      'view any crm_contact',
    ]);

    $this->adminUser = $this->drupalCreateUser([
      'search content',
      'administer crm',
      'administer search',
    ]);

    // Store the initial contact count (some contacts may be auto-created).
    $this->initialContactCount = (int) $this->container->get('database')
      ->query('SELECT COUNT(*) FROM {crm_contact}')
      ->fetchField();

    // Index any existing contacts first.
    $this->getSearchPlugin()->updateIndex();

    // Place the search block.
    $this->drupalPlaceBlock('search_form_block', [
      'page_id' => 'crm_contact_search',
    ]);
  }

  /**
   * Tests that contacts can be indexed and searched.
   */
  public function testContactIndexing(): void {
    // Get initial state.
    $plugin        = $this->getSearchPlugin();
    $initialStatus = $plugin->indexStatus();
    $initialTotal  = (int) $initialStatus['total'];

    // Create a contact with a unique name (single word, long enough).
    $this->drupalCreateContact([
      'name'   => 'Lisa Simpson',
      'bundle' => 'person',
      'status' => 1,
    ]);

    // Check index status - should have one more contact needing indexing.
    $status = $plugin->indexStatus();
    $this->assertEquals($initialTotal + 1, (int) $status['total']);
    $this->assertGreaterThan(0, (int) $status['remaining']);

    // Index the new contact.
    $plugin->updateIndex();

    // Check index status after indexing.
    $status = $plugin->indexStatus();
    $this->assertEquals(0, (int) $status['remaining']);

    // Search for the contact.
    $this->drupalLogin($this->searchUser);
    $this->drupalGet('search/contact_search');
    $this->assertSession()->statusCodeEquals(200);
    $this->submitForm(['keys' => 'Lisa Simpson'], 'Search');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Lisa Simpson');
  }

  /**
   * Tests searching by contact name.
   */
  public function testSearchByName(): void {
    // Create a contact with a unique single-word name.
    $this->drupalCreateContact([
      'name'   => 'Bart Simpson',
      'bundle' => 'person',
      'status' => 1,
    ]);

    // Index the contact.
    $this->getSearchPlugin()->updateIndex();

    // Search for the contact.
    $this->drupalLogin($this->searchUser);
    $this->drupalGet('search/contact_search');
    $this->assertSession()->statusCodeEquals(200);
    $this->submitForm(['keys' => 'Bart Simpson'], 'Search');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Bart Simpson');
  }

  /**
   * Tests permission restrictions for search.
   */
  public function testSearchPermissions(): void {
    // Create an inactive contact with a unique identifier.
    $inactive = $this->drupalCreateContact([
      'name'   => 'Mona Simpson',
      'bundle' => 'person',
      'status' => 0,
    ]);

    // Create an active contact with a unique identifier.
    $active = $this->drupalCreateContact([
      'name'   => 'Homer Simpson',
      'bundle' => 'person',
      'status' => 1,
    ]);

    // Index contacts.
    $plugin = $this->getSearchPlugin();
    $plugin->updateIndex();

    // Verify both contacts are indexed.
    $status = $plugin->indexStatus();
    $this->assertEquals(0, (int) $status['remaining'], 'All contacts should be indexed');

    // Regular user should only see active contacts.
    $this->drupalLogin($this->searchUser);
    $this->drupalGet('search/contact_search');
    $this->assertSession()->statusCodeEquals(200);
    $this->submitForm(['keys' => 'Homer Simpson'], 'Search');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Homer Simpson');
    $this->assertSession()->pageTextNotContains('Mona Simpson');

    // Admin should see both when searching.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('search/contact_search');
    $this->submitForm(['keys' => 'Mona Simpson'], 'Search');
    $this->assertSession()->pageTextContains('Mona Simpson');
  }

  /**
   * Tests that contacts are marked for reindex when updated.
   */
  public function testReindexOnUpdate(): void {
    // Create a contact with a unique name.
    $contact = $this->drupalCreateContact([
      'name'   => 'Homer Simpson',
      'bundle' => 'person',
      'status' => 1,
    ]);

    // Index the contact.
    $plugin = $this->getSearchPlugin();
    $plugin->updateIndex();

    // Verify indexed.
    $status = $plugin->indexStatus();
    $this->assertEquals(0, (int) $status['remaining']);

    // Update the contact.
    $contact->set('name', 'Homer J. Simpson');
    $contact->save();

    // Should be marked for reindex.
    $status = $plugin->indexStatus();
    $this->assertGreaterThan(0, (int) $status['remaining']);

    // Reindex.
    $plugin->updateIndex();

    // Search should find the updated name.
    $this->drupalLogin($this->searchUser);
    $this->drupalGet('search/contact_search');
    $this->assertSession()->statusCodeEquals(200);
    $this->submitForm(['keys' => 'Homer J. Simpson'], 'Search');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Homer J. Simpson');
  }

  /**
   * Tests the indexClear functionality.
   */
  public function testIndexClear(): void {
    // Create contacts.
    $this->drupalCreateContact([
      'name'   => 'Lisa Simpson',
      'bundle' => 'person',
      'status' => 1,
    ]);
    $this->drupalCreateContact([
      'name'   => 'Bart Simpson',
      'bundle' => 'person',
      'status' => 1,
    ]);

    // Index contacts.
    $plugin = $this->getSearchPlugin();
    $plugin->updateIndex();

    // Verify indexed.
    $status = $plugin->indexStatus();
    $this->assertEquals(0, (int) $status['remaining']);

    // Clear the index.
    $plugin->indexClear();

    // All contacts should need reindexing.
    $status = $plugin->indexStatus();
    $this->assertEquals($status['total'], $status['remaining']);
  }

  /**
   * Tests the markForReindex functionality.
   */
  public function testMarkForReindex(): void {
    // Create a contact.
    $this->drupalCreateContact([
      'name'   => 'Maggie Simpson',
      'bundle' => 'person',
      'status' => 1,
    ]);

    // Index the contact.
    $plugin = $this->getSearchPlugin();
    $plugin->updateIndex();

    // Verify indexed.
    $status = $plugin->indexStatus();
    $this->assertEquals(0, (int) $status['remaining']);

    // Mark for reindex.
    $plugin->markForReindex();

    // All should need reindexing.
    $status = $plugin->indexStatus();
    $this->assertEquals($status['total'], $status['remaining']);
  }

  /**
   * Tests search with multiple contact types.
   */
  public function testMultipleContactTypes(): void {
    // Create different types of contacts with unique names.
    $this->drupalCreateContact([
      'name'   => 'Lisa Simpson',
      'bundle' => 'person',
      'status' => 1,
    ]);
    $this->drupalCreateContact([
      'name'   => 'Bart Simpson',
      'bundle' => 'organization',
      'status' => 1,
    ]);
    $this->drupalCreateContact([
      'name'   => 'Maggie Simpson',
      'bundle' => 'household',
      'status' => 1,
    ]);

    // Index all contacts.
    $this->getSearchPlugin()->updateIndex();

    // Search should find each type.
    $this->drupalLogin($this->searchUser);

    $this->drupalGet('search/contact_search');
    $this->submitForm(['keys' => 'Lisa Simpson'], 'Search');
    $this->assertSession()->pageTextContains('Lisa Simpson');

    $this->drupalGet('search/contact_search');
    $this->submitForm(['keys' => 'Bart Simpson'], 'Search');
    $this->assertSession()->pageTextContains('Bart Simpson');

    $this->drupalGet('search/contact_search');
    $this->submitForm(['keys' => 'Maggie Simpson'], 'Search');
    $this->assertSession()->pageTextContains('Maggie Simpson');
  }

  /**
   * Helper method to create a contact.
   */
  protected function drupalCreateContact(array $values): object {
    $contact = $this->container->get('entity_type.manager')
      ->getStorage('crm_contact')
      ->create($values);
    $contact->save();
    return $contact;
  }

  /**
   * Gets the contact search plugin.
   */
  protected function getSearchPlugin(): object {
    $plugin_id              = 'crm_contact_search';
    $search_page_repository = $this->container->get('search.search_page_repository');
    return $search_page_repository->getActiveSearchPages()[$plugin_id]->getPlugin();
  }

}
