<?php

namespace Drupal\Tests\crm\Functional;

use Drupal\Tests\BrowserTestBase;
use Drupal\crm\Entity\Contact;
use Drupal\crm\Entity\Relationship;
use Drupal\crm\Entity\RelationshipType;

/**
 * Tests the RelationshipController.
 *
 * @group crm
 */
class RelationshipControllerTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['crm'];

  /**
   * Tests the RelationshipController.
   */
  public function testRelationshipController(): void {

    $contact_a = Contact::create([
      'bundle' => 'person',
      'name' => 'Some Guy',
    ]);
    $contact_a->save();
    $a_id = $contact_a->id();
    $contact_b = Contact::create([
      'bundle' => 'organization',
      'name' => 'Test Organization',
    ]);
    $contact_b->save();

    // Create a CRM relationship entity.
    $relationship = Relationship::create([
      'contacts' => [$contact_a->id(), $contact_b->id()],
      'bundle' => 'employee',
      'status' => 1,
      'start_date' => strtotime('2023-01-01'),
      'end_date' => strtotime('2023-12-31'),
    ]);
    $relationship->save();

    $user = $this->drupalCreateUser([
      'view any crm_contact',
      'view crm relationship',
      'edit crm relationship',
      'delete crm relationship',
      'administer crm',
    ]);

    $this->drupalLogin($user);

    $this->drupalGet('admin/content/crm/relationship');
    $this->assertSession()->statusCodeEquals(200);

    // Visit the path handled by the RelationshipController.
    $this->drupalGet('crm/contact/' . $a_id . '/relationship');
    $this->assertSession()->statusCodeEquals(200);

    // Assert that the active relationship is displayed in the table.
    $this->assertSession()->responseContains('There are no inactive relationships.');
    $this->assertSession()->responseContains($relationship->bundle->entity->get('label_b'));
    $this->assertSession()->responseContains($relationship->get('contact_b')->entity->label());

    // Assert that the edit and delete links are present.
    $this->assertSession()->linkByHrefExists($relationship->toUrl('edit-form')->toString());
    $this->assertSession()->linkByHrefExists($relationship->toUrl('delete-form')->toString());

    // Update the relationship status to inactive.
    $relationship->set('status', 0)->save();

    // Visit the page again and
    // assert that the inactive relationship is displayed.
    $this->drupalGet('crm/contact/' . $a_id . '/relationship');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->responseContains('There are no active relationships.');
    $this->assertSession()->responseContains($relationship->bundle->entity->get('label_b'));
    $this->assertSession()->responseContains($relationship->get('contact_b')->entity->label());

    // Visit the page again and
    // assert that the inactive relationship is displayed.
    $relationship_1 = Relationship::load(1);
    $relationship_1->delete();
    $this->drupalGet('admin/content/crm/relationship');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->responseContains('There are no crm relationships yet.');

  }

  /**
   * Tests the Add Relationship page shows eligible types for a person contact.
   */
  public function testAddRelationshipPageForPerson(): void {
    $person = Contact::create([
      'bundle' => 'person',
      'name' => 'Test Person',
    ]);
    $person->save();

    $user = $this->drupalCreateUser([
      'view any crm_contact',
      'administer crm',
    ]);

    $this->drupalLogin($user);

    // Visit the add relationship page for the person contact.
    $this->drupalGet('crm/contact/' . $person->id() . '/relationship/add');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Add Relationship');

    // The employee relationship type should be available for a person.
    // Person is eligible for position A (the employee), so we show the
    // inverted label (label_b) which describes what this contact IS.
    $employee_type = RelationshipType::load('employee');
    $this->assertSession()->pageTextContains($employee_type->get('label_b'));
  }

  /**
   * Tests the Add Relationship page shows eligible types for an organization.
   */
  public function testAddRelationshipPageForOrganization(): void {
    $organization = Contact::create([
      'bundle' => 'organization',
      'name' => 'Test Organization',
    ]);
    $organization->save();

    $user = $this->drupalCreateUser([
      'view any crm_contact',
      'administer crm',
    ]);

    $this->drupalLogin($user);

    // Visit the add relationship page for the organization contact.
    $this->drupalGet('crm/contact/' . $organization->id() . '/relationship/add');
    $this->assertSession()->statusCodeEquals(200);

    // The employee relationship type should be available for an organization.
    // Organization is eligible for position B (the employer), so we show the
    // inverted label (label_a) which describes what this contact IS.
    $employee_type = RelationshipType::load('employee');
    $this->assertSession()->pageTextContains($employee_type->get('label_a'));
  }

  /**
   * Tests the Add Relationship page is accessible from the relationship tab.
   */
  public function testAddRelationshipPageAccessible(): void {
    $person = Contact::create([
      'bundle' => 'person',
      'name' => 'Test Person for Action Link',
    ]);
    $person->save();

    $user = $this->drupalCreateUser([
      'view any crm_contact',
      'administer crm',
    ]);

    $this->drupalLogin($user);

    // Visit the relationship tab for the contact.
    $this->drupalGet('crm/contact/' . $person->id() . '/relationship');
    $this->assertSession()->statusCodeEquals(200);

    // Visit the add relationship page directly.
    $this->drupalGet('crm/contact/' . $person->id() . '/relationship/add');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Add Relationship');
  }

  /**
   * Tests the Add Relationship page shows message when no eligible types.
   */
  public function testAddRelationshipPageNoEligibleTypes(): void {
    // Create a relationship type that only allows household contacts.
    // Use a unique ID to avoid conflicts with existing config.
    $type = RelationshipType::load('test_household_only');
    if (!$type) {
      RelationshipType::create([
        'id' => 'test_household_only',
        'label' => 'Test Household Only',
        'asymmetric' => FALSE,
        'contact_type_a' => ['household'],
        'contact_type_b' => ['household'],
        'label_a' => 'Test Household Only',
        'label_b' => 'Test Household Only',
      ])->save();
    }

    // Create a person contact (not a household).
    $person = Contact::create([
      'bundle' => 'person',
      'name' => 'Test Person No Types',
    ]);
    $person->save();

    $user = $this->drupalCreateUser([
      'view any crm_contact',
      'administer crm',
    ]);

    $this->drupalLogin($user);

    $this->drupalGet('crm/contact/' . $person->id() . '/relationship/add');
    $this->assertSession()->statusCodeEquals(200);

    // Test Household Only should not be shown for a person contact.
    $this->assertSession()->pageTextNotContains('Test Household Only');
  }

  /**
   * Tests the form pre-populates contact from query parameters.
   */
  public function testRelationshipFormPrepopulatesContact(): void {
    $person = Contact::create([
      'bundle' => 'person',
      'name' => 'Prepopulate Test Person',
    ]);
    $person->save();

    $user = $this->drupalCreateUser([
      'view any crm_contact',
      'administer crm',
      'create crm relationship',
    ]);

    $this->drupalLogin($user);

    // Visit the add form with contact_a query parameter.
    $this->drupalGet('crm/relationship/add/employee', [
      'query' => [
        'contact_a' => $person->id(),
      ],
    ]);
    $this->assertSession()->statusCodeEquals(200);

    // The contact_a field should be pre-populated and contain the contact ID.
    $field = $this->assertSession()->fieldExists('contact_a[0][target_id]');
    $this->assertStringContainsString('(' . $person->id() . ')', $field->getValue());
  }

}
