<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Functional;

use Drupal\crm\Entity\Contact;
use Drupal\crm\Entity\UserContactMapping;
use Drupal\name\Entity\NameFormat;
use Drupal\Tests\BrowserTestBase;

/**
 * Tests user form alterations functionality.
 *
 * @group crm
 */
class UserFormAlterTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'claro';

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'address',
    'crm',
    'datetime',
    'name',
    'telephone',
    'block',
  ];

  /**
   * The user data service.
   *
   * @var \Drupal\user\UserDataInterface
   */
  protected $userData;

  /**
   * Test user with CRM permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $crmUser;

  /**
   * Test user without CRM permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $regularUser;

  /**
   * Test contact entity.
   *
   * @var \Drupal\crm\Entity\Contact
   */
  protected $testContact;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->userData = $this->container->get('user.data');

    // Enable the display name override feature.
    $config = $this->container->get('config.factory')->getEditable('crm.user_contact_mapping.settings');
    $config->set('display_name', TRUE);
    $config->save();

    // Create a person contact.
    $this->testContact = Contact::create([
      'bundle' => 'person',
      'name' => 'Test Contact',
      'full_name' => [
        'title' => 'Mr.',
        'given' => 'John',
        'middle' => 'Q',
        'family' => 'Doe',
        'generational' => 'Jr.',
      ],
      'preferred_name' => 'Johnny',
      'aliases' => [
        ['value' => 'JD'],
        ['value' => 'John D'],
      ],
    ]);
    $this->testContact->save();

    // Create a user with CRM permissions.
    $this->crmUser = $this->drupalCreateUser([
      'crm user alter display name',
      'access user profiles',
      'change own username',
    ]);

    // Create user contact mapping for CRM user.
    $crm_user_contact_mapping = UserContactMapping::create([
      'user' => $this->crmUser->id(),
      'crm_contact' => $this->testContact->id(),
    ]);
    $crm_user_contact_mapping->save();

    // Create a regular user without CRM permissions.
    $this->regularUser = $this->drupalCreateUser([
      'access user profiles',
      'change own username',
    ]);

    // Create additional name formats for testing.
    $name_format = NameFormat::create([
      'id' => 'test_given_family',
      'label' => 'Given Family',
      'pattern' => 'g f',
    ]);
    $name_format->save();

    $name_format = NameFormat::create([
      'id' => 'test_family_given',
      'label' => 'Family, Given',
      'pattern' => 'f, g',
    ]);
    $name_format->save();

    // Place the local tasks block to see form tabs.
    $this->drupalPlaceBlock('local_tasks_block');
  }

  /**
   * Tests that CRM user sees the name format field.
   */
  public function testCrmUserSeesNameFormatField(): void {
    $this->drupalLogin($this->crmUser);
    $this->drupalGet('user/' . $this->crmUser->id() . '/edit');

    // Check that the CRM details group is present.
    $this->assertSession()->elementExists('css', 'details[data-drupal-selector="edit-crm"]');
    $this->assertSession()->pageTextContains('CRM');

    // Check that the name format field is present.
    $this->assertSession()->fieldExists('name_format');
    $this->assertSession()->pageTextContains('Name Format');

    // Check that the field has the expected options.
    $this->assertSession()->optionExists('name_format', '_none');
    $this->assertSession()->optionExists('name_format', 'test_given_family');
    $this->assertSession()->optionExists('name_format', 'test_family_given');
  }

  /**
   * Tests that regular user doesn't see the name format field.
   */
  public function testRegularUserDoesNotSeeNameFormatField(): void {
    $this->drupalLogin($this->regularUser);
    $this->drupalGet('user/' . $this->regularUser->id() . '/edit');

    // Check that the CRM details group is not present.
    $this->assertSession()->elementNotExists('css', 'details[data-drupal-selector="edit-crm"]');
    $this->assertSession()->pageTextNotContains('Name Format');
  }

  /**
   * Tests that user without associated contact doesn't see the field.
   */
  public function testUserWithoutContactDoesNotSeeField(): void {
    // Create a user with CRM permission but no associated contact.
    $user_no_contact = $this->drupalCreateUser([
      'crm user alter display name',
      'access user profiles',
      'change own username',
    ]);

    $this->drupalLogin($user_no_contact);
    $this->drupalGet('user/' . $user_no_contact->id() . '/edit');

    // Check that the CRM details group is not present.
    $this->assertSession()->elementNotExists('css', 'details[data-drupal-selector="edit-crm"]');
    $this->assertSession()->pageTextNotContains('Name Format');
  }

  /**
   * Tests form submission with name format selection.
   */
  public function testNameFormatFormSubmission(): void {
    $this->drupalLogin($this->crmUser);
    $this->drupalGet('user/' . $this->crmUser->id() . '/edit');

    // Submit the form with a specific name format.
    $edit = [
      'name_format' => 'test_given_family',
    ];
    $this->submitForm($edit, 'Save');

    // Check that the form was submitted successfully.
    $this->assertSession()->pageTextContains('The changes have been saved.');

    // Verify that the user data was saved correctly.
    $saved_format = $this->userData->get('crm', $this->crmUser->id(), 'name_format');
    $this->assertEquals('test_given_family', $saved_format);
  }

  /**
   * Tests form submission with _none option to clear custom format.
   */
  public function testNameFormatClearCustomFormat(): void {
    // First, set a custom format.
    $this->userData->set('crm', $this->crmUser->id(), 'name_format', 'test_family_given');

    $this->drupalLogin($this->crmUser);
    $this->drupalGet('user/' . $this->crmUser->id() . '/edit');

    // Verify the field shows the current value.
    $this->assertSession()->fieldValueEquals('name_format', 'test_family_given');

    // Submit the form with _none to clear the custom format.
    $edit = [
      'name_format' => '_none',
    ];
    $this->submitForm($edit, 'Save');

    // Check that the form was submitted successfully.
    $this->assertSession()->pageTextContains('The changes have been saved.');

    // Verify that the user data was cleared.
    $saved_format = $this->userData->get('crm', $this->crmUser->id(), 'name_format');
    $this->assertNull($saved_format);
  }

  /**
   * Tests that other users cannot edit each other's name format.
   */
  public function testCannotEditOtherUsersNameFormat(): void {
    // Create another CRM user.
    $other_crm_user = $this->drupalCreateUser([
      'crm user alter display name',
      'access user profiles',
      'change own username',
    ]);

    // Create user contact mapping for the other user.
    $other_contact = Contact::create([
      'bundle' => 'person',
      'name' => 'Other Contact',
      'full_name' => [
        'given' => 'Jane',
        'family' => 'Smith',
      ],
    ]);
    $other_contact->save();

    $other_user_contact_mapping = UserContactMapping::create([
      'user' => $other_crm_user->id(),
      'crm_contact' => $other_contact->id(),
    ]);
    $other_user_contact_mapping->save();

    // Login as the first CRM user and try to edit the other user's profile.
    $this->drupalLogin($this->crmUser);
    $this->drupalGet('user/' . $other_crm_user->id() . '/edit');

    // Should not see the name format field since it's not the same user.
    $this->assertSession()->elementNotExists('css', 'details[data-drupal-selector="edit-crm"]');
    $this->assertSession()->pageTextNotContains('Name Format');
  }

  /**
   * Tests field default value reflects current user data.
   */
  public function testFieldDefaultValueFromUserData(): void {
    // Set a custom format in user data.
    $this->userData->set('crm', $this->crmUser->id(), 'name_format', 'test_family_given');

    $this->drupalLogin($this->crmUser);
    $this->drupalGet('user/' . $this->crmUser->id() . '/edit');

    // Verify the field shows the current value as default.
    $this->assertSession()->fieldValueEquals('name_format', 'test_family_given');
  }

  /**
   * Tests field shows _none when no custom format is set.
   */
  public function testFieldDefaultValueNone(): void {
    // Ensure no custom format is set.
    $this->userData->delete('crm', $this->crmUser->id(), 'name_format');

    $this->drupalLogin($this->crmUser);
    $this->drupalGet('user/' . $this->crmUser->id() . '/edit');

    // Verify the field shows _none as default.
    $this->assertSession()->fieldValueEquals('name_format', '_none');
  }

  /**
   * Tests behavior when display name feature is disabled.
   */
  public function testFieldNotShownWhenFeatureDisabled(): void {
    // Disable the display name override feature.
    $config = $this->container->get('config.factory')->getEditable('crm.user_contact_mapping.settings');
    $config->set('display_name', FALSE);
    $config->save();

    $this->drupalLogin($this->crmUser);
    $this->drupalGet('user/' . $this->crmUser->id() . '/edit');

    // Check that the CRM details group is not present.
    $this->assertSession()->elementNotExists('css', 'details[data-drupal-selector="edit-crm"]');
    $this->assertSession()->pageTextNotContains('Name Format');
  }

  /**
   * Tests multiple form submissions preserve data correctly.
   */
  public function testMultipleFormSubmissions(): void {
    $this->drupalLogin($this->crmUser);

    // First submission: set to given_family.
    $this->drupalGet('user/' . $this->crmUser->id() . '/edit');
    $edit = ['name_format' => 'test_given_family'];
    $this->submitForm($edit, 'Save');

    $saved_format = $this->userData->get('crm', $this->crmUser->id(), 'name_format');
    $this->assertEquals('test_given_family', $saved_format);

    // Second submission: change to family_given.
    $this->drupalGet('user/' . $this->crmUser->id() . '/edit');
    $edit = ['name_format' => 'test_family_given'];
    $this->submitForm($edit, 'Save');

    $saved_format = $this->userData->get('crm', $this->crmUser->id(), 'name_format');
    $this->assertEquals('test_family_given', $saved_format);

    // Third submission: clear with _none.
    $this->drupalGet('user/' . $this->crmUser->id() . '/edit');
    $edit = ['name_format' => '_none'];
    $this->submitForm($edit, 'Save');

    $saved_format = $this->userData->get('crm', $this->crmUser->id(), 'name_format');
    $this->assertNull($saved_format);
  }

}
