<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Kernel;

use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;
use Drupal\crm\Entity\Contact;
use Drupal\crm\Entity\UserContactMapping;
use Drupal\user\Entity\Role;
use Drupal\user\Entity\User;

/**
 * Tests the ContactAccessControlHandler.
 *
 * @group crm
 */
class ContactAccessControlHandlerTest extends EntityKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'address',
    'crm',
    'inline_entity_form',
    'primary_entity_reference',
    'datetime',
    'name',
    'telephone',
  ];

  /**
   * The contact entity for testing.
   *
   * @var \Drupal\crm\CrmContactInterface
   */
  protected $contact;

  /**
   * A user with contact permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $privilegedUser;

  /**
   * A user without contact permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $unprivilegedUser;

  /**
   * A user with administer crm permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * A user with mapped contact permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $mappedUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('crm_contact');
    $this->installEntitySchema('crm_contact_detail');
    $this->installEntitySchema('crm_user_contact_mapping');

    $this->installConfig(['crm', 'name', 'user']);

    // Create a contact for testing.
    $this->contact = Contact::create([
      'bundle' => 'person',
      'full_name' => ['given' => 'Test', 'family' => 'Contact'],
    ]);
    $this->contact->save();

    // Create a role with contact permissions.
    $privileged_role = Role::create([
      'id' => 'contact_manager',
      'label' => 'Contact Manager',
    ]);
    $privileged_role->grantPermission('view any crm_contact');
    $privileged_role->grantPermission('edit any crm_contact');
    $privileged_role->grantPermission('delete any crm_contact');
    $privileged_role->grantPermission('create crm_contact');
    $privileged_role->save();

    // Create a user with contact permissions.
    $this->privilegedUser = User::create([
      'name' => 'privileged_user',
      'mail' => 'privileged@example.com',
      'status' => 1,
      'roles' => ['contact_manager'],
    ]);
    $this->privilegedUser->save();

    // Create a user without contact permissions.
    $this->unprivilegedUser = User::create([
      'name' => 'unprivileged_user',
      'mail' => 'unprivileged@example.com',
      'status' => 1,
    ]);
    $this->unprivilegedUser->save();

    // Create a role with administer crm permission.
    $admin_role = Role::create([
      'id' => 'crm_admin',
      'label' => 'CRM Administrator',
    ]);
    $admin_role->grantPermission('administer crm');
    $admin_role->save();

    // Create a user with administer crm permission.
    $this->adminUser = User::create([
      'name' => 'admin_user',
      'mail' => 'admin@example.com',
      'status' => 1,
      'roles' => ['crm_admin'],
    ]);
    $this->adminUser->save();

    // Create a role with mapped contact permissions.
    $mapped_role = Role::create([
      'id' => 'mapped_contact_user',
      'label' => 'Mapped Contact User',
    ]);
    $mapped_role->grantPermission('view mapped crm_contact');
    $mapped_role->grantPermission('edit mapped crm_contact');
    $mapped_role->save();

    // Create a user with mapped contact permissions.
    $this->mappedUser = User::create([
      'name' => 'mapped_user',
      'mail' => 'mapped@example.com',
      'status' => 1,
      'roles' => ['mapped_contact_user'],
    ]);
    $this->mappedUser->save();
  }

  /**
   * Tests view access with 'view any crm_contact' permission.
   */
  public function testViewAccessWithPermission(): void {
    $access = $this->contact->access('view', $this->privilegedUser, TRUE);
    $this->assertTrue($access->isAllowed(), 'User with view any crm_contact permission can view contacts.');
  }

  /**
   * Tests view access without any permissions.
   */
  public function testViewAccessWithoutPermission(): void {
    $access = $this->contact->access('view', $this->unprivilegedUser, TRUE);
    $this->assertFalse($access->isAllowed(), 'User without permissions cannot view contacts.');
  }

  /**
   * Tests view access with bundle-specific permission.
   */
  public function testViewAccessWithBundlePermission(): void {
    // Create a role with bundle-specific permission.
    $bundle_role = Role::create([
      'id' => 'person_viewer',
      'label' => 'Person Viewer',
    ]);
    $bundle_role->grantPermission('view any person crm_contact');
    $bundle_role->save();

    $bundle_user = User::create([
      'name' => 'bundle_user',
      'mail' => 'bundle@example.com',
      'status' => 1,
      'roles' => ['person_viewer'],
    ]);
    $bundle_user->save();

    $access = $this->contact->access('view', $bundle_user, TRUE);
    $this->assertTrue($access->isAllowed(), 'User with bundle-specific view permission can view contacts of that bundle.');
  }

  /**
   * Tests update access with 'edit any crm_contact' permission.
   */
  public function testUpdateAccessWithPermission(): void {
    $access = $this->contact->access('update', $this->privilegedUser, TRUE);
    $this->assertTrue($access->isAllowed(), 'User with edit any crm_contact permission can update contacts.');
  }

  /**
   * Tests update access without any permissions.
   */
  public function testUpdateAccessWithoutPermission(): void {
    $access = $this->contact->access('update', $this->unprivilegedUser, TRUE);
    $this->assertFalse($access->isAllowed(), 'User without permissions cannot update contacts.');
  }

  /**
   * Tests update access with bundle-specific permission.
   */
  public function testUpdateAccessWithBundlePermission(): void {
    // Create a role with bundle-specific permission.
    $bundle_role = Role::create([
      'id' => 'person_editor',
      'label' => 'Person Editor',
    ]);
    $bundle_role->grantPermission('edit any person crm_contact');
    $bundle_role->save();

    $bundle_user = User::create([
      'name' => 'bundle_editor',
      'mail' => 'bundle_editor@example.com',
      'status' => 1,
      'roles' => ['person_editor'],
    ]);
    $bundle_user->save();

    $access = $this->contact->access('update', $bundle_user, TRUE);
    $this->assertTrue($access->isAllowed(), 'User with bundle-specific edit permission can update contacts of that bundle.');
  }

  /**
   * Tests delete access with 'delete any crm_contact' permission.
   */
  public function testDeleteAccessWithPermission(): void {
    $access = $this->contact->access('delete', $this->privilegedUser, TRUE);
    $this->assertTrue($access->isAllowed(), 'User with delete any crm_contact permission can delete contacts.');
  }

  /**
   * Tests delete access without any permissions.
   */
  public function testDeleteAccessWithoutPermission(): void {
    $access = $this->contact->access('delete', $this->unprivilegedUser, TRUE);
    $this->assertFalse($access->isAllowed(), 'User without permissions cannot delete contacts.');
  }

  /**
   * Tests delete access with bundle-specific permission.
   */
  public function testDeleteAccessWithBundlePermission(): void {
    // Create a role with bundle-specific permission.
    $bundle_role = Role::create([
      'id' => 'person_delete',
      'label' => 'Person Delete',
    ]);
    $bundle_role->grantPermission('delete any person crm_contact');
    $bundle_role->save();

    $bundle_user = User::create([
      'name' => 'bundle_delete',
      'mail' => 'bundle_delete@example.com',
      'status' => 1,
      'roles' => ['person_delete'],
    ]);
    $bundle_user->save();

    $access = $this->contact->access('delete', $bundle_user, TRUE);
    $this->assertTrue($access->isAllowed(), 'User with bundle-specific delete permission can delete contacts of that bundle.');
  }

  /**
   * Tests create access with 'create crm_contact' permission.
   */
  public function testCreateAccessWithPermission(): void {
    $access_handler = $this->entityTypeManager->getAccessControlHandler('crm_contact');
    $access = $access_handler->createAccess('person', $this->privilegedUser, [], TRUE);
    $this->assertTrue($access->isAllowed(), 'User with create crm_contact permission can create contacts.');
  }

  /**
   * Tests create access without any permissions.
   */
  public function testCreateAccessWithoutPermission(): void {
    $access_handler = $this->entityTypeManager->getAccessControlHandler('crm_contact');
    $access = $access_handler->createAccess('person', $this->unprivilegedUser, [], TRUE);
    $this->assertFalse($access->isAllowed(), 'User without permissions cannot create contacts.');
  }

  /**
   * Tests create access with bundle-specific permission.
   */
  public function testCreateAccessWithBundlePermission(): void {
    // Create a role with bundle-specific permission.
    $bundle_role = Role::create([
      'id' => 'person_creator',
      'label' => 'Person Creator',
    ]);
    $bundle_role->grantPermission('create person crm_contact');
    $bundle_role->save();

    $bundle_user = User::create([
      'name' => 'bundle_creator',
      'mail' => 'bundle_creator@example.com',
      'status' => 1,
      'roles' => ['person_creator'],
    ]);
    $bundle_user->save();

    $access_handler = $this->entityTypeManager->getAccessControlHandler('crm_contact');
    $access = $access_handler->createAccess('person', $bundle_user, [], TRUE);
    $this->assertTrue($access->isAllowed(), 'User with bundle-specific create permission can create contacts of that bundle.');
  }

  /**
   * Tests view label access with 'view any crm_contact label' permission.
   */
  public function testViewLabelAccessWithPermission(): void {
    // Create a role with view label permission.
    $label_role = Role::create([
      'id' => 'label_viewer',
      'label' => 'Label Viewer',
    ]);
    $label_role->grantPermission('view any crm_contact label');
    $label_role->save();

    $label_user = User::create([
      'name' => 'label_user',
      'mail' => 'label@example.com',
      'status' => 1,
      'roles' => ['label_viewer'],
    ]);
    $label_user->save();

    $access = $this->contact->access('view label', $label_user, TRUE);
    $this->assertTrue($access->isAllowed(), 'User with view any crm_contact label permission can view contact labels.');
  }

  /**
   * Tests view label access without any permissions.
   */
  public function testViewLabelAccessWithoutPermission(): void {
    $access = $this->contact->access('view label', $this->unprivilegedUser, TRUE);
    $this->assertFalse($access->isAllowed(), 'User without permissions cannot view contact labels.');
  }

  /**
   * Tests view access with administer crm permission.
   */
  public function testViewAccessWithAdminPermission(): void {
    $access = $this->contact->access('view', $this->adminUser, TRUE);
    $this->assertTrue($access->isAllowed(), 'User with administer crm permission can view contacts.');
  }

  /**
   * Tests update access with administer crm permission.
   */
  public function testUpdateAccessWithAdminPermission(): void {
    $access = $this->contact->access('update', $this->adminUser, TRUE);
    $this->assertTrue($access->isAllowed(), 'User with administer crm permission can update contacts.');
  }

  /**
   * Tests delete access with administer crm permission.
   */
  public function testDeleteAccessWithAdminPermission(): void {
    $access = $this->contact->access('delete', $this->adminUser, TRUE);
    $this->assertTrue($access->isAllowed(), 'User with administer crm permission can delete contacts.');
  }

  /**
   * Tests create access with administer crm permission.
   */
  public function testCreateAccessWithAdminPermission(): void {
    $access_handler = $this->entityTypeManager->getAccessControlHandler('crm_contact');
    $access = $access_handler->createAccess('person', $this->adminUser, [], TRUE);
    $this->assertTrue($access->isAllowed(), 'User with administer crm permission can create contacts.');
  }

  /**
   * Tests view access with mapping and 'view mapped crm_contact' permission.
   */
  public function testViewAccessWithMappingAndPermission(): void {
    // Create a user-contact mapping.
    $mapping = UserContactMapping::create([
      'user' => $this->mappedUser->id(),
      'crm_contact' => $this->contact->id(),
    ]);
    $mapping->save();

    $access = $this->contact->access('view', $this->mappedUser, TRUE);
    $this->assertTrue($access->isAllowed(), 'User with mapping and view mapped crm_contact permission can view mapped contacts.');
  }

  /**
   * Tests update access with mapping and 'edit mapped crm_contact' permission.
   */
  public function testUpdateAccessWithMappingAndPermission(): void {
    // Create a user-contact mapping.
    $mapping = UserContactMapping::create([
      'user' => $this->mappedUser->id(),
      'crm_contact' => $this->contact->id(),
    ]);
    $mapping->save();

    $access = $this->contact->access('update', $this->mappedUser, TRUE);
    $this->assertTrue($access->isAllowed(), 'User with mapping and edit mapped crm_contact permission can update mapped contacts.');
  }

  /**
   * Test view access with mapping without 'view mapped crm_contact' permission.
   */
  public function testViewAccessWithMappingWithoutPermission(): void {
    // Create a user-contact mapping for unprivileged user.
    $mapping = UserContactMapping::create([
      'user' => $this->unprivilegedUser->id(),
      'crm_contact' => $this->contact->id(),
    ]);
    $mapping->save();

    $access = $this->contact->access('view', $this->unprivilegedUser, TRUE);
    $this->assertFalse($access->isAllowed(), 'User with mapping but without view mapped crm_contact permission cannot view mapped contacts.');
  }

  /**
   * Test update access mapping without 'edit mapped crm_contact' permission.
   */
  public function testUpdateAccessWithMappingWithoutPermission(): void {
    // Create a user-contact mapping for unprivileged user.
    $mapping = UserContactMapping::create([
      'user' => $this->unprivilegedUser->id(),
      'crm_contact' => $this->contact->id(),
    ]);
    $mapping->save();

    $access = $this->contact->access('update', $this->unprivilegedUser, TRUE);
    $this->assertFalse($access->isAllowed(), 'User with mapping but without edit mapped crm_contact permission cannot update mapped contacts.');
  }

  /**
   * Test view access without mapping with 'view mapped crm_contact' permission.
   */
  public function testViewAccessWithoutMappingWithPermission(): void {
    // mappedUser has the permission but no mapping to this contact.
    $access = $this->contact->access('view', $this->mappedUser, TRUE);
    $this->assertFalse($access->isAllowed(), 'User with view mapped crm_contact permission but no mapping cannot view contacts.');
  }

  /**
   * Test update access without mapping 'edit mapped crm_contact' permission.
   */
  public function testUpdateAccessWithoutMappingWithPermission(): void {
    // mappedUser has the permission but no mapping to this contact.
    $access = $this->contact->access('update', $this->mappedUser, TRUE);
    $this->assertFalse($access->isAllowed(), 'User with edit mapped crm_contact permission but no mapping cannot update contacts.');
  }

  /**
   * Tests that standard permissions take precedence over mapped permissions.
   */
  public function testStandardPermissionsTakePrecedence(): void {
    // Create a user with both standard and mapped permissions.
    $combined_role = Role::create([
      'id' => 'combined_user',
      'label' => 'Combined User',
    ]);
    $combined_role->grantPermission('view any crm_contact');
    $combined_role->grantPermission('view mapped crm_contact');
    $combined_role->save();

    $combined_user = User::create([
      'name' => 'combined_user',
      'mail' => 'combined@example.com',
      'status' => 1,
      'roles' => ['combined_user'],
    ]);
    $combined_user->save();

    // User should have access without needing a mapping.
    $access = $this->contact->access('view', $combined_user, TRUE);
    $this->assertTrue($access->isAllowed(), 'User with standard view permission can view contacts without mapping.');
  }

  /**
   * Tests delete access is not affected by mapping.
   */
  public function testDeleteAccessNotAffectedByMapping(): void {
    // Create a user-contact mapping.
    $mapping = UserContactMapping::create([
      'user' => $this->mappedUser->id(),
      'crm_contact' => $this->contact->id(),
    ]);
    $mapping->save();

    // Delete should not be granted by mapping (only view and update are).
    $access = $this->contact->access('delete', $this->mappedUser, TRUE);
    $this->assertFalse($access->isAllowed(), 'Delete access is not granted by mapping.');
  }

}
