<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Kernel;

use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;
use Drupal\crm\Entity\Contact;
use Drupal\crm\Entity\ContactDetail;
use Drupal\user\Entity\Role;
use Drupal\user\Entity\User;

/**
 * Tests the ContactDetailAccessControlHandler.
 *
 * @group crm
 */
class ContactDetailAccessControlHandlerTest extends EntityKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'address',
    'crm',
    'inline_entity_form',
    'primary_entity_reference',
    'datetime',
    'name',
    'telephone',
  ];

  /**
   * The contact entity for testing.
   *
   * @var \Drupal\crm\CrmContactInterface
   */
  protected $contact;

  /**
   * The contact detail entity for testing.
   *
   * @var \Drupal\crm\CrmContactDetailInterface
   */
  protected $contactDetail;

  /**
   * A user with contact permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $privilegedUser;

  /**
   * A user without contact permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $unprivilegedUser;

  /**
   * A user with administer crm permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('crm_contact');
    $this->installEntitySchema('crm_contact_detail');
    $this->installEntitySchema('crm_user_contact_mapping');

    $this->installConfig(['crm', 'name', 'user']);

    // Create a contact for testing.
    $this->contact = Contact::create([
      'bundle' => 'person',
      'full_name' => ['given' => 'Test', 'family' => 'Contact'],
    ]);
    $this->contact->save();

    // Create a contact detail associated with the contact.
    $this->contactDetail = ContactDetail::create([
      'bundle' => 'email',
      'email' => 'test@example.com',
      'type' => 'main',
      'crm_contact' => $this->contact->id(),
    ]);
    $this->contactDetail->save();

    // Create a role with contact permissions.
    $privileged_role = Role::create([
      'id' => 'contact_manager',
      'label' => 'Contact Manager',
    ]);
    $privileged_role->grantPermission('view any crm_contact');
    $privileged_role->grantPermission('edit any crm_contact');
    $privileged_role->grantPermission('delete any crm_contact');
    $privileged_role->grantPermission('create crm_contact');
    $privileged_role->save();

    // Create a user with contact permissions.
    $this->privilegedUser = User::create([
      'name' => 'privileged_user',
      'mail' => 'privileged@example.com',
      'status' => 1,
      'roles' => ['contact_manager'],
    ]);
    $this->privilegedUser->save();

    // Create a user without contact permissions.
    $this->unprivilegedUser = User::create([
      'name' => 'unprivileged_user',
      'mail' => 'unprivileged@example.com',
      'status' => 1,
    ]);
    $this->unprivilegedUser->save();

    // Create a role with administer crm permission.
    $admin_role = Role::create([
      'id' => 'crm_admin',
      'label' => 'CRM Administrator',
    ]);
    $admin_role->grantPermission('administer crm');
    $admin_role->save();

    // Create a user with administer crm permission.
    $this->adminUser = User::create([
      'name' => 'admin_user',
      'mail' => 'admin@example.com',
      'status' => 1,
      'roles' => ['crm_admin'],
    ]);
    $this->adminUser->save();
  }

  /**
   * Tests view access when user has contact view permissions.
   */
  public function testViewAccessWithPermissions(): void {
    $access = $this->contactDetail->access('view', $this->privilegedUser, TRUE);
    $this->assertTrue($access->isAllowed(), 'User with contact view permissions can view contact details.');
  }

  /**
   * Tests view access when user lacks contact view permissions.
   */
  public function testViewAccessWithoutPermissions(): void {
    $access = $this->contactDetail->access('view', $this->unprivilegedUser, TRUE);
    $this->assertFalse($access->isAllowed(), 'User without contact view permissions cannot view contact details.');
  }

  /**
   * Tests update access when user has contact edit permissions.
   */
  public function testUpdateAccessWithPermissions(): void {
    $access = $this->contactDetail->access('update', $this->privilegedUser, TRUE);
    $this->assertTrue($access->isAllowed(), 'User with contact edit permissions can update contact details.');
  }

  /**
   * Tests update access when user lacks contact edit permissions.
   */
  public function testUpdateAccessWithoutPermissions(): void {
    $access = $this->contactDetail->access('update', $this->unprivilegedUser, TRUE);
    $this->assertFalse($access->isAllowed(), 'User without contact edit permissions cannot update contact details.');
  }

  /**
   * Tests delete access when user has contact delete permissions.
   */
  public function testDeleteAccessWithPermissions(): void {
    $access = $this->contactDetail->access('delete', $this->privilegedUser, TRUE);
    $this->assertTrue($access->isAllowed(), 'User with contact delete permissions can delete contact details.');
  }

  /**
   * Tests delete access when user lacks contact delete permissions.
   */
  public function testDeleteAccessWithoutPermissions(): void {
    $access = $this->contactDetail->access('delete', $this->unprivilegedUser, TRUE);
    $this->assertFalse($access->isAllowed(), 'User without contact delete permissions cannot delete contact details.');
  }

  /**
   * Tests access when contact detail has no parent contact.
   */
  public function testAccessWithoutParentContact(): void {
    // Create a contact detail without a parent contact.
    $orphaned_detail = ContactDetail::create([
      'bundle' => 'email',
      'email' => 'orphan@example.com',
      'type' => 'main',
    ]);
    $orphaned_detail->save();

    $access = $orphaned_detail->access('view', $this->unprivilegedUser, TRUE);

    $this->assertTrue($access->isForbidden(), 'Access is forbidden when contact detail has no parent contact.');
    $this->assertEquals('Contact detail must have a parent contact.', $access->getReason());
  }

  /**
   * Tests create access with contact creation permissions.
   */
  public function testCreateAccessWithPermissions(): void {
    $access_handler = $this->entityTypeManager->getAccessControlHandler('crm_contact_detail');
    $access = $access_handler->createAccess('email', $this->privilegedUser, [], TRUE);
    $this->assertTrue($access->isAllowed(), 'User with contact creation permissions can create contact details.');
  }

  /**
   * Tests create access without contact creation permissions.
   */
  public function testCreateAccessWithoutPermissions(): void {
    $access_handler = $this->entityTypeManager->getAccessControlHandler('crm_contact_detail');
    $access = $access_handler->createAccess('email', $this->unprivilegedUser, [], TRUE);
    $this->assertFalse($access->isAllowed(), 'User without contact creation permissions cannot create contact details.');
  }

  /**
   * Tests view access for user with administer crm permission.
   */
  public function testViewAccessWithAdminPermission(): void {
    $access = $this->contactDetail->access('view', $this->adminUser, TRUE);
    $this->assertTrue($access->isAllowed(), 'User with administer crm permission can view contact details.');
  }

  /**
   * Tests update access for user with administer crm permission.
   */
  public function testUpdateAccessWithAdminPermission(): void {
    $access = $this->contactDetail->access('update', $this->adminUser, TRUE);
    $this->assertTrue($access->isAllowed(), 'User with administer crm permission can update contact details.');
  }

  /**
   * Tests delete access for user with administer crm permission.
   */
  public function testDeleteAccessWithAdminPermission(): void {
    $access = $this->contactDetail->access('delete', $this->adminUser, TRUE);
    $this->assertTrue($access->isAllowed(), 'User with administer crm permission can delete contact details.');
  }

  /**
   * Tests admin access when contact detail has no parent contact.
   */
  public function testAdminAccessWithoutParentContact(): void {
    // Create a contact detail without a parent contact.
    $orphaned_detail = ContactDetail::create([
      'bundle' => 'email',
      'email' => 'orphan@example.com',
      'type' => 'main',
    ]);
    $orphaned_detail->save();

    $access = $orphaned_detail->access('view', $this->adminUser, TRUE);
    $this->assertTrue($access->isAllowed(), 'User with administer crm permission can access contact details even without parent contact.');
  }

  /**
   * Tests create access with administer crm permission.
   */
  public function testCreateAccessWithAdminPermission(): void {
    $access_handler = $this->entityTypeManager->getAccessControlHandler('crm_contact_detail');
    $access = $access_handler->createAccess('email', $this->adminUser, [], TRUE);
    $this->assertTrue($access->isAllowed(), 'User with administer crm permission can create contact details.');
  }

  /**
   * Tests access inheritance across different contact bundles.
   */
  public function testAccessWithDifferentContactBundles(): void {
    // Test with organization bundle.
    $organization = Contact::create([
      'bundle' => 'organization',
      'aliases' => ['Test Organization'],
    ]);
    $organization->save();

    $org_detail = ContactDetail::create([
      'bundle' => 'telephone',
      'telephone' => '555-0123',
      'type' => 'work',
      'crm_contact' => $organization->id(),
    ]);
    $org_detail->save();

    // Create a role with organization-specific permissions.
    $org_role = Role::create([
      'id' => 'org_manager',
      'label' => 'Organization Manager',
    ]);
    $org_role->grantPermission('view any organization crm_contact');
    $org_role->grantPermission('edit any organization crm_contact');
    $org_role->save();

    $org_user = User::create([
      'name' => 'org_user',
      'mail' => 'org@example.com',
      'status' => 1,
      'roles' => ['org_manager'],
    ]);
    $org_user->save();

    $access = $org_detail->access('view', $org_user, TRUE);
    $this->assertTrue($access->isAllowed(), 'User with organization contact permissions can view organization contact details.');

    $access = $org_detail->access('update', $org_user, TRUE);
    $this->assertTrue($access->isAllowed(), 'User with organization contact permissions can update organization contact details.');
  }

  /**
   * Tests that access properly delegates to contact access control.
   */
  public function testAccessDelegationToContact(): void {
    // Mock the contact to verify delegation happens.
    $access = $this->contactDetail->access('view', $this->privilegedUser, TRUE);
    $contact_access = $this->contact->access('view', $this->privilegedUser, TRUE);

    $this->assertEquals($access->isAllowed(), $contact_access->isAllowed(), 'Contact detail access matches parent contact access.');
  }

}
