<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Kernel;

use Drupal\crm\ContactDetailListBuilder;
use Drupal\crm\Entity\Contact;
use Drupal\crm\Entity\ContactDetail;
use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;

/**
 * Tests the ContactDetailListBuilder.
 *
 * @group crm
 * @coversDefaultClass \Drupal\crm\ContactDetailListBuilder
 */
class ContactDetailListBuilderTest extends EntityKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'address',
    'crm',
    'inline_entity_form',
    'primary_entity_reference',
    'datetime',
    'name',
    'telephone',
  ];

  /**
   * The list builder.
   *
   * @var \Drupal\crm\ContactDetailListBuilder
   */
  protected $listBuilder;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('crm_contact');
    $this->installEntitySchema('crm_contact_detail');
    $this->installEntitySchema('crm_user_contact_mapping');

    $this->installConfig(['crm', 'name', 'system']);

    $this->listBuilder = $this->entityTypeManager->getListBuilder('crm_contact_detail');
  }

  /**
   * Tests that the list builder is the correct class.
   *
   * @covers ::createInstance
   */
  public function testListBuilderClass(): void {
    $this->assertInstanceOf(ContactDetailListBuilder::class, $this->listBuilder);
  }

  /**
   * Tests that createInstance properly injects dependencies.
   *
   * @covers ::createInstance
   */
  public function testCreateInstance(): void {
    $entityType = $this->entityTypeManager->getDefinition('crm_contact_detail');
    $listBuilder = ContactDetailListBuilder::createInstance(
      \Drupal::getContainer(),
      $entityType
    );

    $this->assertInstanceOf(ContactDetailListBuilder::class, $listBuilder);
  }

  /**
   * Tests render with no details.
   *
   * @covers ::render
   */
  public function testRenderWithNoDetails(): void {
    $build = $this->listBuilder->render();

    // Verify summary is present with zero count.
    $this->assertArrayHasKey('summary', $build);
    $this->assertArrayHasKey('#markup', $build['summary']);
    $this->assertStringContainsString('Total details: 0', (string) $build['summary']['#markup']);

    // Verify table is present with empty message.
    $this->assertArrayHasKey('table', $build);
    $this->assertArrayHasKey('#empty', $build['table']);
    $this->assertStringContainsString('No crm detail types available', (string) $build['table']['#empty']);
  }

  /**
   * Tests render with details.
   *
   * @covers ::render
   */
  public function testRenderWithDetails(): void {
    // Create test data.
    $this->createEmailDetail('test@example.com');

    $build = $this->listBuilder->render();

    // Verify summary shows correct count.
    $this->assertArrayHasKey('summary', $build);
    $this->assertStringContainsString('Total details: 1', (string) $build['summary']['#markup']);

    // Verify table is present.
    $this->assertArrayHasKey('table', $build);
  }

  /**
   * Tests render with multiple details.
   *
   * @covers ::render
   */
  public function testRenderWithMultipleDetails(): void {
    // Create multiple test details.
    for ($i = 0; $i < 3; $i++) {
      $this->createEmailDetail('test' . $i . '@example.com');
    }

    $build = $this->listBuilder->render();

    // Verify summary shows correct count.
    $this->assertStringContainsString('Total details: 3', (string) $build['summary']['#markup']);
  }

  /**
   * Tests buildHeader returns expected columns.
   *
   * @covers ::buildHeader
   */
  public function testBuildHeader(): void {
    $header = $this->listBuilder->buildHeader();

    $this->assertArrayHasKey('id', $header);
    $this->assertArrayHasKey('label', $header);
    $this->assertArrayHasKey('type', $header);
    $this->assertArrayHasKey('reference', $header);
    $this->assertArrayHasKey('status', $header);
    $this->assertArrayHasKey('created', $header);
    $this->assertArrayHasKey('changed', $header);
    $this->assertArrayHasKey('operations', $header);

    $this->assertEquals('ID', (string) $header['id']);
    $this->assertEquals('Label', (string) $header['label']);
    $this->assertEquals('Type', (string) $header['type']);
    $this->assertEquals('Reference', (string) $header['reference']);
    $this->assertEquals('Status', (string) $header['status']);
    $this->assertEquals('Created', (string) $header['created']);
    $this->assertEquals('Updated', (string) $header['changed']);
  }

  /**
   * Tests buildRow with a detail entity.
   *
   * @covers ::buildRow
   */
  public function testBuildRow(): void {
    $detail = $this->createEmailDetail('test@example.com');

    $row = $this->listBuilder->buildRow($detail);

    $this->assertEquals($detail->id(), $row['id']);
    $this->assertArrayHasKey('label', $row);
    $this->assertEquals('Email address', $row['type']);
    $this->assertEquals('Enabled', (string) $row['status']);
    $this->assertArrayHasKey('created', $row);
    $this->assertArrayHasKey('changed', $row);
  }

  /**
   * Tests buildRow with disabled detail.
   *
   * @covers ::buildRow
   */
  public function testBuildRowWithDisabledDetail(): void {
    $detail = ContactDetail::create([
      'bundle' => 'email',
      'status' => 0,
      'email' => 'disabled@example.com',
    ]);
    $detail->save();

    $row = $this->listBuilder->buildRow($detail);

    $this->assertEquals($detail->id(), $row['id']);
    $this->assertEquals('Disabled', (string) $row['status']);
  }

  /**
   * Tests getReferenceCount for email details.
   *
   * @covers ::getReferenceCount
   */
  public function testGetReferenceCountForEmail(): void {
    $detail = $this->createEmailDetail('test@example.com');

    // Create a contact that references the email.
    $contact = Contact::create([
      'bundle' => 'person',
      'full_name' => ['given' => 'Test'],
      'emails' => [$detail->id()],
    ]);
    $contact->save();

    $row = $this->listBuilder->buildRow($detail);

    $this->assertEquals(1, $row['references']);
  }

  /**
   * Tests getReferenceCount for telephone details.
   *
   * @covers ::getReferenceCount
   */
  public function testGetReferenceCountForTelephone(): void {
    $detail = ContactDetail::create([
      'bundle' => 'telephone',
      'status' => 1,
      'telephone' => '555-1234',
    ]);
    $detail->save();

    // Create a contact that references the telephone.
    $contact = Contact::create([
      'bundle' => 'person',
      'full_name' => ['given' => 'Test'],
      'telephones' => [$detail->id()],
    ]);
    $contact->save();

    $row = $this->listBuilder->buildRow($detail);

    $this->assertEquals(1, $row['references']);
  }

  /**
   * Tests getReferenceCount for address details.
   *
   * @covers ::getReferenceCount
   */
  public function testGetReferenceCountForAddress(): void {
    $detail = ContactDetail::create([
      'bundle' => 'address',
      'status' => 1,
      'address' => [
        'country_code' => 'US',
        'address_line1' => '123 Test St',
        'locality' => 'Test City',
        'administrative_area' => 'CA',
        'postal_code' => '12345',
      ],
    ]);
    $detail->save();

    // Create a contact that references the address.
    $contact = Contact::create([
      'bundle' => 'person',
      'full_name' => ['given' => 'Test'],
      'addresses' => [$detail->id()],
    ]);
    $contact->save();

    $row = $this->listBuilder->buildRow($detail);

    $this->assertEquals(1, $row['references']);
  }

  /**
   * Tests getReferenceCount with multiple references.
   *
   * @covers ::getReferenceCount
   */
  public function testGetReferenceCountWithMultipleReferences(): void {
    // Create email detail first.
    $detail = $this->createEmailDetail('shared@example.com');

    // Create multiple contacts that reference the same email.
    for ($i = 0; $i < 3; $i++) {
      $contact = Contact::create([
        'bundle' => 'person',
        'full_name' => ['given' => 'Person ' . $i],
        'emails' => [$detail->id()],
      ]);
      $contact->save();
    }

    // Reload the detail to ensure fresh state.
    $detail = ContactDetail::load($detail->id());
    $row = $this->listBuilder->buildRow($detail);

    $this->assertEquals(3, $row['references']);
  }

  /**
   * Tests getDefaultOperations adjusts edit weight.
   *
   * @covers ::getDefaultOperations
   */
  public function testGetDefaultOperationsAdjustsEditWeight(): void {
    $detail = $this->createEmailDetail('test@example.com');

    // Use reflection to access the protected method.
    $reflectionMethod = new \ReflectionMethod(ContactDetailListBuilder::class, 'getDefaultOperations');

    $operations = $reflectionMethod->invoke($this->listBuilder, $detail);

    // Verify edit operation has weight 30 if present.
    if (isset($operations['edit'])) {
      $this->assertEquals(30, $operations['edit']['weight']);
    }
  }

  /**
   * Creates a test email detail entity.
   *
   * @param string $email
   *   The email address.
   *
   * @return \Drupal\crm\Entity\ContactDetail
   *   The created contact detail entity.
   */
  protected function createEmailDetail(string $email = 'test@example.com'): ContactDetail {
    $detail = ContactDetail::create([
      'bundle' => 'email',
      'status' => 1,
      'email' => $email,
    ]);
    $detail->save();
    return $detail;
  }

}
