<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Kernel;

use Drupal\crm\ContactListBuilder;
use Drupal\crm\Entity\Contact;
use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;

/**
 * Tests the ContactListBuilder.
 *
 * @group crm
 * @coversDefaultClass \Drupal\crm\ContactListBuilder
 */
class ContactListBuilderTest extends EntityKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'address',
    'crm',
    'inline_entity_form',
    'primary_entity_reference',
    'datetime',
    'name',
    'telephone',
  ];

  /**
   * The list builder.
   *
   * @var \Drupal\crm\ContactListBuilder
   */
  protected $listBuilder;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('crm_contact');
    $this->installEntitySchema('crm_contact_detail');
    $this->installEntitySchema('crm_user_contact_mapping');

    $this->installConfig(['crm', 'name', 'system']);

    $this->listBuilder = $this->entityTypeManager->getListBuilder('crm_contact');
  }

  /**
   * Tests that the list builder is the correct class.
   *
   * @covers ::createInstance
   */
  public function testListBuilderClass(): void {
    $this->assertInstanceOf(ContactListBuilder::class, $this->listBuilder);
  }

  /**
   * Tests that createInstance properly injects dependencies.
   *
   * @covers ::createInstance
   */
  public function testCreateInstance(): void {
    $entityType = $this->entityTypeManager->getDefinition('crm_contact');
    $listBuilder = ContactListBuilder::createInstance(
      \Drupal::getContainer(),
      $entityType
    );

    $this->assertInstanceOf(ContactListBuilder::class, $listBuilder);
  }

  /**
   * Tests render with no contacts.
   *
   * @covers ::render
   */
  public function testRenderWithNoContacts(): void {
    $build = $this->listBuilder->render();

    // Verify summary is present.
    $this->assertArrayHasKey('summary', $build);
    $this->assertArrayHasKey('#markup', $build['summary']);
    $this->assertStringContainsString('Total contacts: 0', (string) $build['summary']['#markup']);

    // Verify table is present.
    $this->assertArrayHasKey('table', $build);
  }

  /**
   * Tests render with contacts.
   *
   * @covers ::render
   */
  public function testRenderWithContacts(): void {
    // Create test data.
    $this->createContact('Test Person 1');

    $build = $this->listBuilder->render();

    // Verify summary shows correct count.
    $this->assertArrayHasKey('summary', $build);
    $this->assertStringContainsString('Total contacts: 1', (string) $build['summary']['#markup']);

    // Verify table is present.
    $this->assertArrayHasKey('table', $build);
  }

  /**
   * Tests render with multiple contacts.
   *
   * @covers ::render
   */
  public function testRenderWithMultipleContacts(): void {
    // Create multiple test contacts.
    for ($i = 0; $i < 5; $i++) {
      $this->createContact('Person ' . $i);
    }

    $build = $this->listBuilder->render();

    // Verify summary shows correct count.
    $this->assertStringContainsString('Total contacts: 5', (string) $build['summary']['#markup']);
  }

  /**
   * Tests buildHeader returns expected columns.
   *
   * @covers ::buildHeader
   */
  public function testBuildHeader(): void {
    $header = $this->listBuilder->buildHeader();

    $this->assertArrayHasKey('id', $header);
    $this->assertArrayHasKey('name', $header);
    $this->assertArrayHasKey('type', $header);
    $this->assertArrayHasKey('status', $header);
    $this->assertArrayHasKey('created', $header);
    $this->assertArrayHasKey('changed', $header);
    $this->assertArrayHasKey('operations', $header);

    $this->assertEquals('ID', (string) $header['id']);
    $this->assertEquals('Name', (string) $header['name']);
    $this->assertEquals('Type', (string) $header['type']);
    $this->assertEquals('Status', (string) $header['status']);
    $this->assertEquals('Created', (string) $header['created']);
    $this->assertEquals('Updated', (string) $header['changed']);
  }

  /**
   * Tests buildRow with a contact entity.
   *
   * @covers ::buildRow
   */
  public function testBuildRow(): void {
    $contact = $this->createContact('John Doe');

    $row = $this->listBuilder->buildRow($contact);

    $this->assertEquals($contact->id(), $row['id']);
    $this->assertArrayHasKey('name', $row);
    $this->assertEquals('Person', $row['type']);
    $this->assertEquals('Active', (string) $row['status']);
    $this->assertArrayHasKey('created', $row);
    $this->assertArrayHasKey('changed', $row);
    $this->assertArrayHasKey('operations', $row);
  }

  /**
   * Tests buildRow with inactive contact.
   *
   * @covers ::buildRow
   */
  public function testBuildRowWithInactiveContact(): void {
    $contact = Contact::create([
      'bundle' => 'person',
      'full_name' => ['given' => 'Inactive Person'],
      'status' => 0,
    ]);
    $contact->save();

    $row = $this->listBuilder->buildRow($contact);

    $this->assertEquals($contact->id(), $row['id']);
    $this->assertEquals('Inactive', (string) $row['status']);
  }

  /**
   * Creates a test contact entity.
   *
   * @param string $name
   *   The name for the contact.
   *
   * @return \Drupal\crm\Entity\Contact
   *   The created contact entity.
   */
  protected function createContact(string $name = 'Test Person'): Contact {
    $contact = Contact::create([
      'bundle' => 'person',
      'full_name' => ['given' => $name],
    ]);
    $contact->save();
    return $contact;
  }

}
