<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Kernel;

use Drupal\crm\ContactTypeListBuilder;
use Drupal\crm\Entity\ContactType;
use Drupal\KernelTests\KernelTestBase;

/**
 * Tests the ContactTypeListBuilder.
 *
 * @group crm
 * @coversDefaultClass \Drupal\crm\ContactTypeListBuilder
 */
class ContactTypeListBuilderTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'address',
    'crm',
    'inline_entity_form',
    'primary_entity_reference',
    'datetime',
    'name',
    'telephone',
  ];

  /**
   * The list builder.
   *
   * @var \Drupal\crm\ContactTypeListBuilder
   */
  protected $listBuilder;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('crm_contact_type');
    $this->installEntitySchema('crm_contact');

    $this->entityTypeManager = $this->container->get('entity_type.manager');
    $this->listBuilder = $this->entityTypeManager->getListBuilder('crm_contact_type');
  }

  /**
   * Tests that the list builder is the correct class.
   */
  public function testListBuilderClass(): void {
    $this->assertInstanceOf(ContactTypeListBuilder::class, $this->listBuilder);
  }

  /**
   * Tests render with no contact types.
   *
   * @covers ::render
   */
  public function testRenderWithNoContactTypes(): void {
    $build = $this->listBuilder->render();

    // Verify table is present with empty message.
    $this->assertArrayHasKey('table', $build);
    $this->assertArrayHasKey('#empty', $build['table']);
    $this->assertStringContainsString('No contact types available', (string) $build['table']['#empty']);
  }

  /**
   * Tests render with contact types.
   *
   * @covers ::render
   */
  public function testRenderWithContactTypes(): void {
    // Create test contact type.
    $this->createContactType('test_person', 'Test Person', 'A test person type.');

    $build = $this->listBuilder->render();

    // Verify table is present.
    $this->assertArrayHasKey('table', $build);
    $this->assertArrayHasKey('#rows', $build['table']);
  }

  /**
   * Tests buildHeader returns expected columns.
   *
   * @covers ::buildHeader
   */
  public function testBuildHeader(): void {
    $header = $this->listBuilder->buildHeader();

    $this->assertArrayHasKey('label', $header);
    $this->assertArrayHasKey('description', $header);
    $this->assertArrayHasKey('operations', $header);

    $this->assertEquals('Label', (string) $header['label']);
    $this->assertEquals('Description', (string) $header['description']['data']);
  }

  /**
   * Tests buildRow with a contact type entity.
   *
   * @covers ::buildRow
   */
  public function testBuildRow(): void {
    $contactType = $this->createContactType('person', 'Person', 'A person contact type.');

    $row = $this->listBuilder->buildRow($contactType);

    $this->assertArrayHasKey('label', $row);
    $this->assertEquals('Person', $row['label']['data']);
    $this->assertArrayHasKey('description', $row);
    $this->assertEquals('A person contact type.', $row['description']['data']['#markup']);
    $this->assertArrayHasKey('operations', $row);
  }

  /**
   * Tests buildRow with different contact type.
   *
   * @covers ::buildRow
   */
  public function testBuildRowWithDifferentData(): void {
    $contactType = $this->createContactType('organization', 'Organization', 'An organization type.');

    $row = $this->listBuilder->buildRow($contactType);

    $this->assertArrayHasKey('label', $row);
    $this->assertEquals('Organization', $row['label']['data']);
    $this->assertArrayHasKey('description', $row);
    $this->assertEquals('An organization type.', $row['description']['data']['#markup']);
  }

  /**
   * Tests buildRow with empty description.
   *
   * @covers ::buildRow
   */
  public function testBuildRowWithEmptyDescription(): void {
    $contactType = $this->createContactType('household', 'Household', '');

    $row = $this->listBuilder->buildRow($contactType);

    $this->assertArrayHasKey('label', $row);
    $this->assertEquals('Household', $row['label']['data']);
    $this->assertArrayHasKey('description', $row);
    $this->assertEquals('', $row['description']['data']['#markup']);
  }

  /**
   * Tests getDefaultOperations adjusts edit weight.
   *
   * @covers ::getDefaultOperations
   */
  public function testGetDefaultOperationsAdjustsEditWeight(): void {
    $contactType = $this->createContactType('test_type', 'Test Type', 'A test type.');

    // Use reflection to access the protected method.
    $reflectionMethod = new \ReflectionMethod(ContactTypeListBuilder::class, 'getDefaultOperations');

    $operations = $reflectionMethod->invoke($this->listBuilder, $contactType);

    // Verify edit operation has weight 30 if present.
    if (isset($operations['edit'])) {
      $this->assertEquals(30, $operations['edit']['weight']);
    }
  }

  /**
   * Tests render with multiple contact types.
   *
   * @covers ::render
   */
  public function testRenderWithMultipleContactTypes(): void {
    // Create multiple contact types.
    $this->createContactType('person', 'Person', 'A person.');
    $this->createContactType('organization', 'Organization', 'An organization.');
    $this->createContactType('household', 'Household', 'A household.');

    $build = $this->listBuilder->render();

    // Verify table has rows.
    $this->assertArrayHasKey('table', $build);
    $this->assertArrayHasKey('#rows', $build['table']);
    $this->assertCount(3, $build['table']['#rows']);
  }

  /**
   * Gets the default date configuration for contact types.
   *
   * @return array
   *   The date configuration array.
   */
  protected function getDefaultDateConfig(): array {
    return [
      'start_date' => [
        'label' => '',
        'description' => '',
      ],
      'end_date' => [
        'label' => '',
        'description' => '',
      ],
    ];
  }

  /**
   * Creates a test contact type entity.
   *
   * @param string $id
   *   The machine name.
   * @param string $label
   *   The label.
   * @param string $description
   *   The description.
   *
   * @return \Drupal\crm\Entity\ContactType
   *   The created contact type entity.
   */
  protected function createContactType(string $id, string $label, string $description): ContactType {
    $contactType = ContactType::create([
      'id' => $id,
      'label' => $label,
      'description' => $description,
      'date' => $this->getDefaultDateConfig(),
    ]);
    $contactType->save();
    return $contactType;
  }

}
