<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Kernel\Controller;

use Drupal\crm\Controller\CommentController;
use Drupal\crm\Entity\Contact;
use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;

/**
 * Kernel tests for the CommentController.
 *
 * @group crm
 * @covers \Drupal\crm\Controller\CommentController
 */
class CommentControllerTest extends EntityKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'crm',
    'datetime',
    'inline_entity_form',
    'primary_entity_reference',
    'name',
    'telephone',
    'address',
    'comment',
  ];

  /**
   * The controller under test.
   *
   * @var \Drupal\crm\Controller\CommentController
   */
  protected $controller;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('crm_contact');
    $this->installEntitySchema('crm_contact_detail');
    $this->installEntitySchema('crm_relationship');
    $this->installEntitySchema('crm_user_contact_mapping');
    $this->installEntitySchema('comment');
    $this->installConfig(['system', 'crm']);

    $this->controller = CommentController::create($this->container);
  }

  /**
   * Creates a contact for testing.
   *
   * @param string $bundle
   *   The contact bundle.
   * @param string $name
   *   The contact name.
   *
   * @return \Drupal\crm\Entity\Contact
   *   The created contact.
   */
  protected function createContact(string $bundle, string $name): Contact {
    $contact = Contact::create([
      'bundle' => $bundle,
      'name' => $name,
      'status' => TRUE,
    ]);
    $contact->save();
    return $contact;
  }

  /**
   * Tests commentsPage builds correct render array structure.
   */
  public function testCommentsPageReturnsRenderArray(): void {
    $contact = $this->createContact('person', 'Test Person');

    $build = $this->controller->commentsPage($contact);

    $this->assertIsArray($build);
    $this->assertArrayHasKey('#cache', $build);
    $this->assertArrayHasKey('tags', $build['#cache']);
    $this->assertContains('crm_contact:' . $contact->id(), $build['#cache']['tags']);
  }

  /**
   * Tests commentsPage includes title with contact label.
   */
  public function testCommentsPageIncludesTitle(): void {
    $contact = $this->createContact('person', 'Jane Smith');

    $build = $this->controller->commentsPage($contact);

    $this->assertArrayHasKey('#title', $build);
    $this->assertStringContainsString('Comments about', (string) $build['#title']);
    $this->assertStringContainsString('Jane Smith', (string) $build['#title']);
  }

  /**
   * Tests commentsPage does not include comment section when field is missing.
   */
  public function testCommentsPageWithoutCommentField(): void {
    $contact = $this->createContact('person', 'No Comments Person');

    // The default person contact type may not have a comment field.
    // Test that commentsPage handles this gracefully.
    $build = $this->controller->commentsPage($contact);

    $this->assertIsArray($build);
    // If there's no comment field, the 'comment' key shouldn't be set.
    if (!$contact->hasField('comment')) {
      $this->assertArrayNotHasKey('comment', $build);
    }
  }

  /**
   * Tests controller creation via dependency injection.
   */
  public function testControllerCreation(): void {
    $controller = CommentController::create($this->container);

    $this->assertInstanceOf(CommentController::class, $controller);
  }

  /**
   * Tests title method returns translatable markup.
   */
  public function testTitleReturnsTranslatableMarkup(): void {
    $contact = $this->createContact('person', 'Title Test Person');

    $title = $this->controller->title($contact);

    $this->assertStringContainsString('Comments for', (string) $title);
    $this->assertStringContainsString('Title Test Person', (string) $title);
  }

}
