<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Kernel\Controller;

use Drupal\crm\Controller\RelationshipController;
use Drupal\crm\Entity\Contact;
use Drupal\crm\Entity\Relationship;
use Drupal\crm\Entity\RelationshipType;
use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;

/**
 * Kernel tests for the RelationshipController.
 *
 * @group crm
 * @covers \Drupal\crm\Controller\RelationshipController
 */
class RelationshipControllerTest extends EntityKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'crm',
    'datetime',
    'inline_entity_form',
    'primary_entity_reference',
    'name',
    'telephone',
    'address',
  ];

  /**
   * The controller under test.
   *
   * @var \Drupal\crm\Controller\RelationshipController
   */
  protected $controller;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('crm_contact');
    $this->installEntitySchema('crm_contact_detail');
    $this->installEntitySchema('crm_relationship');
    $this->installEntitySchema('crm_user_contact_mapping');
    $this->installConfig(['system', 'crm']);

    $this->entityTypeManager = $this->container->get('entity_type.manager');
    $this->controller = RelationshipController::create($this->container);
  }

  /**
   * Creates a contact for testing.
   *
   * @param string $bundle
   *   The contact bundle.
   * @param string $name
   *   The contact name.
   *
   * @return \Drupal\crm\Entity\Contact
   *   The created contact.
   */
  protected function createContact(string $bundle, string $name): Contact {
    $contact = Contact::create([
      'bundle' => $bundle,
      'name' => $name,
      'status' => TRUE,
    ]);
    $contact->save();
    return $contact;
  }

  /**
   * Creates a relationship type for testing.
   *
   * @param string $id
   *   The relationship type ID.
   * @param array $values
   *   Additional values.
   *
   * @return \Drupal\crm\Entity\RelationshipType
   *   The created relationship type.
   */
  protected function createRelationshipType(string $id, array $values = []): RelationshipType {
    $defaults = [
      'id' => $id,
      'label' => ucfirst(str_replace('_', ' ', $id)),
      'asymmetric' => TRUE,
      'label_a' => 'Contact A',
      'label_b' => 'Contact B',
      'contact_type_a' => ['person'],
      'contact_type_b' => ['organization'],
    ];
    $relationship_type = RelationshipType::create($values + $defaults);
    $relationship_type->save();
    return $relationship_type;
  }

  /**
   * Tests controller creation via dependency injection.
   */
  public function testControllerCreation(): void {
    $controller = RelationshipController::create($this->container);

    $this->assertInstanceOf(RelationshipController::class, $controller);
  }

  /**
   * Tests build method returns correct structure with empty relationships.
   */
  public function testBuildReturnsCorrectStructureWithNoRelationships(): void {
    $contact = $this->createContact('person', 'Test Person');

    $build = $this->controller->build($contact);

    $this->assertIsArray($build);
    $this->assertArrayHasKey('active', $build);
    $this->assertArrayHasKey('inactive', $build);

    // Check table structure.
    $this->assertEquals('table', $build['active']['#type']);
    $this->assertEquals('table', $build['inactive']['#type']);

    // Check empty messages.
    $this->assertStringContainsString('no active relationships', (string) $build['active']['#empty']);
    $this->assertStringContainsString('no inactive relationships', (string) $build['inactive']['#empty']);

    // Check headers exist.
    $this->assertArrayHasKey('#header', $build['active']);
    $this->assertArrayHasKey('#header', $build['inactive']);
  }

  /**
   * Tests build method shows active relationships.
   */
  public function testBuildShowsActiveRelationships(): void {
    $contact_a = $this->createContact('person', 'Person A');
    $contact_b = $this->createContact('organization', 'Organization B');

    $relationship_type = $this->createRelationshipType('test_active');

    $relationship = Relationship::create([
      'bundle' => 'test_active',
      'contacts' => [$contact_a->id(), $contact_b->id()],
      'status' => TRUE,
      'start_date' => strtotime('2023-01-01'),
    ]);
    $relationship->save();

    $build = $this->controller->build($contact_a);

    $this->assertIsArray($build);
    $this->assertArrayHasKey('active', $build);
    $this->assertArrayHasKey('#rows', $build['active']);
    $this->assertNotEmpty($build['active']['#rows']);

    // Verify the relationship is in the active table.
    $this->assertCount(1, $build['active']['#rows']);

    // Verify inactive table is empty.
    $this->assertEmpty($build['inactive']['#rows']);
  }

  /**
   * Tests build method shows inactive relationships.
   */
  public function testBuildShowsInactiveRelationships(): void {
    $contact_a = $this->createContact('person', 'Person Inactive');
    $contact_b = $this->createContact('organization', 'Organization Inactive');

    $relationship_type = $this->createRelationshipType('test_inactive');

    $relationship = Relationship::create([
      'bundle' => 'test_inactive',
      'contacts' => [$contact_a->id(), $contact_b->id()],
      'status' => FALSE,
      'start_date' => strtotime('2022-01-01'),
      'end_date' => strtotime('2022-12-31'),
    ]);
    $relationship->save();

    $build = $this->controller->build($contact_a);

    $this->assertIsArray($build);
    $this->assertArrayHasKey('inactive', $build);
    $this->assertArrayHasKey('#rows', $build['inactive']);
    $this->assertNotEmpty($build['inactive']['#rows']);

    // Verify the relationship is in the inactive table.
    $this->assertCount(1, $build['inactive']['#rows']);

    // Verify active table is empty.
    $this->assertEmpty($build['active']['#rows']);
  }

  /**
   * Tests build method shows both active and inactive relationships.
   */
  public function testBuildShowsBothActiveAndInactive(): void {
    $contact_a = $this->createContact('person', 'Person Both');
    $contact_b = $this->createContact('organization', 'Organization Both');
    $contact_c = $this->createContact('organization', 'Organization Both 2');

    $relationship_type = $this->createRelationshipType('test_both');

    // Create active relationship.
    $active_relationship = Relationship::create([
      'bundle' => 'test_both',
      'contacts' => [$contact_a->id(), $contact_b->id()],
      'status' => TRUE,
      'start_date' => strtotime('2023-01-01'),
    ]);
    $active_relationship->save();

    // Create inactive relationship.
    $inactive_relationship = Relationship::create([
      'bundle' => 'test_both',
      'contacts' => [$contact_a->id(), $contact_c->id()],
      'status' => FALSE,
      'start_date' => strtotime('2022-01-01'),
      'end_date' => strtotime('2022-12-31'),
    ]);
    $inactive_relationship->save();

    $build = $this->controller->build($contact_a);

    $this->assertCount(1, $build['active']['#rows']);
    $this->assertCount(1, $build['inactive']['#rows']);
  }

  /**
   * Tests build method includes contextual links.
   */
  public function testBuildIncludesContextualLinks(): void {
    $contact = $this->createContact('person', 'Contextual Test');

    $build = $this->controller->build($contact);

    $this->assertArrayHasKey('#contextual_links', $build);
    $this->assertArrayHasKey('crm_contact_relationship', $build['#contextual_links']);
    $this->assertEquals(
      ['crm_contact' => $contact->id()],
      $build['#contextual_links']['crm_contact_relationship']['route_parameters']
    );
  }

  /**
   * Tests build method disables cache.
   */
  public function testBuildDisablesCache(): void {
    $contact = $this->createContact('person', 'Cache Test');

    $build = $this->controller->build($contact);

    $this->assertArrayHasKey('#cache', $build);
    $this->assertEquals(0, $build['#cache']['max-age']);
  }

  /**
   * Tests addPage returns empty message when no eligible types.
   */
  public function testAddPageReturnsEmptyMessageWhenNoEligibleTypes(): void {
    // Create a household contact that has no eligible relationship types
    // (assuming default relationship types don't include household).
    $contact = $this->createContact('household', 'Test Household');

    // Create a relationship type that doesn't include household.
    $this->createRelationshipType('test_no_household', [
      'contact_type_a' => ['person'],
      'contact_type_b' => ['organization'],
    ]);

    $build = $this->controller->addPage($contact);

    $this->assertIsArray($build);
    // When no eligible types, it should either return a markup message
    // or an empty bundles array.
    if (isset($build['#markup'])) {
      $this->assertStringContainsString('no relationship types available', (string) $build['#markup']);
    }
    else {
      $this->assertArrayHasKey('#theme', $build);
    }
  }

  /**
   * Tests addPage returns eligible types for a person contact.
   */
  public function testAddPageReturnsEligibleTypesForPerson(): void {
    $person = $this->createContact('person', 'Test Person for Add');

    // The default 'employee' relationship type should be eligible for person.
    $build = $this->controller->addPage($person);

    $this->assertIsArray($build);
    $this->assertArrayHasKey('#theme', $build);
    $this->assertEquals('entity_add_list', $build['#theme']);
    $this->assertArrayHasKey('#bundles', $build);
    $this->assertArrayHasKey('#cache', $build);
  }

  /**
   * Tests addPage includes cache tags.
   */
  public function testAddPageIncludesCacheTags(): void {
    $person = $this->createContact('person', 'Cache Tag Test');

    $build = $this->controller->addPage($person);

    // Only check cache tags if we have an entity_add_list theme.
    if (isset($build['#theme']) && $build['#theme'] === 'entity_add_list') {
      $this->assertArrayHasKey('#cache', $build);
      $this->assertArrayHasKey('tags', $build['#cache']);
      $this->assertContains('config:crm_relationship_type_list', $build['#cache']['tags']);
    }
  }

  /**
   * Tests build method shows correct related contact label.
   */
  public function testBuildShowsCorrectRelatedContactLabel(): void {
    $contact_a = $this->createContact('person', 'Alice Person');
    $contact_b = $this->createContact('organization', 'Acme Corp');

    $relationship_type = $this->createRelationshipType('test_labels', [
      'label_a' => 'Works for',
      'label_b' => 'Employs',
    ]);

    $relationship = Relationship::create([
      'bundle' => 'test_labels',
      'contacts' => [$contact_a->id(), $contact_b->id()],
      'status' => TRUE,
      'start_date' => strtotime('2023-01-01'),
    ]);
    $relationship->save();

    // When viewing from contact_a's perspective, we should see contact_b
    // and the relationship label should be from position A's perspective.
    $build = $this->controller->build($contact_a);

    $this->assertNotEmpty($build['active']['#rows']);
    $row = $build['active']['#rows'][0];

    // The contact column should show the other contact (Acme Corp).
    $this->assertStringContainsString('Acme Corp', (string) $row['contact']);
  }

  /**
   * Tests row includes operations links.
   */
  public function testRowIncludesOperationsLinks(): void {
    $contact_a = $this->createContact('person', 'Operations Test');
    $contact_b = $this->createContact('organization', 'Operations Org');

    $relationship_type = $this->createRelationshipType('test_operations');

    $relationship = Relationship::create([
      'bundle' => 'test_operations',
      'contacts' => [$contact_a->id(), $contact_b->id()],
      'status' => TRUE,
      'start_date' => strtotime('2023-01-01'),
    ]);
    $relationship->save();

    $build = $this->controller->build($contact_a);

    $this->assertNotEmpty($build['active']['#rows']);
    $row = $build['active']['#rows'][0];

    $this->assertArrayHasKey('operations', $row);
    $this->assertArrayHasKey('data', $row['operations']);
    $this->assertEquals('operations', $row['operations']['data']['#type']);
    $this->assertArrayHasKey('edit', $row['operations']['data']['#links']);
    $this->assertArrayHasKey('delete', $row['operations']['data']['#links']);
  }

}
