<?php

namespace Drupal\Tests\crm\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\crm\Entity\DetailType;
use Drupal\crm\Entity\ContactDetailType;

/**
 * Tests the DetailTypeListBuilder.
 *
 * @group crm
 */
class DetailTypeListBuilderTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'address',
    'crm',
    'inline_entity_form',
    'primary_entity_reference',
    'datetime',
    'name',
    'telephone',
  ];

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();

    // Install necessary config/install schemas if needed.
    $this->installEntitySchema('crm_detail_type');
    $this->installEntitySchema('crm_contact_detail_type');
  }

  /**
   * Test with bundles.
   */
  public function testBuildRowWithBundles() {
    $bundle = ContactDetailType::create([
      'id' => 'email',
      'label' => 'Email',
    ]);
    $bundle->save();

    $detail = DetailType::create([
      'id' => 'work',
      'label' => 'Work',
      'status' => TRUE,
      'bundles' => ['email'],
      'negate' => FALSE,
    ]);
    $detail->save();

    $list_builder = \Drupal::entityTypeManager()
      ->getListBuilder('crm_detail_type');

    $row = $list_builder->buildRow($detail);

    $this->assertEquals('Work', $row['label']);
    $this->assertEquals('work', $row['id']);
    $this->assertEquals('Enabled', $row['status']);
    $this->assertStringContainsString('Only: Email', (string) $row['bundles']);
  }

  /**
   * Test without bundles.
   */
  public function testBuildRowWithOutBundles() {

    $detail = DetailType::create([
      'id' => 'work',
      'label' => 'Work',
      'status' => TRUE,
      'bundles' => [],
      'negate' => FALSE,
    ]);
    $detail->save();

    $list_builder = \Drupal::entityTypeManager()
      ->getListBuilder('crm_detail_type');

    $row = $list_builder->buildRow($detail);

    $this->assertEquals('Work', $row['label']);
    $this->assertEquals('work', $row['id']);
    $this->assertEquals('Enabled', $row['status']);
    $this->assertStringContainsString('Any', (string) $row['bundles']);
  }

  /**
   * Test negated bundles.
   */
  public function testBuildRowWithNegateBundles() {
    $bundle = ContactDetailType::create([
      'id' => 'email',
      'label' => 'Email',
    ]);
    $bundle->save();

    $detail = DetailType::create([
      'id' => 'other',
      'label' => 'Other',
      'status' => TRUE,
      'bundles' => ['email'],
      'negate' => TRUE,
    ]);
    $detail->save();

    $list_builder = \Drupal::entityTypeManager()
      ->getListBuilder('crm_detail_type');

    $row = $list_builder->buildRow($detail);

    $this->assertEquals('Other', $row['label']);
    $this->assertEquals('other', $row['id']);
    $this->assertEquals('Enabled', $row['status']);
    $this->assertStringContainsString('Except: Email', (string) $row['bundles']);
  }

  /**
   * Test without bundles.
   */
  public function testBuildHeader() {

    $list_builder = \Drupal::entityTypeManager()
      ->getListBuilder('crm_detail_type');

    $header = $list_builder->buildHeader();

    $this->assertCount(5, $header);
    $this->assertEquals('Label', $header['label']);
    $this->assertEquals('Machine name', $header['id']);
    $this->assertEquals('Status', $header['status']);
    $this->assertEquals('Bundles', $header['bundles']);
  }

}
