<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Kernel\Entity;

use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;

/**
 * Kernel tests for the ContactType::isLocked() method.
 *
 * @group crm
 * @covers \Drupal\crm\Entity\ContactType::isLocked
 */
class ContactTypeIsLockedTest extends EntityKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'crm',
    'datetime',
    'inline_entity_form',
    'primary_entity_reference',
    'name',
    'telephone',
    'address',
  ];

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('crm_contact');
    $this->installEntitySchema('crm_contact_type');
    $this->installConfig(['crm']);

    $this->entityTypeManager = $this->container->get('entity_type.manager');
    $this->state = $this->container->get('state');
  }

  /**
   * Gets the default date configuration for contact types.
   *
   * @return array
   *   The date configuration array.
   */
  protected function getDefaultDateConfig(): array {
    return [
      'start_date' => [
        'label' => '',
        'description' => '',
      ],
      'end_date' => [
        'label' => '',
        'description' => '',
      ],
    ];
  }

  /**
   * Tests that isLocked returns FALSE when no state is set.
   */
  public function testIsLockedReturnsFalseByDefault(): void {
    $contact_type = $this->entityTypeManager
      ->getStorage('crm_contact_type')
      ->create([
        'id' => 'test_type',
        'label' => 'Test Type',
        'date' => $this->getDefaultDateConfig(),
      ]);
    $contact_type->save();

    $this->assertFalse($contact_type->isLocked());
  }

  /**
   * Tests that isLocked returns TRUE when the type is locked in state.
   */
  public function testIsLockedReturnsTrueWhenLocked(): void {
    $contact_type = $this->entityTypeManager
      ->getStorage('crm_contact_type')
      ->create([
        'id' => 'locked_type',
        'label' => 'Locked Type',
        'date' => $this->getDefaultDateConfig(),
      ]);
    $contact_type->save();

    // Set the locked state.
    $this->state->set('crm.contact_type.locked', [
      'locked_type' => 'crm',
    ]);

    $this->assertEquals('crm', $contact_type->isLocked());
  }

  /**
   * Tests that isLocked returns FALSE for unlocked type when others are locked.
   */
  public function testIsLockedReturnsFalseForUnlockedType(): void {
    $locked_type = $this->entityTypeManager
      ->getStorage('crm_contact_type')
      ->create([
        'id' => 'locked_type',
        'label' => 'Locked Type',
        'date' => $this->getDefaultDateConfig(),
      ]);
    $locked_type->save();

    $unlocked_type = $this->entityTypeManager
      ->getStorage('crm_contact_type')
      ->create([
        'id' => 'unlocked_type',
        'label' => 'Unlocked Type',
        'date' => $this->getDefaultDateConfig(),
      ]);
    $unlocked_type->save();

    // Only lock one type.
    $this->state->set('crm.contact_type.locked', [
      'locked_type' => 'crm',
    ]);

    $this->assertEquals('crm', $locked_type->isLocked());
    $this->assertFalse($unlocked_type->isLocked());
  }

}
