<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Kernel\Entity;

use Drupal\crm\Entity\Contact;
use Drupal\crm\Entity\UserContactMapping;
use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;

/**
 * Kernel tests for the UserContactMapping entity.
 *
 * @group crm
 * @coversDefaultClass \Drupal\crm\Entity\UserContactMapping
 */
class UserContactMappingTest extends EntityKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'address',
    'crm',
    'inline_entity_form',
    'primary_entity_reference',
    'datetime',
    'name',
    'telephone',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('crm_contact');
    $this->installEntitySchema('crm_contact_detail');
    $this->installEntitySchema('crm_user_contact_mapping');

    $this->installConfig(['crm', 'name']);
  }

  /**
   * Tests entity creation and save.
   *
   * @covers ::baseFieldDefinitions
   */
  public function testEntityCreation(): void {
    $user = $this->createUser();
    $contact = $this->createContact();

    $mapping = UserContactMapping::create([
      'user' => $user->id(),
      'crm_contact' => $contact->id(),
    ]);
    $mapping->save();

    $this->assertNotNull($mapping->id());
    $this->assertEquals($user->id(), $mapping->getUserId());
    $this->assertEquals($contact->id(), $mapping->getContactId());
  }

  /**
   * Tests getUser() and setUser() methods.
   *
   * @covers ::getUser
   * @covers ::setUser
   */
  public function testGetSetUser(): void {
    $user = $this->createUser();
    $contact = $this->createContact();

    $mapping = UserContactMapping::create([
      'crm_contact' => $contact->id(),
    ]);
    $mapping->setUser($user);
    $mapping->save();

    $loadedMapping = UserContactMapping::load($mapping->id());
    $this->assertEquals($user->id(), $loadedMapping->getUser()->id());
  }

  /**
   * Tests getUserId() and setUserId() methods.
   *
   * @covers ::getUserId
   * @covers ::setUserId
   */
  public function testGetSetUserId(): void {
    $user = $this->createUser();
    $contact = $this->createContact();

    $mapping = UserContactMapping::create([
      'crm_contact' => $contact->id(),
    ]);
    $mapping->setUserId($user->id());
    $mapping->save();

    $loadedMapping = UserContactMapping::load($mapping->id());
    $this->assertEquals($user->id(), $loadedMapping->getUserId());
  }

  /**
   * Tests getContact() and setContact() methods.
   *
   * @covers ::getContact
   * @covers ::setContact
   */
  public function testGetSetContact(): void {
    $user = $this->createUser();
    $contact = $this->createContact();

    $mapping = UserContactMapping::create([
      'user' => $user->id(),
    ]);
    $mapping->setContact($contact);
    $mapping->save();

    $loadedMapping = UserContactMapping::load($mapping->id());
    $this->assertEquals($contact->id(), $loadedMapping->getContact()->id());
  }

  /**
   * Tests getContactId() and setContactId() methods.
   *
   * @covers ::getContactId
   * @covers ::setContactId
   */
  public function testGetSetContactId(): void {
    $user = $this->createUser();
    $contact = $this->createContact();

    $mapping = UserContactMapping::create([
      'user' => $user->id(),
    ]);
    $mapping->setContactId($contact->id());
    $mapping->save();

    $loadedMapping = UserContactMapping::load($mapping->id());
    $this->assertEquals($contact->id(), $loadedMapping->getContactId());
  }

  /**
   * Tests baseFieldDefinitions() returns expected fields.
   *
   * @covers ::baseFieldDefinitions
   */
  public function testBaseFieldDefinitions(): void {
    $entityType = $this->entityTypeManager->getDefinition('crm_user_contact_mapping');
    $fields = UserContactMapping::baseFieldDefinitions($entityType);

    $this->assertArrayHasKey('id', $fields);
    $this->assertArrayHasKey('uuid', $fields);
    $this->assertArrayHasKey('user', $fields);
    $this->assertArrayHasKey('crm_contact', $fields);
    $this->assertArrayHasKey('created', $fields);
    $this->assertArrayHasKey('changed', $fields);

    // Check user field configuration.
    $this->assertEquals('entity_reference', $fields['user']->getType());
    $this->assertEquals('user', $fields['user']->getSetting('target_type'));

    // Check crm_contact field configuration.
    $this->assertEquals('entity_reference', $fields['crm_contact']->getType());
    $this->assertEquals('crm_contact', $fields['crm_contact']->getSetting('target_type'));
  }

  /**
   * Tests label() with real entities.
   *
   * @covers ::label
   */
  public function testLabelWithRealEntities(): void {
    $user = $this->createUser([], 'test_user');
    $contact = $this->createContact('John Doe');

    $mapping = UserContactMapping::create([
      'user' => $user->id(),
      'crm_contact' => $contact->id(),
    ]);
    $mapping->save();

    $this->assertStringContainsString('test_user', $mapping->label());
    $this->assertStringContainsString('John Doe', $mapping->label());
  }

  /**
   * Tests label() with missing user reference.
   *
   * @covers ::label
   */
  public function testLabelWithMissingUser(): void {
    $contact = $this->createContact('Jane Doe');

    $mapping = UserContactMapping::create([
      'crm_contact' => $contact->id(),
    ]);

    $this->assertStringContainsString('Unknown User', $mapping->label());
    $this->assertStringContainsString('Jane Doe', $mapping->label());
  }

  /**
   * Tests label() with missing contact reference.
   *
   * @covers ::label
   */
  public function testLabelWithMissingContact(): void {
    $user = $this->createUser([], 'orphan_user');

    $mapping = UserContactMapping::create([
      'user' => $user->id(),
    ]);

    $this->assertStringContainsString('orphan_user', $mapping->label());
    $this->assertStringContainsString('Unknown Contact', $mapping->label());
  }

  /**
   * Creates a test contact entity.
   *
   * @param string $name
   *   The name for the contact.
   *
   * @return \Drupal\crm\Entity\Contact
   *   The created contact entity.
   */
  protected function createContact(string $name = 'Test Person'): Contact {
    $contact = Contact::create([
      'bundle' => 'person',
      'full_name' => ['given' => $name],
    ]);
    $contact->save();
    return $contact;
  }

}
