<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Kernel\Field;

use Drupal\crm\Entity\Relationship;
use Drupal\crm\Entity\RelationshipType;
use Drupal\crm\Entity\Contact;
use Drupal\crm\Entity\ContactType;
use Drupal\KernelTests\KernelTestBase;

/**
 * Tests the AgeFieldItemList computed field.
 *
 * @group crm
 * @coversDefaultClass \Drupal\crm\Field\AgeFieldItemList
 */
class AgeFieldItemListTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'address',
    'crm',
    'datetime',
    'field',
    'inline_entity_form',
    'primary_entity_reference',
    'name',
    'system',
    'telephone',
    'text',
    'user',
  ];

  /**
   * A relationship type entity.
   *
   * @var \Drupal\crm\Entity\RelationshipType
   */
  protected $relationshipType;

  /**
   * A contact type entity.
   *
   * @var \Drupal\crm\Entity\ContactType
   */
  protected $contactType;

  /**
   * Contact entities for testing.
   *
   * @var \Drupal\crm\Entity\Contact[]
   */
  protected $contacts;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('crm_contact');
    $this->installEntitySchema('crm_contact_type');
    $this->installEntitySchema('crm_relationship');
    $this->installEntitySchema('crm_relationship_type');
    $this->installSchema('system', ['sequences']);

    // Create a contact type.
    $this->contactType = ContactType::create([
      'id' => 'person',
      'label' => 'Person',
      'date' => [
        'start_date' => [
          'label' => 'Date of birth',
        ],
        'end_date' => [
          'label' => 'Date of death',
        ],
      ],
    ]);
    $this->contactType->save();

    // Create a relationship type.
    $this->relationshipType = RelationshipType::create([
      'id' => 'test_relationship',
      'label' => 'Test Relationship',
      'contact_type_a' => ['person'],
      'asymmetric' => FALSE,
    ]);
    $this->relationshipType->save();

    // Create test contacts.
    $this->contacts[0] = Contact::create([
      'bundle' => 'person',
      'name' => 'John Doe',
    ]);
    $this->contacts[0]->save();

    $this->contacts[1] = Contact::create([
      'bundle' => 'person',
      'name' => 'Jane Doe',
    ]);
    $this->contacts[1]->save();
  }

  /**
   * Tests age computation with start and end dates.
   *
   * @covers ::computeValue
   */
  public function testAgeComputationWithBothDates() {
    $start_date = '2020-01-01';
    // Exactly 1 year = 365 days (2020 was a leap year, so 366 days)
    $end_date = '2021-01-01';

    $relationship = Relationship::create([
      'bundle' => 'test_relationship',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[1]->id()],
      'start_date' => $start_date,
      'end_date' => $end_date,
    ]);

    $age_field = $relationship->get('age');
    $age_value = $age_field->value;

    $expected_days = (new \DateTime($end_date))->diff(new \DateTime($start_date))->days;
    $this->assertEquals($expected_days, $age_value,
      "Age should be $expected_days days between $start_date and $end_date");
  }

  /**
   * Tests age computation with only start date (end date defaults to now).
   *
   * @covers ::computeValue
   */
  public function testAgeComputationWithStartDateOnly() {
    $start_date = '2023-01-01';

    $relationship = Relationship::create([
      'bundle' => 'test_relationship',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[1]->id()],
      'start_date' => $start_date,
      // end_date not set, should default to 'now'.
    ]);

    $age_field = $relationship->get('age');
    $age_value = $age_field->value;

    $expected_days = (new \DateTime('now'))->diff(new \DateTime($start_date))->days;

    // Allow for some variance (1 day) in case test runs over midnight.
    $this->assertGreaterThanOrEqual($expected_days - 1, $age_value);
    $this->assertLessThanOrEqual($expected_days + 1, $age_value);
  }

  /**
   * Tests age computation with no start date.
   *
   * @covers ::computeValue
   */
  public function testAgeComputationWithoutStartDate() {
    $relationship = Relationship::create([
      'bundle' => 'test_relationship',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[1]->id()],
      'end_date' => '2021-01-01',
      // start_date not set.
    ]);

    $age_field = $relationship->get('age');

    // Should be empty when no start date is provided.
    $this->assertTrue($age_field->isEmpty(), 'Age field should be empty when no start date is provided');
  }

  /**
   * Tests age computation with various date ranges.
   *
   * @covers ::computeValue
   */
  public function testAgeComputationVariousRanges() {
    $test_cases = [
      // Same day.
      [
        'start' => '2023-06-15',
        'end' => '2023-06-15',
        'expected' => 0,
      ],
      // One day difference.
      [
        'start' => '2023-06-15',
        'end' => '2023-06-16',
        'expected' => 1,
      ],
      // One month difference.
      [
        'start' => '2023-06-15',
        'end' => '2023-07-15',
        'expected' => 30,
      ],
      // Leap year test - 2020 was a leap year.
      [
        'start' => '2020-02-28',
        'end' => '2020-03-01',
      // Feb 29 exists in 2020.
        'expected' => 2,
      ],
      // Non-leap year test.
      [
        'start' => '2021-02-28',
        'end' => '2021-03-01',
      // Feb 29 doesn't exist in 2021.
        'expected' => 1,
      ],
      // Multiple years.
      [
        'start' => '2020-01-01',
        'end' => '2023-01-01',
      // 3 years including leap year 2020
        'expected' => 1096,
      ],
    ];

    foreach ($test_cases as $i => $case) {
      $relationship = Relationship::create([
        'bundle' => 'test_relationship',
        'contacts' => [$this->contacts[0]->id(), $this->contacts[1]->id()],
        'start_date' => $case['start'],
        'end_date' => $case['end'],
      ]);

      $age_field = $relationship->get('age');
      $age_value = $age_field->value;

      $this->assertEquals($case['expected'], $age_value,
        "Test case $i: Expected {$case['expected']} days between {$case['start']} and {$case['end']}, got $age_value");
    }
  }

  /**
   * Tests that the field is properly computed.
   *
   * @covers ::computeValue
   */
  public function testFieldIsComputed() {
    $relationship = Relationship::create([
      'bundle' => 'test_relationship',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[1]->id()],
      'start_date' => '2023-01-01',
      'end_date' => '2023-12-31',
    ]);

    // The age field should compute to an integer value.
    $age_value = $relationship->get('age')->value;
    $this->assertIsInt($age_value);

    // Verify the computed value is correct (364 days for this date range)
    $expected_days = (new \DateTime('2023-12-31'))->diff(new \DateTime('2023-01-01'))->days;
    $this->assertEquals($expected_days, $age_value);
  }

  /**
   * Tests field computation with different date ranges.
   *
   * @covers ::computeValue
   */
  public function testFieldComputationWithDifferentDateRanges() {
    // Test first date range.
    $relationship1 = Relationship::create([
      'bundle' => 'test_relationship',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[1]->id()],
      'start_date' => '2023-01-01',
    // 1 day
      'end_date' => '2023-01-02',
    ]);

    $age1 = $relationship1->get('age')->value;
    $this->assertEquals(1, $age1);

    // Test second date range.
    $relationship2 = Relationship::create([
      'bundle' => 'test_relationship',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[1]->id()],
      'start_date' => '2023-01-01',
    // 9 days
      'end_date' => '2023-01-10',
    ]);

    $age2 = $relationship2->get('age')->value;
    $this->assertEquals(9, $age2);
  }

  /**
   * Tests field with future start date.
   *
   * @covers ::computeValue
   */
  public function testFieldWithFutureStartDate() {
    $future_date = (new \DateTime('+1 year'))->format('Y-m-d');

    $relationship = Relationship::create([
      'bundle' => 'test_relationship',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[1]->id()],
      'start_date' => $future_date,
      // end_date defaults to 'now'.
    ]);

    $age_field = $relationship->get('age');
    $age_value = $age_field->value;

    $expected_days = (new \DateTime('now'))
      ->diff(new \DateTime($future_date))
      ->days;

    $this->assertEquals($expected_days, $age_value);
  }

}
