<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Kernel\Form;

use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;
use Drupal\crm\Form\ContactForm;
use Drupal\crm\Entity\Contact;
use Drupal\Core\Form\FormState;

/**
 * Kernel tests for the ContactForm.
 *
 * @group crm
 * @covers \Drupal\crm\Form\ContactForm
 */
class ContactFormTest extends EntityKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'crm',
    'datetime',
    'inline_entity_form',
    'primary_entity_reference',
    'name',
    'telephone',
    'address',
  ];

  /**
   * The form under test.
   *
   * @var \Drupal\crm\Form\ContactForm
   */
  protected $form;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The contact storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $contactStorage;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('crm_contact');
    $this->installEntitySchema('crm_contact_detail');
    $this->installEntitySchema('crm_relationship');
    $this->installEntitySchema('crm_user_contact_mapping');
    $this->installConfig(['system', 'crm']);

    $this->entityTypeManager = $this->container->get('entity_type.manager');
    $this->contactStorage = $this->entityTypeManager->getStorage('crm_contact');

    $this->form = ContactForm::create($this->container);
    $this->form->setModuleHandler($this->container->get('module_handler'));
    $this->form->setEntityTypeManager($this->entityTypeManager);
  }

  /**
   * Tests form creation and dependency injection.
   */
  public function testFormCreation(): void {
    $this->assertInstanceOf(ContactForm::class, $this->form);
  }

  /**
   * Tests building the form with a new contact.
   */
  public function testBuildFormWithNewContact(): void {
    $contact = Contact::create([
      'bundle' => 'person',
      'name' => 'Test Person',
    ]);

    $this->form->setEntity($contact);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    $this->assertIsArray($form);

    // Check for meta section.
    $this->assertArrayHasKey('meta', $form);
    $this->assertEquals('details', $form['meta']['#type']);

    // Check for age section.
    $this->assertArrayHasKey('age', $form);
    $this->assertEquals('details', $form['age']['#type']);

    // Check that status is moved to meta.
    $this->assertArrayHasKey('status', $form['meta']);

    // Check that revision is set to TRUE by default.
    $this->assertTrue($form['revision']['#default_value']);
  }

  /**
   * Tests building the form with an existing contact.
   */
  public function testBuildFormWithExistingContact(): void {
    // Create and save a contact.
    $contact = Contact::create([
      'bundle' => 'organization',
      'name' => 'Test Organization',
      'status' => TRUE,
    ]);
    $contact->save();

    // Reload the contact to get a fresh entity.
    $contact = $this->contactStorage->load($contact->id());

    $this->form->setEntity($contact);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    $this->assertIsArray($form);

    // Check that meta section shows correct status.
    $this->assertArrayHasKey('meta', $form);
    $this->assertArrayHasKey('published', $form['meta']);

    // Check that changed time is displayed.
    $this->assertArrayHasKey('changed', $form['meta']);
  }

  /**
   * Tests date field label customization based on contact type settings.
   */
  public function testDateFieldLabelCustomization(): void {
    // Load and update the person contact type with custom date labels.
    $contact_type = $this->entityTypeManager->getStorage('crm_contact_type')->load('person');
    $contact_type->set('date', [
      'start_date' => [
        'label' => 'Birth Date',
        'description' => 'The date of birth',
      ],
      'end_date' => [
        'label' => 'Death Date',
        'description' => 'The date of death',
      ],
    ]);
    $contact_type->save();

    $contact = Contact::create([
      'bundle' => 'person',
      'name' => 'Test Person',
    ]);

    $this->form->setEntity($contact);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    // Check that the date fields are in the age section.
    $this->assertArrayHasKey('age', $form);
    $this->assertArrayHasKey('start_date', $form['age']);
    $this->assertArrayHasKey('end_date', $form['age']);

    // Check that custom labels are applied if widget exists.
    if (isset($form['age']['start_date']['widget'][0]['#title'])) {
      $this->assertEquals('Birth Date', $form['age']['start_date']['widget'][0]['#title']);
    }
    if (isset($form['age']['end_date']['widget'][0]['#title'])) {
      $this->assertEquals('Death Date', $form['age']['end_date']['widget'][0]['#title']);
    }
  }

  /**
   * Tests saving a new contact.
   */
  public function testSaveNewContact(): void {
    $contact = Contact::create([
      'bundle' => 'person',
      'name' => 'New Test Person',
      'status' => TRUE,
    ]);

    $this->form->setEntity($contact);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    // Save the entity directly (form submission is tested in functional tests).
    $result = $this->form->save($form, $form_state);

    // Verify the entity was saved.
    $this->assertEquals(SAVED_NEW, $result);

    // Verify the entity exists in the database.
    $saved_contact = $this->contactStorage->load($contact->id());
    $this->assertNotNull($saved_contact);
    $this->assertEquals('New Test Person', $saved_contact->get('name')->value);
  }

  /**
   * Tests saving an updated contact creates a new revision.
   */
  public function testSaveUpdatedContact(): void {
    // Create and save a contact first.
    $contact = Contact::create([
      'bundle' => 'organization',
      'name' => 'Original Name',
      'status' => TRUE,
    ]);
    $contact->save();
    $original_revision_id = $contact->getRevisionId();

    // Reload the contact.
    $contact = $this->contactStorage->load($contact->id());
    $contact->set('name', 'Updated Name');

    $this->form->setEntity($contact);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    // Save the entity directly (form submission is tested in functional tests).
    $result = $this->form->save($form, $form_state);

    // Verify the entity was updated.
    $this->assertEquals(SAVED_UPDATED, $result);

    // Reload and verify a new revision was created.
    $this->contactStorage->resetCache([$contact->id()]);
    $updated_contact = $this->contactStorage->load($contact->id());
    $this->assertNotEquals($original_revision_id, $updated_contact->getRevisionId());
    $this->assertEquals('Updated Name', $updated_contact->get('name')->value);
  }

  /**
   * Tests form structure includes advanced section.
   */
  public function testFormAdvancedSection(): void {
    $contact = Contact::create([
      'bundle' => 'household',
      'name' => 'Test Household',
    ]);

    $this->form->setEntity($contact);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    // Check advanced section has the entity-meta class.
    $this->assertArrayHasKey('advanced', $form);
    $this->assertContains('entity-meta', $form['advanced']['#attributes']['class']);
  }

  /**
   * Tests that created field is moved to meta section when present.
   */
  public function testCreatedFieldInMetaSection(): void {
    $contact = Contact::create([
      'bundle' => 'person',
      'name' => 'Test Person',
    ]);
    $contact->save();

    $contact = $this->contactStorage->load($contact->id());

    $this->form->setEntity($contact);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    // The created field should be moved to meta if it exists.
    if (isset($form['meta']['created'])) {
      $this->assertEquals(200, $form['meta']['created']['#weight']);
    }
  }

  /**
   * Tests redirect after save for contacts with view access.
   */
  public function testRedirectAfterSave(): void {
    $contact = Contact::create([
      'bundle' => 'person',
      'name' => 'Redirect Test Person',
      'status' => TRUE,
    ]);

    $this->form->setEntity($contact);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    // Save the entity directly (form submission is tested in functional tests).
    $this->form->save($form, $form_state);

    // Verify that a redirect was set.
    $redirect = $form_state->getRedirect();
    $this->assertNotNull($redirect);
  }

}
