<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Kernel\Form;

use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;
use Drupal\crm\Form\ContactTypeForm;
use Drupal\Core\Form\FormState;

/**
 * Kernel tests for the ContactTypeForm.
 *
 * @group crm
 * @covers \Drupal\crm\Form\ContactTypeForm
 */
class ContactTypeFormTest extends EntityKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'crm',
    'datetime',
    'inline_entity_form',
    'primary_entity_reference',
    'name',
    'telephone',
    'address',
  ];

  /**
   * The form under test.
   *
   * @var \Drupal\crm\Form\ContactTypeForm
   */
  protected $form;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The relationship type storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $relationshipTypeStorage;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('crm_contact');
    $this->installEntitySchema('crm_contact_type');
    $this->installEntitySchema('crm_relationship_type');
    $this->installConfig(['crm']);

    $this->entityTypeManager = $this->container->get('entity_type.manager');
    $this->relationshipTypeStorage = $this->entityTypeManager->getStorage('crm_relationship_type');

    $this->form = ContactTypeForm::create($this->container);
    $this->form->setModuleHandler($this->container->get('module_handler'));
  }

  /**
   * Tests form creation and dependency injection.
   */
  public function testFormCreation(): void {
    $this->assertInstanceOf(ContactTypeForm::class, $this->form);

    // Test that the form was created with proper dependency injection.
    $reflection = new \ReflectionClass($this->form);
    $storageProperty = $reflection->getProperty('storage');
    $storageProperty->setAccessible(TRUE);
    $storage = $storageProperty->getValue($this->form);

    $this->assertSame($this->relationshipTypeStorage, $storage);
  }

  /**
   * Tests building the form with a new contact type.
   */
  public function testBuildFormWithNewContactType(): void {
    $contact_type = $this->entityTypeManager
      ->getStorage('crm_contact_type')
      ->create([
        'id' => 'test_type',
        'label' => 'Test Type',
        'date' => [
          'start_date' => [
            'label' => '',
            'description' => '',
          ],
          'end_date' => [
            'label' => '',
            'description' => '',
          ],
        ],
      ]);

    $this->form->setEntity($contact_type);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    $this->assertIsArray($form);
    $this->assertArrayHasKey('label', $form);
    $this->assertArrayHasKey('id', $form);
    $this->assertArrayHasKey('description', $form);
    $this->assertArrayHasKey('additional_settings', $form);
    $this->assertArrayHasKey('start_date', $form);
    $this->assertArrayHasKey('end_date', $form);

    // Test default values for new entity.
    $this->assertEquals('Test Type', $form['label']['#default_value']);
    $this->assertEquals('test_type', $form['id']['#default_value']);
    $this->assertEquals('', $form['description']['#default_value']);
  }

  /**
   * Tests building the form with an existing contact type.
   */
  public function testBuildFormWithExistingContactType(): void {
    $contact_type = $this->entityTypeManager
      ->getStorage('crm_contact_type')
      ->create([
        'id' => 'existing_type',
        'label' => 'Existing Type',
        'description' => 'Test description',
        'date' => [
          'start_date' => [
            'label' => 'Start Date',
            'description' => 'Start description',
          ],
          'end_date' => [
            'label' => 'End Date',
            'description' => 'End description',
          ],
        ],
      ]);
    $contact_type->save();

    $this->form->setEntity($contact_type);
    $this->form->setOperation('edit');

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    // Test that form contains edit title.
    $this->assertArrayHasKey('#title', $form);
    $this->assertEquals('Edit Existing Type contact type', $form['#title']);

    // Test that default values are populated from entity.
    $this->assertEquals('Existing Type', $form['label']['#default_value']);
    $this->assertEquals('existing_type', $form['id']['#default_value']);
    $this->assertEquals('Test description', $form['description']['#default_value']);
    $this->assertEquals('Start Date', $form['start_date']['start_date_label']['#default_value']);
    $this->assertEquals('Start description', $form['start_date']['start_date_description']['#default_value']);
    $this->assertEquals('End Date', $form['end_date']['end_date_label']['#default_value']);
    $this->assertEquals('End description', $form['end_date']['end_date_description']['#default_value']);
  }

  /**
   * Tests form validation through form elements.
   */
  public function testFormValidation(): void {
    $contact_type = $this->entityTypeManager
      ->getStorage('crm_contact_type')
      ->create([
        'id' => 'test_type',
        'label' => 'Test Type',
        'date' => [
          'start_date' => [
            'label' => '',
            'description' => '',
          ],
          'end_date' => [
            'label' => '',
            'description' => '',
          ],
        ],
      ]);

    $this->form->setEntity($contact_type);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    // Test that required field is properly marked.
    $this->assertTrue($form['label']['#required']);
    $this->assertEquals('textfield', $form['label']['#type']);

    // Test that machine name field has proper validation callback.
    $this->assertEquals('machine_name', $form['id']['#type']);
    $this->assertArrayHasKey('exists', $form['id']['#machine_name']);
    $this->assertEquals(['Drupal\crm\Entity\ContactType', 'load'], $form['id']['#machine_name']['exists']);
  }

  /**
   * Tests saving a new contact type.
   */
  public function testSaveNewContactType(): void {
    $contact_type = $this->entityTypeManager
      ->getStorage('crm_contact_type')
      ->create([
        'id' => 'new_test_type',
        'label' => 'New Test Type',
        'date' => [
          'start_date' => [
            'label' => '',
            'description' => '',
          ],
          'end_date' => [
            'label' => '',
            'description' => '',
          ],
        ],
      ]);

    $this->form->setEntity($contact_type);

    $form_state = new FormState();
    $form_state->setValues([
      'label' => 'New Test Type',
      'id' => 'new_test_type',
      'description' => 'New test description',
      'start_date_label' => 'Custom Start',
      'start_date_description' => 'Custom start description',
      'end_date_label' => 'Custom End',
      'end_date_description' => 'Custom end description',
    ]);

    $form = $this->form->buildForm([], $form_state);
    $result = $this->form->save($form, $form_state);

    $this->assertEquals(SAVED_NEW, $result);

    // Verify the entity was saved with correct values.
    $saved_entity = $this->entityTypeManager
      ->getStorage('crm_contact_type')
      ->load('new_test_type');

    $this->assertNotNull($saved_entity);
    $this->assertEquals('New Test Type', $saved_entity->label());
    $this->assertEquals('new_test_type', $saved_entity->id());
    $this->assertEquals('New test description', $saved_entity->get('description'));

    $date_config = $saved_entity->get('date');
    $this->assertEquals('Custom Start', $date_config['start_date']['label']);
    $this->assertEquals('Custom start description', $date_config['start_date']['description']);
    $this->assertEquals('Custom End', $date_config['end_date']['label']);
    $this->assertEquals('Custom end description', $date_config['end_date']['description']);
  }

  /**
   * Tests updating an existing contact type.
   */
  public function testSaveExistingContactType(): void {
    // Create and save initial contact type.
    $contact_type = $this->entityTypeManager
      ->getStorage('crm_contact_type')
      ->create([
        'id' => 'existing_test_type',
        'label' => 'Original Label',
        'description' => 'Original description',
        'date' => [
          'start_date' => [
            'label' => '',
            'description' => '',
          ],
          'end_date' => [
            'label' => '',
            'description' => '',
          ],
        ],
      ]);
    $contact_type->save();

    $this->form->setEntity($contact_type);
    $this->form->setOperation('edit');

    $form_state = new FormState();
    $form_state->setValues([
      'label' => 'Updated Label',
      'description' => 'Updated description',
      'start_date_label' => 'Updated Start',
      'start_date_description' => 'Updated start description',
      'end_date_label' => 'Updated End',
      'end_date_description' => 'Updated end description',
    ]);

    $form = $this->form->buildForm([], $form_state);
    $result = $this->form->save($form, $form_state);

    $this->assertEquals(SAVED_UPDATED, $result);

    // Reload and verify the entity was updated.
    $this->entityTypeManager->getStorage('crm_contact_type')->resetCache();
    $updated_entity = $this->entityTypeManager
      ->getStorage('crm_contact_type')
      ->load('existing_test_type');

    $this->assertEquals('Updated Label', $updated_entity->label());
    $this->assertEquals('Updated description', $updated_entity->get('description'));

    $date_config = $updated_entity->get('date');
    $this->assertEquals('Updated Start', $date_config['start_date']['label']);
    $this->assertEquals('Updated start description', $date_config['start_date']['description']);
    $this->assertEquals('Updated End', $date_config['end_date']['label']);
    $this->assertEquals('Updated end description', $date_config['end_date']['description']);
  }

  /**
   * Tests the getRelationshipTypeOptions method indirectly.
   */
  public function testRelationshipTypeOptionsIntegration(): void {
    // Create a contact type.
    $contact_type = $this->entityTypeManager
      ->getStorage('crm_contact_type')
      ->create([
        'id' => 'test_contact_type',
        'label' => 'Test Contact Type',
        'date' => [
          'start_date' => [
            'label' => '',
            'description' => '',
          ],
          'end_date' => [
            'label' => '',
            'description' => '',
          ],
        ],
      ]);
    $contact_type->save();

    // Create relationship types that reference this contact type.
    $relationship_type_1 = $this->entityTypeManager
      ->getStorage('crm_relationship_type')
      ->create([
        'id' => 'test_relationship_1',
        'label' => 'Test Relationship 1',
        'label_a' => 'Test Relationship 1 A',
        'label_b' => 'Test Relationship 1 B',
        'asymmetric' => TRUE,
        'contact_type_a' => ['test_contact_type'],
        'contact_type_b' => ['person'],
      ]);
    $relationship_type_1->save();

    $relationship_type_2 = $this->entityTypeManager
      ->getStorage('crm_relationship_type')
      ->create([
        'id' => 'test_relationship_2',
        'label' => 'Test Relationship 2',
        'label_a' => 'Test Relationship 2 A',
        'label_b' => 'Test Relationship 2 B',
        'asymmetric' => TRUE,
        'contact_type_a' => ['person'],
        'contact_type_b' => ['test_contact_type'],
      ]);
    $relationship_type_2->save();

    $this->form->setEntity($contact_type);

    // Use reflection to test the private method.
    $reflection = new \ReflectionClass($this->form);
    $method = $reflection->getMethod('getRelationshipTypeOptions');
    $method->setAccessible(TRUE);

    $options = $method->invoke($this->form);

    $this->assertIsArray($options);
    $this->assertArrayHasKey('test_relationship_1', $options);
    $this->assertArrayHasKey('test_relationship_2', $options);

    // Verify the correct labels are used based on the relationship direction.
    $this->assertEquals('Test Relationship 1', $options['test_relationship_1']);
    $this->assertEquals('Test Relationship 2 B', $options['test_relationship_2']);
  }

  /**
   * Tests form actions.
   */
  public function testFormActions(): void {
    $contact_type = $this->entityTypeManager
      ->getStorage('crm_contact_type')
      ->create([
        'id' => 'test_type',
        'label' => 'Test Type',
        'date' => [
          'start_date' => [
            'label' => '',
            'description' => '',
          ],
          'end_date' => [
            'label' => '',
            'description' => '',
          ],
        ],
      ]);

    $this->form->setEntity($contact_type);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    $this->assertArrayHasKey('actions', $form);
    $this->assertArrayHasKey('submit', $form['actions']);
    $this->assertArrayHasKey('delete', $form['actions']);

    $this->assertEquals('Save contact type', $form['actions']['submit']['#value']);
    $this->assertEquals('Delete contact type', $form['actions']['delete']['#value']);
  }

}
