<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Kernel\Form;

use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;
use Drupal\crm\Form\DetailTypeForm;
use Drupal\crm\Entity\DetailType;
use Drupal\Core\Form\FormState;

/**
 * Kernel tests for the DetailTypeForm.
 *
 * @group crm
 * @covers \Drupal\crm\Form\DetailTypeForm
 */
class DetailTypeFormTest extends EntityKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'crm',
    'datetime',
    'inline_entity_form',
    'primary_entity_reference',
    'name',
    'telephone',
    'address',
  ];

  /**
   * The form under test.
   *
   * @var \Drupal\crm\Form\DetailTypeForm
   */
  protected $form;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The detail type storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $detailTypeStorage;

  /**
   * The contact detail type storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $contactDetailTypeStorage;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('crm_contact');
    $this->installEntitySchema('crm_contact_detail');
    $this->installEntitySchema('crm_contact_detail_type');
    $this->installEntitySchema('crm_detail_type');
    $this->installConfig(['crm']);

    $this->entityTypeManager = $this->container->get('entity_type.manager');
    $this->detailTypeStorage = $this->entityTypeManager->getStorage('crm_detail_type');
    $this->contactDetailTypeStorage = $this->entityTypeManager->getStorage('crm_contact_detail_type');

    $this->form = DetailTypeForm::create($this->container);
    $this->form->setModuleHandler($this->container->get('module_handler'));
    $this->form->setEntityTypeManager($this->entityTypeManager);
  }

  /**
   * Tests form creation and dependency injection.
   */
  public function testFormCreation(): void {
    $this->assertInstanceOf(DetailTypeForm::class, $this->form);
  }

  /**
   * Tests building the form with a new detail type.
   */
  public function testBuildFormWithNewDetailType(): void {
    // Verify the contact detail types exist.
    $address_type = $this->contactDetailTypeStorage->load('address');
    $email_type = $this->contactDetailTypeStorage->load('email');
    $telephone_type = $this->contactDetailTypeStorage->load('telephone');
    $this->assertNotNull($address_type);
    $this->assertNotNull($email_type);
    $this->assertNotNull($telephone_type);

    $detail_type = DetailType::create([
      'id' => 'kernel_test_type',
      'label' => 'Test Detail Type',
      'status' => TRUE,
    ]);

    $this->form->setEntity($detail_type);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    $this->assertIsArray($form);
    $this->assertArrayHasKey('label', $form);
    $this->assertArrayHasKey('id', $form);
    $this->assertArrayHasKey('status', $form);
    $this->assertArrayHasKey('description', $form);
    $this->assertArrayHasKey('bundles', $form);
    $this->assertArrayHasKey('negate', $form);

    // Test form field properties.
    $this->assertEquals('textfield', $form['label']['#type']);
    $this->assertEquals('Test Detail Type', $form['label']['#default_value']);
    $this->assertTrue($form['label']['#required']);

    $this->assertEquals('machine_name', $form['id']['#type']);
    $this->assertEquals('kernel_test_type', $form['id']['#default_value']);
    // Should not be disabled for new entity.
    $this->assertFalse($form['id']['#disabled']);

    $this->assertEquals('checkbox', $form['status']['#type']);
    // Default status should be TRUE.
    $this->assertTrue($form['status']['#default_value']);

    $this->assertEquals('textarea', $form['description']['#type']);
    $this->assertEquals('', $form['description']['#default_value']);

    $this->assertEquals('checkboxes', $form['bundles']['#type']);
    // The options should match the labels from
    // the installed contact detail types.
    $expected_options = [
      'address' => 'Address',
      'email' => 'Email address',
      'telephone' => 'Telephone',
    ];
    $this->assertEquals($expected_options, $form['bundles']['#options']);
    $this->assertEquals([], $form['bundles']['#default_value']);
    $this->assertTrue($form['bundles']['#multiple']);

    $this->assertEquals('checkbox', $form['negate']['#type']);
    $this->assertFalse($form['negate']['#default_value']);
  }

  /**
   * Tests building the form with an existing detail type.
   */
  public function testBuildFormWithExistingDetailType(): void {
    // Use the existing contact detail types that are installed by default.
    // Create and save a detail type.
    $detail_type = DetailType::create([
      'id' => 'kernel_existing_test',
      'label' => 'Work',
      'description' => 'Work-related contact information',
      'bundles' => ['address', 'telephone'],
      'negate' => TRUE,
      'status' => TRUE,
    ]);
    $detail_type->save();

    $this->form->setEntity($detail_type);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    // Test that existing values are properly loaded.
    $this->assertEquals('Work', $form['label']['#default_value']);
    $this->assertEquals('kernel_existing_test', $form['id']['#default_value']);
    // Should be disabled for existing entity.
    $this->assertTrue($form['id']['#disabled']);
    $this->assertTrue($form['status']['#default_value']);
    $this->assertEquals('Work-related contact information', $form['description']['#default_value']);
    $this->assertEquals(['address', 'telephone'], $form['bundles']['#default_value']);
    $this->assertTrue($form['negate']['#default_value']);
  }

  /**
   * Tests submitting the form to create a new detail type.
   */
  public function testSubmitFormCreateNewDetailType(): void {
    // Use the existing contact detail types that are installed by default.
    $detail_type = DetailType::create([
      'id' => 'kernel_home_test',
      'label' => 'Home',
      'status' => TRUE,
    ]);

    $this->form->setEntity($detail_type);

    $form_state = new FormState();
    $form_state->setValues([
      'label' => 'Home',
      'id' => 'kernel_home_test',
      'status' => TRUE,
      'description' => 'Home contact information',
      'bundles' => ['address' => 'address', 'email' => 'email'],
      'negate' => FALSE,
    ]);

    $form = $this->form->buildForm([], $form_state);
    $this->form->submitForm($form, $form_state);
    $this->form->save($form, $form_state);

    // Verify the entity was saved.
    $saved_detail_type = $this->detailTypeStorage->load('kernel_home_test');
    $this->assertNotNull($saved_detail_type);
    $this->assertEquals('Home', $saved_detail_type->label());
    $this->assertEquals('Home contact information', $saved_detail_type->get('description'));
    $this->assertEquals(['address', 'email'], $saved_detail_type->get('bundles'));
    $this->assertFalse($saved_detail_type->get('negate'));
    $this->assertTrue($saved_detail_type->status());
  }

  /**
   * Tests submitting the form to update an existing detail type.
   */
  public function testSubmitFormUpdateExistingDetailType(): void {
    // Use the existing contact detail types that are installed by default.
    // Create and save a detail type.
    $detail_type = DetailType::create([
      'id' => 'kernel_mobile_test',
      'label' => 'Mobile',
      'description' => 'Mobile phone',
      'bundles' => ['telephone'],
      'negate' => FALSE,
      'status' => TRUE,
    ]);
    $detail_type->save();

    $this->form->setEntity($detail_type);

    $form_state = new FormState();
    $form_state->setValues([
      'label' => 'Mobile Phone',
      'id' => 'kernel_mobile_test',
      'status' => TRUE,
      'description' => 'Mobile phone number',
      'bundles' => ['telephone' => 'telephone', 'address' => 0],
      'negate' => TRUE,
    ]);

    $form = $this->form->buildForm([], $form_state);
    $this->form->submitForm($form, $form_state);
    $this->form->save($form, $form_state);

    // Verify the entity was updated.
    $updated_detail_type = $this->detailTypeStorage->load('kernel_mobile_test');
    $this->assertNotNull($updated_detail_type);
    $this->assertEquals('Mobile Phone', $updated_detail_type->label());
    $this->assertEquals('Mobile phone number', $updated_detail_type->get('description'));
    $this->assertEquals(['telephone'], $updated_detail_type->get('bundles'));
    $this->assertTrue($updated_detail_type->get('negate'));
  }

  /**
   * Tests form validation setup for required fields.
   */
  public function testFormValidationSetup(): void {
    $detail_type = DetailType::create([
      'id' => 'kernel_validation_test',
      'label' => '',
      'status' => TRUE,
    ]);

    $this->form->setEntity($detail_type);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    // Test that required field is properly marked.
    $this->assertTrue($form['label']['#required']);
    $this->assertEquals('textfield', $form['label']['#type']);

    // Test that machine name field has proper validation callback.
    $this->assertEquals('machine_name', $form['id']['#type']);
    $this->assertArrayHasKey('exists', $form['id']['#machine_name']);
    $this->assertEquals('\Drupal\crm\Entity\DetailType::load', $form['id']['#machine_name']['exists']);
  }

  /**
   * Tests form with bundles filtering (negate functionality).
   */
  public function testFormWithNegateOption(): void {
    // Use the existing contact detail types that are installed by default.
    // Create detail type that applies to all bundles except selected ones.
    $detail_type = DetailType::create([
      'id' => 'kernel_negate_test',
      'label' => 'Personal',
      'description' => 'Personal contact information',
    // Only exclude address.
      'bundles' => ['address'],
    // Apply to all except address.
      'negate' => TRUE,
      'status' => TRUE,
    ]);

    $this->form->setEntity($detail_type);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    // Verify negate field is properly set.
    $this->assertTrue($form['negate']['#default_value']);
    $this->assertEquals(['address'], $form['bundles']['#default_value']);
  }

}
