<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Kernel\Form;

use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;
use Drupal\crm\Form\RelationshipForm;
use Drupal\crm\Entity\Contact;
use Drupal\crm\Entity\Relationship;
use Drupal\crm\Entity\RelationshipType;
use Drupal\Core\Form\FormState;

/**
 * Kernel tests for the RelationshipForm.
 *
 * @group crm
 * @covers \Drupal\crm\Form\RelationshipForm
 */
class RelationshipFormTest extends EntityKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'crm',
    'datetime',
    'inline_entity_form',
    'primary_entity_reference',
    'name',
    'telephone',
    'address',
  ];

  /**
   * The form under test.
   *
   * @var \Drupal\crm\Form\RelationshipForm
   */
  protected $form;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The relationship storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $relationshipStorage;

  /**
   * The contact storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $contactStorage;

  /**
   * The relationship type storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $relationshipTypeStorage;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('crm_contact');
    $this->installEntitySchema('crm_contact_detail');
    $this->installEntitySchema('crm_relationship');
    $this->installEntitySchema('crm_user_contact_mapping');
    $this->installConfig(['system', 'crm']);

    $this->entityTypeManager = $this->container->get('entity_type.manager');
    $this->relationshipStorage = $this->entityTypeManager->getStorage('crm_relationship');
    $this->contactStorage = $this->entityTypeManager->getStorage('crm_contact');
    $this->relationshipTypeStorage = $this->entityTypeManager->getStorage('crm_relationship_type');

    $this->form = RelationshipForm::create($this->container);
    $this->form->setModuleHandler($this->container->get('module_handler'));
    $this->form->setEntityTypeManager($this->entityTypeManager);
  }

  /**
   * Creates a relationship type for testing.
   *
   * @param string $id
   *   The relationship type ID.
   * @param array $values
   *   Additional values.
   *
   * @return \Drupal\crm\Entity\RelationshipType
   *   The created relationship type.
   */
  protected function createRelationshipType(string $id, array $values = []): RelationshipType {
    $defaults = [
      'id' => $id,
      'label' => ucfirst(str_replace('_', ' ', $id)),
      'asymmetric' => TRUE,
      'label_a' => 'Contact A',
      'label_b' => 'Contact B',
      'contact_type_a' => ['person'],
      'contact_type_b' => ['organization'],
    ];
    $relationship_type = RelationshipType::create($values + $defaults);
    $relationship_type->save();
    return $relationship_type;
  }

  /**
   * Creates a contact for testing.
   *
   * @param string $bundle
   *   The contact bundle.
   * @param string $name
   *   The contact name.
   *
   * @return \Drupal\crm\Entity\Contact
   *   The created contact.
   */
  protected function createContact(string $bundle, string $name): Contact {
    $contact = Contact::create([
      'bundle' => $bundle,
      'name' => $name,
      'status' => TRUE,
    ]);
    $contact->save();
    return $contact;
  }

  /**
   * Tests form creation and dependency injection.
   */
  public function testFormCreation(): void {
    $this->assertInstanceOf(RelationshipForm::class, $this->form);
  }

  /**
   * Tests building the form with a new relationship.
   */
  public function testBuildFormWithNewRelationship(): void {
    $relationship_type = $this->createRelationshipType('test_new_form');

    $relationship = Relationship::create([
      'bundle' => 'test_new_form',
    ]);

    $this->form->setEntity($relationship);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    $this->assertIsArray($form);

    // Check for contact fields.
    $this->assertArrayHasKey('contact_a', $form);
    $this->assertArrayHasKey('contact_b', $form);

    // Check for meta section.
    $this->assertArrayHasKey('meta', $form);
    $this->assertEquals('details', $form['meta']['#type']);

    // Check that status is moved to meta.
    $this->assertArrayHasKey('status', $form['meta']);
  }

  /**
   * Tests contact type restrictions are applied to the form.
   */
  public function testContactTypeRestrictions(): void {
    $relationship_type = RelationshipType::create([
      'id' => 'test_type_restrictions',
      'label' => 'Test Type Restrictions',
      'asymmetric' => TRUE,
      'label_a' => 'Employee',
      'label_b' => 'Company',
      'contact_type_a' => ['person'],
      'contact_type_b' => ['organization'],
    ]);
    $relationship_type->save();

    $relationship = Relationship::create([
      'bundle' => 'test_type_restrictions',
    ]);

    $this->form->setEntity($relationship);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    // Check that target_bundles are set correctly for contact_a.
    $target_bundles_a = $form['contact_a']['widget'][0]['target_id']['#selection_settings']['target_bundles'];
    $this->assertEquals(['person' => 'person'], $target_bundles_a);

    // Check that target_bundles are set correctly for contact_b.
    $target_bundles_b = $form['contact_b']['widget'][0]['target_id']['#selection_settings']['target_bundles'];
    $this->assertEquals(['organization' => 'organization'], $target_bundles_b);

    // Check that custom labels are applied.
    $this->assertEquals('Employee', $form['contact_a']['widget'][0]['target_id']['#title']);
    $this->assertEquals('Company', $form['contact_b']['widget'][0]['target_id']['#title']);
  }

  /**
   * Tests contact type restrictions with multiple types.
   */
  public function testContactTypeRestrictionsMultipleTypes(): void {
    $relationship_type = RelationshipType::create([
      'id' => 'test_multi_types',
      'label' => 'Test Multiple Types',
      'asymmetric' => TRUE,
      'label_a' => 'Contact A',
      'label_b' => 'Contact B',
      'contact_type_a' => ['person', 'household'],
      'contact_type_b' => ['organization', 'household'],
    ]);
    $relationship_type->save();

    $relationship = Relationship::create([
      'bundle' => 'test_multi_types',
    ]);

    $this->form->setEntity($relationship);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    // Check that multiple target_bundles are set correctly for contact_a.
    $target_bundles_a = $form['contact_a']['widget'][0]['target_id']['#selection_settings']['target_bundles'];
    $this->assertEquals(['person' => 'person', 'household' => 'household'], $target_bundles_a);

    // Check that multiple target_bundles are set correctly for contact_b.
    $target_bundles_b = $form['contact_b']['widget'][0]['target_id']['#selection_settings']['target_bundles'];
    $this->assertEquals(['organization' => 'organization', 'household' => 'household'], $target_bundles_b);
  }

  /**
   * Tests valid contact constraints with a single valid contact.
   */
  public function testApplyValidContactConstraintsSingleContact(): void {
    // Create a single valid contact.
    $valid_contact = $this->createContact('person', 'Single Valid Contact');

    $relationship_type = RelationshipType::create([
      'id' => 'test_single_valid',
      'label' => 'Test Single Valid',
      'asymmetric' => TRUE,
      'label_a' => 'Contact A',
      'label_b' => 'Contact B',
      'contact_type_a' => ['person'],
      'contact_type_b' => ['organization'],
      'valid_contacts_a' => [$valid_contact->id()],
    ]);
    $relationship_type->save();

    $relationship = Relationship::create([
      'bundle' => 'test_single_valid',
    ]);

    $this->form->setEntity($relationship);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    // Check that the contact is auto-selected as default value.
    $default_value = $form['contact_a']['widget'][0]['target_id']['#default_value'];
    $this->assertEquals($valid_contact->id(), $default_value->id());

    // Check that the field is disabled.
    $this->assertTrue($form['contact_a']['widget']['#disabled']);
  }

  /**
   * Tests valid contact constraints with multiple valid contacts.
   */
  public function testApplyValidContactConstraintsMultipleContacts(): void {
    // Create multiple valid contacts.
    $valid_contact_1 = $this->createContact('person', 'Valid Contact 1');
    $valid_contact_2 = $this->createContact('person', 'Valid Contact 2');

    $relationship_type = RelationshipType::create([
      'id' => 'test_multi_valid',
      'label' => 'Test Multiple Valid',
      'asymmetric' => TRUE,
      'label_a' => 'Contact A',
      'label_b' => 'Contact B',
      'contact_type_a' => ['person'],
      'contact_type_b' => ['organization'],
      'valid_contacts_a' => [$valid_contact_1->id(), $valid_contact_2->id()],
    ]);
    $relationship_type->save();

    $relationship = Relationship::create([
      'bundle' => 'test_multi_valid',
    ]);

    $this->form->setEntity($relationship);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    // Check that the selection handler is set for valid contacts.
    $this->assertEquals('valid_contacts:crm_contact', $form['contact_a']['widget'][0]['target_id']['#selection_handler']);

    // Check that valid_contact_ids are passed (IDs may be integers or strings).
    $valid_ids = $form['contact_a']['widget'][0]['target_id']['#selection_settings']['valid_contact_ids'];
    $this->assertContains((int) $valid_contact_1->id(), array_map('intval', $valid_ids));
    $this->assertContains((int) $valid_contact_2->id(), array_map('intval', $valid_ids));
  }

  /**
   * Tests readonly contact fields on existing relationships.
   */
  public function testReadonlyContactFields(): void {
    $contact_a = $this->createContact('person', 'Person Contact');
    $contact_b = $this->createContact('organization', 'Org Contact');

    $relationship_type = RelationshipType::create([
      'id' => 'test_readonly',
      'label' => 'Test Readonly',
      'asymmetric' => TRUE,
      'label_a' => 'Contact A',
      'label_b' => 'Contact B',
      'contact_type_a' => ['person'],
      'contact_type_b' => ['organization'],
      'readonly_contact_a' => TRUE,
      'readonly_contact_b' => TRUE,
    ]);
    $relationship_type->save();

    // Create and save a relationship.
    $relationship = Relationship::create([
      'bundle' => 'test_readonly',
      'contact_a' => $contact_a->id(),
      'contact_b' => $contact_b->id(),
      'status' => TRUE,
    ]);
    $relationship->save();

    // Reload the relationship.
    $relationship = $this->relationshipStorage->load($relationship->id());

    $this->form->setEntity($relationship);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    // Check that contact fields are hidden for readonly contacts.
    $this->assertFalse($form['contact_a']['#access']);
    $this->assertFalse($form['contact_b']['#access']);
  }

  /**
   * Tests readonly setting only applies when contact is set.
   */
  public function testReadonlyOnlyWhenContactIsSet(): void {
    $relationship_type = RelationshipType::create([
      'id' => 'test_readonly_empty',
      'label' => 'Test Readonly Empty',
      'asymmetric' => TRUE,
      'label_a' => 'Contact A',
      'label_b' => 'Contact B',
      'contact_type_a' => ['person'],
      'contact_type_b' => ['organization'],
      'readonly_contact_a' => TRUE,
      'readonly_contact_b' => TRUE,
    ]);
    $relationship_type->save();

    // Create a new relationship without contacts.
    $relationship = Relationship::create([
      'bundle' => 'test_readonly_empty',
    ]);

    $this->form->setEntity($relationship);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    // New relationships should still have accessible contact fields.
    $this->assertTrue($form['contact_a']['#access'] ?? TRUE);
    $this->assertTrue($form['contact_b']['#access'] ?? TRUE);
  }

  /**
   * Tests saving a new relationship.
   */
  public function testSaveNewRelationship(): void {
    $contact_a = $this->createContact('person', 'Test Person');
    $contact_b = $this->createContact('organization', 'Test Org');

    $relationship_type = $this->createRelationshipType('test_save_new');

    $relationship = Relationship::create([
      'bundle' => 'test_save_new',
      'contact_a' => $contact_a->id(),
      'contact_b' => $contact_b->id(),
      'status' => TRUE,
    ]);

    $this->form->setEntity($relationship);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    // Save the entity directly (form submission is tested in functional tests).
    $result = $this->form->save($form, $form_state);

    // Verify the entity was saved.
    $this->assertEquals(SAVED_NEW, $result);

    // Verify the entity exists in the database.
    $saved_relationship = $this->relationshipStorage->load($relationship->id());
    $this->assertNotNull($saved_relationship);
  }

  /**
   * Tests saving an updated relationship.
   */
  public function testSaveUpdatedRelationship(): void {
    $contact_a = $this->createContact('person', 'Update Test Person');
    $contact_b = $this->createContact('organization', 'Update Test Org');

    $relationship_type = $this->createRelationshipType('test_save_update');

    // Create and save a relationship first.
    $relationship = Relationship::create([
      'bundle' => 'test_save_update',
      'contact_a' => $contact_a->id(),
      'contact_b' => $contact_b->id(),
      'status' => TRUE,
    ]);
    $relationship->save();

    // Reload the relationship.
    $relationship = $this->relationshipStorage->load($relationship->id());
    $relationship->set('status', FALSE);

    $this->form->setEntity($relationship);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    // Save the entity directly (form submission is tested in functional tests).
    $result = $this->form->save($form, $form_state);

    // Verify the entity was updated.
    $this->assertEquals(SAVED_UPDATED, $result);

    // Verify a redirect was set.
    $redirect = $form_state->getRedirect();
    $this->assertNotNull($redirect);
  }

  /**
   * Tests form structure includes advanced section.
   */
  public function testFormAdvancedSection(): void {
    $relationship_type = $this->createRelationshipType('test_advanced');

    $relationship = Relationship::create([
      'bundle' => 'test_advanced',
    ]);

    $this->form->setEntity($relationship);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    // Check advanced section has the entity-meta class.
    $this->assertArrayHasKey('advanced', $form);
    $this->assertContains('entity-meta', $form['advanced']['#attributes']['class']);
  }

  /**
   * Tests that uid field is removed from the form.
   */
  public function testUidFieldRemoved(): void {
    $relationship_type = $this->createRelationshipType('test_uid_removed');

    $relationship = Relationship::create([
      'bundle' => 'test_uid_removed',
    ]);

    $this->form->setEntity($relationship);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    // Check that uid field is removed.
    $this->assertArrayNotHasKey('uid', $form);
  }

  /**
   * Tests building form with existing relationship shows correct status.
   */
  public function testBuildFormWithExistingRelationship(): void {
    $contact_a = $this->createContact('person', 'Existing Test Person');
    $contact_b = $this->createContact('organization', 'Existing Test Org');

    $relationship_type = $this->createRelationshipType('test_existing');

    $relationship = Relationship::create([
      'bundle' => 'test_existing',
      'contact_a' => $contact_a->id(),
      'contact_b' => $contact_b->id(),
      'status' => TRUE,
    ]);
    $relationship->save();

    // Reload the relationship.
    $relationship = $this->relationshipStorage->load($relationship->id());

    $this->form->setEntity($relationship);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    // Check that meta section shows correct status.
    $this->assertArrayHasKey('meta', $form);
    $this->assertArrayHasKey('published', $form['meta']);

    // Check that changed time is displayed.
    $this->assertArrayHasKey('changed', $form['meta']);
  }

}
