<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Kernel\Form;

use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;
use Drupal\crm\Form\RelationshipTypeForm;
use Drupal\crm\Entity\Contact;
use Drupal\crm\Entity\Relationship;
use Drupal\crm\Entity\RelationshipType;

/**
 * Kernel tests for the RelationshipTypeForm::contactTypeInUse() method.
 *
 * @group crm
 * @covers \Drupal\crm\Form\RelationshipTypeForm::contactTypeInUse
 */
class RelationshipTypeFormContactTypeInUseTest extends EntityKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'crm',
    'datetime',
    'inline_entity_form',
    'primary_entity_reference',
    'name',
    'telephone',
    'address',
  ];

  /**
   * The form under test.
   *
   * @var \Drupal\crm\Form\RelationshipTypeForm
   */
  protected $form;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The reflection method for contactTypeInUse.
   *
   * @var \ReflectionMethod
   */
  protected $contactTypeInUseMethod;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('crm_contact');
    $this->installEntitySchema('crm_contact_detail');
    $this->installEntitySchema('crm_relationship');
    $this->installConfig(['crm', 'name']);

    $this->entityTypeManager = $this->container->get('entity_type.manager');

    $this->form = RelationshipTypeForm::create($this->container);
    $this->form->setModuleHandler($this->container->get('module_handler'));
    $this->form->setEntityTypeManager($this->entityTypeManager);

    // Set up reflection to access the protected method.
    $reflection = new \ReflectionClass(RelationshipTypeForm::class);
    $this->contactTypeInUseMethod = $reflection->getMethod('contactTypeInUse');
    $this->contactTypeInUseMethod->setAccessible(TRUE);
  }

  /**
   * Creates a relationship type for testing.
   *
   * @param string $id
   *   The relationship type ID.
   * @param array $values
   *   Additional values.
   *
   * @return \Drupal\crm\Entity\RelationshipType
   *   The created relationship type.
   */
  protected function createRelationshipType(string $id, array $values = []): RelationshipType {
    $defaults = [
      'id' => $id,
      'label' => ucfirst(str_replace('_', ' ', $id)),
      'asymmetric' => TRUE,
      'label_a' => 'Contact A',
      'label_b' => 'Contact B',
      'contact_type_a' => ['person', 'organization'],
      'contact_type_b' => ['person', 'organization'],
    ];
    $relationship_type = RelationshipType::create($values + $defaults);
    $relationship_type->save();
    return $relationship_type;
  }

  /**
   * Creates a contact for testing.
   *
   * @param string $bundle
   *   The contact bundle.
   * @param string $name
   *   The contact name.
   *
   * @return \Drupal\crm\Entity\Contact
   *   The created contact.
   */
  protected function createContact(string $bundle, string $name): Contact {
    $contact = Contact::create([
      'bundle' => $bundle,
      'name' => $name,
      'status' => TRUE,
    ]);
    $contact->save();
    return $contact;
  }

  /**
   * Invokes the contactTypeInUse method on the form.
   *
   * @param string $contact_type
   *   The contact type to check.
   * @param int $delta
   *   The delta (0 for contact_a, 1 for contact_b).
   *
   * @return bool
   *   TRUE if the contact type is in use.
   */
  protected function invokeContactTypeInUse(string $contact_type, int $delta): bool {
    return $this->contactTypeInUseMethod->invoke($this->form, $contact_type, $delta);
  }

  /**
   * Tests contactTypeInUse returns FALSE when no relationships exist.
   */
  public function testContactTypeInUseWithNoRelationships(): void {
    $relationship_type = $this->createRelationshipType('test_no_relationships');

    $this->form->setEntity($relationship_type);

    // Should return FALSE for any contact type when no relationships exist.
    $result = $this->invokeContactTypeInUse('person', 0);
    $this->assertFalse($result, 'Should return FALSE when no relationships exist for delta 0.');

    $result = $this->invokeContactTypeInUse('person', 1);
    $this->assertFalse($result, 'Should return FALSE when no relationships exist for delta 1.');

    $result = $this->invokeContactTypeInUse('organization', 0);
    $this->assertFalse($result, 'Should return FALSE for organization at delta 0.');

    $result = $this->invokeContactTypeInUse('organization', 1);
    $this->assertFalse($result, 'Should return FALSE for organization at delta 1.');
  }

  /**
   * Tests contactTypeInUse returns TRUE when contact type is used at delta 0.
   */
  public function testContactTypeInUseAtDelta0(): void {
    $relationship_type = $this->createRelationshipType('test_delta_0');

    // Create contacts.
    $person = $this->createContact('person', 'Test Person');
    $organization = $this->createContact('organization', 'Test Organization');

    // Create a relationship with person at delta 0 (contact_a).
    $relationship = Relationship::create([
      'bundle' => 'test_delta_0',
      'contacts' => [
        ['target_id' => $person->id()],
        ['target_id' => $organization->id()],
      ],
    ]);
    $relationship->save();

    $this->form->setEntity($relationship_type);

    // Should return TRUE for 'person' at delta 0.
    $result = $this->invokeContactTypeInUse('person', 0);
    $this->assertTrue($result, 'Should return TRUE when person is used at delta 0.');

    // Should return FALSE for 'organization' at delta 0.
    $result = $this->invokeContactTypeInUse('organization', 0);
    $this->assertFalse($result, 'Should return FALSE when organization is not used at delta 0.');
  }

  /**
   * Tests contactTypeInUse returns TRUE when contact type is used at delta 1.
   */
  public function testContactTypeInUseAtDelta1(): void {
    $relationship_type = $this->createRelationshipType('test_delta_1');

    // Create contacts.
    $person = $this->createContact('person', 'Test Person');
    $organization = $this->createContact('organization', 'Test Organization');

    // Create a relationship with organization at delta 1 (contact_b).
    $relationship = Relationship::create([
      'bundle' => 'test_delta_1',
      'contacts' => [
        ['target_id' => $person->id()],
        ['target_id' => $organization->id()],
      ],
    ]);
    $relationship->save();

    $this->form->setEntity($relationship_type);

    // Should return TRUE for 'organization' at delta 1.
    $result = $this->invokeContactTypeInUse('organization', 1);
    $this->assertTrue($result, 'Should return TRUE when organization is used at delta 1.');

    // Should return FALSE for 'person' at delta 1.
    $result = $this->invokeContactTypeInUse('person', 1);
    $this->assertFalse($result, 'Should return FALSE when person is not used at delta 1.');
  }

  /**
   * Test contactTypeInUse returns FALSE when type is not used at queried delta.
   */
  public function testContactTypeNotInUse(): void {
    $relationship_type = $this->createRelationshipType('test_not_in_use');

    // Create contacts of same type for both positions.
    $person_a = $this->createContact('person', 'Person A');
    $person_b = $this->createContact('person', 'Person B');

    // Create a relationship with persons at both deltas.
    $relationship = Relationship::create([
      'bundle' => 'test_not_in_use',
      'contacts' => [
        ['target_id' => $person_a->id()],
        ['target_id' => $person_b->id()],
      ],
    ]);
    $relationship->save();

    $this->form->setEntity($relationship_type);

    // Should return FALSE for 'organization' at both deltas.
    $result = $this->invokeContactTypeInUse('organization', 0);
    $this->assertFalse($result, 'Should return FALSE when organization is not used at delta 0.');

    $result = $this->invokeContactTypeInUse('organization', 1);
    $this->assertFalse($result, 'Should return FALSE when organization is not used at delta 1.');

    // Should return FALSE for 'household' which is not used at all.
    $result = $this->invokeContactTypeInUse('household', 0);
    $this->assertFalse($result, 'Should return FALSE for household at delta 0.');

    $result = $this->invokeContactTypeInUse('household', 1);
    $this->assertFalse($result, 'Should return FALSE for household at delta 1.');
  }

  /**
   * Tests contactTypeInUse with multiple relationships, some using the type.
   */
  public function testContactTypeInUseWithMixedRelationships(): void {
    $relationship_type = $this->createRelationshipType('test_mixed');

    // Create contacts.
    $person_1 = $this->createContact('person', 'Person 1');
    $person_2 = $this->createContact('person', 'Person 2');
    $person_3 = $this->createContact('person', 'Person 3');
    $organization = $this->createContact('organization', 'Test Organization');

    // Create a relationship with persons at both deltas.
    $relationship_1 = Relationship::create([
      'bundle' => 'test_mixed',
      'contacts' => [
        ['target_id' => $person_1->id()],
        ['target_id' => $person_2->id()],
      ],
    ]);
    $relationship_1->save();

    // Create a relationship with person at delta 0 and organization at delta 1.
    $relationship_2 = Relationship::create([
      'bundle' => 'test_mixed',
      'contacts' => [
        ['target_id' => $person_3->id()],
        ['target_id' => $organization->id()],
      ],
    ]);
    $relationship_2->save();

    $this->form->setEntity($relationship_type);

    // Should return TRUE for 'person' at delta 0 (used in both relationships).
    $result = $this->invokeContactTypeInUse('person', 0);
    $this->assertTrue($result, 'Should return TRUE when person is used at delta 0 in any relationship.');

    // Should return TRUE for 'person' at delta 1 (used in first relationship).
    $result = $this->invokeContactTypeInUse('person', 1);
    $this->assertTrue($result, 'Should return TRUE when person is used at delta 1 in any relationship.');

    // Should return TRUE for 'organization' at delta 1.
    $result = $this->invokeContactTypeInUse('organization', 1);
    $this->assertTrue($result, 'Should return TRUE when organization is used at delta 1.');

    // Should return FALSE for 'organization' at delta 0 (not used there).
    $result = $this->invokeContactTypeInUse('organization', 0);
    $this->assertFalse($result, 'Should return FALSE when organization is not used at delta 0.');
  }

  /**
   * Test contactTypeInUse with relationships from different relationship types.
   */
  public function testContactTypeInUseOnlyChecksOwnRelationships(): void {
    // Create two relationship types.
    $relationship_type_1 = $this->createRelationshipType('test_type_1');
    $relationship_type_2 = $this->createRelationshipType('test_type_2');

    // Create contacts.
    $person = $this->createContact('person', 'Test Person');
    $organization = $this->createContact('organization', 'Test Organization');

    // Create a relationship for type_2 with organization at delta 0.
    $relationship = Relationship::create([
      'bundle' => 'test_type_2',
      'contacts' => [
        ['target_id' => $organization->id()],
        ['target_id' => $person->id()],
      ],
    ]);
    $relationship->save();

    // Set the form entity to type_1 (which has no relationships).
    $this->form->setEntity($relationship_type_1);

    // Should return FALSE because type_1 has no relationships.
    $result = $this->invokeContactTypeInUse('organization', 0);
    $this->assertFalse($result, 'Should return FALSE when checking type_1 which has no relationships.');

    // Now set the form entity to type_2.
    $this->form->setEntity($relationship_type_2);

    // Should return TRUE has a relationship with organization at delta 0.
    $result = $this->invokeContactTypeInUse('organization', 0);
    $this->assertTrue($result, 'Should return TRUE when checking type_2 which has a relationship.');
  }

  /**
   * Tests contactTypeInUse with household contact type.
   */
  public function testContactTypeInUseWithHousehold(): void {
    $relationship_type = $this->createRelationshipType('test_household', [
      'contact_type_a' => ['household'],
      'contact_type_b' => ['person'],
    ]);

    // Create contacts.
    $household = $this->createContact('household', 'Test Household');
    $person = $this->createContact('person', 'Test Person');

    // Create a relationship with household at delta 0.
    $relationship = Relationship::create([
      'bundle' => 'test_household',
      'contacts' => [
        ['target_id' => $household->id()],
        ['target_id' => $person->id()],
      ],
    ]);
    $relationship->save();

    $this->form->setEntity($relationship_type);

    // Should return TRUE for 'household' at delta 0.
    $result = $this->invokeContactTypeInUse('household', 0);
    $this->assertTrue($result, 'Should return TRUE when household is used at delta 0.');

    // Should return FALSE for 'household' at delta 1.
    $result = $this->invokeContactTypeInUse('household', 1);
    $this->assertFalse($result, 'Should return FALSE when household is not used at delta 1.');
  }

  /**
   * Tests contactTypeInUse with many relationships for performance.
   */
  public function testContactTypeInUseWithManyRelationships(): void {
    $relationship_type = $this->createRelationshipType('test_many');

    // Create many relationships with persons at both positions.
    $contacts = [];
    for ($i = 0; $i < 10; $i++) {
      $contacts[] = $this->createContact('person', 'Person ' . $i);
    }

    // Create 5 relationships.
    for ($i = 0; $i < 5; $i++) {
      $relationship = Relationship::create([
        'bundle' => 'test_many',
        'contacts' => [
          ['target_id' => $contacts[$i * 2]->id()],
          ['target_id' => $contacts[$i * 2 + 1]->id()],
        ],
      ]);
      $relationship->save();
    }

    $this->form->setEntity($relationship_type);

    // Should return TRUE for person at both deltas.
    $result = $this->invokeContactTypeInUse('person', 0);
    $this->assertTrue($result, 'Should return TRUE when person is used at delta 0.');

    $result = $this->invokeContactTypeInUse('person', 1);
    $this->assertTrue($result, 'Should return TRUE when person is used at delta 1.');

    // Should return FALSE for organization at both deltas.
    $result = $this->invokeContactTypeInUse('organization', 0);
    $this->assertFalse($result, 'Should return FALSE when organization is not used at delta 0.');
  }

}
