<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Kernel\Form;

use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;
use Drupal\crm\Form\RelationshipTypeForm;
use Drupal\crm\Entity\RelationshipType;
use Drupal\Core\Form\FormState;

/**
 * Kernel tests for the RelationshipTypeForm.
 *
 * @group crm
 * @covers \Drupal\crm\Form\RelationshipTypeForm
 */
class RelationshipTypeFormTest extends EntityKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'crm',
    'datetime',
    'inline_entity_form',
    'primary_entity_reference',
    'name',
    'telephone',
    'address',
  ];

  /**
   * The form under test.
   *
   * @var \Drupal\crm\Form\RelationshipTypeForm
   */
  protected $form;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The relationship type storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $relationshipTypeStorage;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('crm_contact');
    $this->installEntitySchema('crm_contact_detail');
    $this->installEntitySchema('crm_relationship');
    $this->installEntitySchema('crm_relationship_type');
    $this->installConfig(['crm']);

    $this->entityTypeManager = $this->container->get('entity_type.manager');
    $this->relationshipTypeStorage = $this->entityTypeManager->getStorage('crm_relationship_type');

    $this->form = RelationshipTypeForm::create($this->container);
    $this->form->setModuleHandler($this->container->get('module_handler'));
    $this->form->setEntityTypeManager($this->entityTypeManager);
  }

  /**
   * Tests form creation and dependency injection.
   */
  public function testFormCreation(): void {
    $this->assertInstanceOf(RelationshipTypeForm::class, $this->form);
  }

  /**
   * Tests building the form with a new relationship type.
   */
  public function testBuildFormWithNewRelationshipType(): void {
    $relationship_type = RelationshipType::create([
      'id' => 'kernel_test_type',
      'label' => 'Test Relationship Type',
      'status' => TRUE,
    ]);

    $this->form->setEntity($relationship_type);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    $this->assertIsArray($form);
    $this->assertArrayHasKey('label', $form);
    $this->assertArrayHasKey('id', $form);
    $this->assertArrayHasKey('description', $form);
    $this->assertArrayHasKey('asymmetric', $form);
    $this->assertArrayHasKey('label_a', $form);
    $this->assertArrayHasKey('label_a_plural', $form);
    $this->assertArrayHasKey('contact_type_a', $form);
    $this->assertArrayHasKey('label_b', $form);
    $this->assertArrayHasKey('label_b_plural', $form);
    $this->assertArrayHasKey('contact_type_b', $form);

    // Test form field properties.
    $this->assertEquals('textfield', $form['label']['#type']);
    $this->assertEquals('Test Relationship Type', $form['label']['#default_value']);
    $this->assertTrue($form['label']['#required']);

    $this->assertEquals('machine_name', $form['id']['#type']);
    $this->assertEquals('kernel_test_type', $form['id']['#default_value']);

    $this->assertEquals('textarea', $form['description']['#type']);
    $this->assertEquals('', $form['description']['#default_value']);

    $this->assertEquals('checkbox', $form['asymmetric']['#type']);
    // Default asymmetric should be TRUE for new entities.
    $this->assertEquals(1, $form['asymmetric']['#default_value']);

    $this->assertEquals('textfield', $form['label_a']['#type']);
    $this->assertTrue($form['label_a']['#required']);

    $this->assertEquals('textfield', $form['label_a_plural']['#type']);
    $this->assertArrayNotHasKey('#required', $form['label_a_plural']);

    $this->assertEquals('select', $form['contact_type_a']['#type']);
    $this->assertTrue($form['contact_type_a']['#required']);

    $this->assertEquals('textfield', $form['label_b']['#type']);
    $this->assertTrue($form['label_b']['#required']);
    $this->assertArrayHasKey('#states', $form['label_b']);

    $this->assertEquals('textfield', $form['label_b_plural']['#type']);
    $this->assertArrayNotHasKey('#required', $form['label_b_plural']);
    $this->assertArrayHasKey('#states', $form['label_b_plural']);

    $this->assertEquals('select', $form['contact_type_b']['#type']);
    $this->assertTrue($form['contact_type_b']['#required']);
    $this->assertArrayHasKey('#states', $form['contact_type_b']);

    // Test contact type options.
    $expected_options = [
      'person' => 'Person',
      'household' => 'Household',
      'organization' => 'Organization',
    ];
    $this->assertEquals($expected_options, $form['contact_type_a']['#options']);
    $this->assertEquals($expected_options, $form['contact_type_b']['#options']);
  }

  /**
   * Tests building the form with an existing relationship type.
   */
  public function testBuildFormWithExistingRelationshipType(): void {
    // Create and save a relationship type.
    $relationship_type = RelationshipType::create([
      'id' => 'kernel_existing_test',
      'label' => 'Parent-Child',
      'description' => 'Parent to child relationship',
      'asymmetric' => TRUE,
      'label_a' => 'Parent',
      'label_a_plural' => 'Parents',
      'contact_type_a' => ['person'],
      'label_b' => 'Child',
      'label_b_plural' => 'Children',
      'contact_type_b' => ['person'],
    ]);
    $relationship_type->save();

    $this->form->setEntity($relationship_type);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    // Test that existing values are properly loaded.
    $this->assertEquals('Parent-Child', $form['label']['#default_value']);
    $this->assertEquals('kernel_existing_test', $form['id']['#default_value']);
    $this->assertEquals('Parent to child relationship', $form['description']['#default_value']);
    $this->assertTrue($form['asymmetric']['#default_value']);
    $this->assertEquals('Parent', $form['label_a']['#default_value']);
    $this->assertEquals('Parents', $form['label_a_plural']['#default_value']);
    $this->assertEquals(['person'], $form['contact_type_a']['#default_value']);
    $this->assertEquals('Child', $form['label_b']['#default_value']);
    $this->assertEquals('Children', $form['label_b_plural']['#default_value']);
    $this->assertEquals(['person'], $form['contact_type_b']['#default_value']);
  }

  /**
   * Tests submitting the form to create a new asymmetric relationship type.
   */
  public function testSubmitFormCreateAsymmetricRelationshipType(): void {
    $relationship_type = RelationshipType::create([
      'id' => 'kernel_employee_test',
      'label' => 'Employee',
      'status' => TRUE,
    ]);

    $this->form->setEntity($relationship_type);

    $form_state = new FormState();
    $form_state->setValues([
      'label' => 'Employee',
      'id' => 'kernel_employee_test',
      'description' => 'Employee to organization relationship',
      'asymmetric' => TRUE,
      'label_a' => 'Employee',
      'label_a_plural' => 'Employees',
      'contact_type_a' => ['person'],
      'label_b' => 'Employer',
      'label_b_plural' => 'Employers',
      'contact_type_b' => ['organization'],
    ]);

    $form = $this->form->buildForm([], $form_state);
    $this->form->submitForm($form, $form_state);
    $this->form->save($form, $form_state);

    // Verify the entity was saved.
    $saved_relationship_type = $this->relationshipTypeStorage->load('kernel_employee_test');
    $this->assertNotNull($saved_relationship_type);
    $this->assertEquals('Employee', $saved_relationship_type->label());
    $this->assertEquals('Employee to organization relationship', $saved_relationship_type->get('description'));
    $this->assertTrue($saved_relationship_type->get('asymmetric'));
    $this->assertEquals('Employee', $saved_relationship_type->get('label_a'));
    $this->assertEquals('Employees', $saved_relationship_type->get('label_a_plural'));
    $this->assertEquals(['person'], $saved_relationship_type->get('contact_type_a'));
    $this->assertEquals('Employer', $saved_relationship_type->get('label_b'));
    $this->assertEquals('Employers', $saved_relationship_type->get('label_b_plural'));
    $this->assertEquals(['organization'], $saved_relationship_type->get('contact_type_b'));
  }

  /**
   * Tests submitting the form to create a symmetric relationship type.
   */
  public function testSubmitFormCreateSymmetricRelationshipType(): void {
    $relationship_type = RelationshipType::create([
      'id' => 'kernel_spouse_test',
      'label' => 'Spouse',
      'status' => TRUE,
    ]);

    $this->form->setEntity($relationship_type);

    $form_state = new FormState();
    $form_state->setValues([
      'label' => 'Spouse',
      'id' => 'kernel_spouse_test',
      'description' => 'Married couple relationship',
      'asymmetric' => FALSE,
      'label_a' => 'Spouse',
      'label_a_plural' => 'Spouses',
      'contact_type_a' => ['person'],
      // These should be automatically set to match label_a and contact_type_a.
      'label_b' => 'Different Label',
      'label_b_plural' => 'Different Labels',
      'contact_type_b' => ['organization'],
    ]);

    $form = $this->form->buildForm([], $form_state);
    $this->form->submitForm($form, $form_state);
    $this->form->save($form, $form_state);

    // Verify the entity was saved with symmetric values.
    $saved_relationship_type = $this->relationshipTypeStorage->load('kernel_spouse_test');
    $this->assertNotNull($saved_relationship_type);
    $this->assertEquals('Spouse', $saved_relationship_type->label());
    $this->assertEquals('Married couple relationship', $saved_relationship_type->get('description'));
    $this->assertFalse($saved_relationship_type->get('asymmetric'));
    $this->assertEquals('Spouse', $saved_relationship_type->get('label_a'));
    $this->assertEquals('Spouses', $saved_relationship_type->get('label_a_plural'));
    $this->assertEquals(['person'], $saved_relationship_type->get('contact_type_a'));
    // These should match label_a and contact_type_a due to symmetric logic.
    $this->assertEquals('Spouse', $saved_relationship_type->get('label_b'));
    $this->assertEquals('Spouses', $saved_relationship_type->get('label_b_plural'));
    $this->assertEquals(['person'], $saved_relationship_type->get('contact_type_b'));
  }

  /**
   * Tests updating an existing relationship type.
   */
  public function testSubmitFormUpdateExistingRelationshipType(): void {
    // Create and save a relationship type.
    $relationship_type = RelationshipType::create([
      'id' => 'kernel_update_test',
      'label' => 'Original Label',
      'description' => 'Original description',
      'asymmetric' => FALSE,
      'label_a' => 'Original A',
      'contact_type_a' => ['person'],
      'label_b' => 'Original B',
      'contact_type_b' => ['household'],
    ]);
    $relationship_type->save();

    $this->form->setEntity($relationship_type);

    $form_state = new FormState();
    $form_state->setValues([
      'label' => 'Updated Label',
      'id' => 'kernel_update_test',
      'description' => 'Updated description',
      'asymmetric' => TRUE,
      'label_a' => 'Updated A',
      'contact_type_a' => ['organization'],
      'label_b' => 'Updated B',
      'contact_type_b' => ['person'],
    ]);

    $form = $this->form->buildForm([], $form_state);
    $this->form->submitForm($form, $form_state);
    $this->form->save($form, $form_state);

    // Verify the entity was updated.
    $updated_relationship_type = $this->relationshipTypeStorage->load('kernel_update_test');
    $this->assertNotNull($updated_relationship_type);
    $this->assertEquals('Updated Label', $updated_relationship_type->label());
    $this->assertEquals('Updated description', $updated_relationship_type->get('description'));
    $this->assertTrue($updated_relationship_type->get('asymmetric'));
    $this->assertEquals('Updated A', $updated_relationship_type->get('label_a'));
    $this->assertEquals(['organization'], $updated_relationship_type->get('contact_type_a'));
    $this->assertEquals('Updated B', $updated_relationship_type->get('label_b'));
    $this->assertEquals(['person'], $updated_relationship_type->get('contact_type_b'));
  }

  /**
   * Tests form validation setup for required fields.
   */
  public function testFormValidationSetup(): void {
    $relationship_type = RelationshipType::create([
      'id' => 'kernel_validation_test',
      'label' => '',
    ]);

    $this->form->setEntity($relationship_type);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    // Test that required fields are properly marked.
    $this->assertTrue($form['label']['#required']);
    $this->assertEquals('textfield', $form['label']['#type']);

    $this->assertTrue($form['label_a']['#required']);
    $this->assertEquals('textfield', $form['label_a']['#type']);

    $this->assertTrue($form['contact_type_a']['#required']);
    $this->assertEquals('select', $form['contact_type_a']['#type']);

    $this->assertTrue($form['label_b']['#required']);
    $this->assertTrue($form['contact_type_b']['#required']);

    // Test that machine name field has proper validation callback.
    $this->assertEquals('machine_name', $form['id']['#type']);
    $this->assertArrayHasKey('exists', $form['id']['#machine_name']);
    $this->assertEquals(['Drupal\crm\Entity\RelationshipType', 'load'], $form['id']['#machine_name']['exists']);
  }

  /**
   * Tests form conditional visibility states.
   */
  public function testFormConditionalStates(): void {
    $relationship_type = RelationshipType::create([
      'id' => 'kernel_states_test',
      'label' => 'States Test',
      'asymmetric' => TRUE,
    ]);

    $this->form->setEntity($relationship_type);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    // Test that label_b, label_b_plural, and contact_type_b have proper
    // visibility states.
    $this->assertArrayHasKey('#states', $form['label_b']);
    $this->assertArrayHasKey('visible', $form['label_b']['#states']);
    $this->assertEquals(['checked' => TRUE], $form['label_b']['#states']['visible'][':input[name="asymmetric"]']);

    $this->assertArrayHasKey('#states', $form['label_b_plural']);
    $this->assertArrayHasKey('visible', $form['label_b_plural']['#states']);
    $this->assertEquals(['checked' => TRUE], $form['label_b_plural']['#states']['visible'][':input[name="asymmetric"]']);

    $this->assertArrayHasKey('#states', $form['contact_type_b']);
    $this->assertArrayHasKey('visible', $form['contact_type_b']['#states']);
    $this->assertEquals(['checked' => TRUE], $form['contact_type_b']['#states']['visible'][':input[name="asymmetric"]']);
  }

  /**
   * Tests the getContactTypeOptions method integration.
   */
  public function testContactTypeOptionsIntegration(): void {
    $relationship_type = RelationshipType::create([
      'id' => 'kernel_options_test',
      'label' => 'Options Test',
    ]);

    $this->form->setEntity($relationship_type);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    // Verify that contact type options are properly loaded from bundle info.
    $expected_options = [
      'person' => 'Person',
      'household' => 'Household',
      'organization' => 'Organization',
    ];
    $this->assertEquals($expected_options, $form['contact_type_a']['#options']);
    $this->assertEquals($expected_options, $form['contact_type_b']['#options']);
  }

  /**
   * Tests form with edit operation title.
   */
  public function testFormWithEditOperation(): void {
    $relationship_type = RelationshipType::create([
      'id' => 'kernel_edit_test',
      'label' => 'Edit Test Type',
      'label_a' => 'Test A',
      'contact_type_a' => ['person'],
      'label_b' => 'Test B',
      'contact_type_b' => ['organization'],
    ]);
    $relationship_type->save();

    $this->form->setEntity($relationship_type);
    $this->form->setOperation('edit');

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    $this->assertArrayHasKey('#title', $form);
    // The title contains HTML markup, so we need to render it to get the final
    // string.
    $title = (string) $form['#title'];
    $this->assertStringContainsString('Edit', $title);
    $this->assertStringContainsString('Edit Test Type', $title);
    $this->assertStringContainsString('crm relationship type', $title);
  }

}
