<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Kernel\Form;

use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;
use Drupal\crm\Form\RelationshipTypeForm;
use Drupal\crm\Entity\RelationshipType;
use Drupal\crm\Entity\Contact;
use Drupal\crm\Entity\Relationship;
use Drupal\Core\Form\FormState;

/**
 * Kernel tests for RelationshipTypeForm contact type validation.
 *
 * @group crm
 * @covers \Drupal\crm\Form\RelationshipTypeForm::validateForm
 */
class RelationshipTypeFormValidationTest extends EntityKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'crm',
    'datetime',
    'inline_entity_form',
    'primary_entity_reference',
    'name',
    'telephone',
    'address',
  ];

  /**
   * The form under test.
   *
   * @var \Drupal\crm\Form\RelationshipTypeForm
   */
  protected $form;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('crm_contact');
    $this->installEntitySchema('crm_contact_detail');
    $this->installEntitySchema('crm_relationship');

    $this->installConfig(['crm', 'name']);

    $this->entityTypeManager = $this->container->get('entity_type.manager');

    $this->form = RelationshipTypeForm::create($this->container);
    $this->form->setModuleHandler($this->container->get('module_handler'));
    $this->form->setEntityTypeManager($this->entityTypeManager);
  }

  /**
   * Tests that new relationship types have no validation errors.
   */
  public function testNewRelationshipTypeHasNoValidationErrors(): void {
    $relationship_type = RelationshipType::create([
      'id' => 'test_new_type',
      'label' => 'Test New Type',
      'asymmetric' => TRUE,
      'label_a' => 'Contact A',
      'contact_type_a' => ['person'],
      'label_b' => 'Contact B',
      'contact_type_b' => ['organization'],
    ]);

    $this->form->setEntity($relationship_type);

    $form_state = new FormState();
    $form_state->setValues([
      'label' => 'Test New Type',
      'id' => 'test_new_type',
      'asymmetric' => TRUE,
      'label_a' => 'Contact A',
      'contact_type_a' => ['person'],
      'label_b' => 'Contact B',
      'contact_type_b' => ['organization'],
    ]);

    $form = $this->form->buildForm([], $form_state);
    $this->form->validateForm($form, $form_state);

    $this->assertFalse($form_state->hasAnyErrors(), 'New relationship type should have no validation errors.');
  }

  /**
   * Tests that removing an unused contact type succeeds.
   */
  public function testRemovingUnusedContactTypeSucceeds(): void {
    // Create a relationship type with multiple contact types.
    $relationship_type = RelationshipType::create([
      'id' => 'test_unused_removal',
      'label' => 'Test Unused Removal',
      'asymmetric' => TRUE,
      'label_a' => 'Contact A',
      'contact_type_a' => ['person', 'organization'],
      'label_b' => 'Contact B',
      'contact_type_b' => ['person', 'organization'],
    ]);
    $relationship_type->save();

    // Create contacts of type 'person' only.
    $person_a = Contact::create([
      'bundle' => 'person',
      'name' => 'Person A',
    ]);
    $person_a->save();

    $person_b = Contact::create([
      'bundle' => 'person',
      'name' => 'Person B',
    ]);
    $person_b->save();

    // Create a relationship using only 'person' contacts.
    $relationship = Relationship::create([
      'bundle' => 'test_unused_removal',
      'contacts' => [
        ['target_id' => $person_a->id()],
        ['target_id' => $person_b->id()],
      ],
    ]);
    $relationship->save();

    // Reload the relationship type to get a fresh copy.
    $relationship_type = $this->entityTypeManager->getStorage('crm_relationship_type')->load('test_unused_removal');

    $this->form->setEntity($relationship_type);

    // Try to remove 'organization' which is not in use.
    $form_state = new FormState();
    $form_state->setValues([
      'label' => 'Test Unused Removal',
      'id' => 'test_unused_removal',
      'asymmetric' => TRUE,
      'label_a' => 'Contact A',
      'contact_type_a' => ['person'],
      'label_b' => 'Contact B',
      'contact_type_b' => ['person'],
    ]);

    $form = $this->form->buildForm([], $form_state);
    $this->form->validateForm($form, $form_state);

    $this->assertFalse($form_state->hasAnyErrors(), 'Removing unused contact type should not cause validation errors.');
  }

  /**
   * Tests that removing a contact type in use by contact_a fails.
   */
  public function testRemovingContactTypeInUseByContact0Fails(): void {
    // Create a relationship type.
    $relationship_type = RelationshipType::create([
      'id' => 'test_contact_a_in_use',
      'label' => 'Test Contact A In Use',
      'asymmetric' => TRUE,
      'label_a' => 'Contact A',
      'contact_type_a' => ['person', 'organization'],
      'label_b' => 'Contact B',
      'contact_type_b' => ['organization'],
    ]);
    $relationship_type->save();

    // Create contacts.
    $person = Contact::create([
      'bundle' => 'person',
      'name' => 'Test Person',
    ]);
    $person->save();

    $organization = Contact::create([
      'bundle' => 'organization',
      'name' => 'Test Organization',
    ]);
    $organization->save();

    // Create a relationship where contact_a is a person.
    $relationship = Relationship::create([
      'bundle' => 'test_contact_a_in_use',
      'contacts' => [
        ['target_id' => $person->id()],
        ['target_id' => $organization->id()],
      ],
    ]);
    $relationship->save();

    // Reload the relationship type.
    $relationship_type = $this->entityTypeManager->getStorage('crm_relationship_type')->load('test_contact_a_in_use');

    $this->form->setEntity($relationship_type);

    // Try to remove 'person' from contact_type_a which is in use.
    $form_state = new FormState();
    $form_state->setValues([
      'label' => 'Test Contact A In Use',
      'id' => 'test_contact_a_in_use',
      'asymmetric' => TRUE,
      'label_a' => 'Contact A',
      'contact_type_a' => ['organization'],
      'label_b' => 'Contact B',
      'contact_type_b' => ['organization'],
    ]);

    $form = $this->form->buildForm([], $form_state);
    $this->form->validateForm($form, $form_state);

    $this->assertTrue($form_state->hasAnyErrors(), 'Removing contact type in use by contact_a should cause validation error.');

    $errors = $form_state->getErrors();
    $error_messages = array_map('strval', $errors);
    $this->assertStringContainsString('Cannot remove contact type', implode(' ', $error_messages));
    $this->assertStringContainsString('person', implode(' ', $error_messages));
    $this->assertStringContainsString('Contact A', implode(' ', $error_messages));
  }

  /**
   * Tests that removing a contact type in use by contact_b fails.
   */
  public function testRemovingContactTypeInUseByContact1Fails(): void {
    // Create a relationship type.
    $relationship_type = RelationshipType::create([
      'id' => 'test_contact_b_in_use',
      'label' => 'Test Contact B In Use',
      'asymmetric' => TRUE,
      'label_a' => 'Contact A',
      'contact_type_a' => ['person'],
      'label_b' => 'Contact B',
      'contact_type_b' => ['person', 'organization'],
    ]);
    $relationship_type->save();

    // Create contacts.
    $person_a = Contact::create([
      'bundle' => 'person',
      'name' => 'Person A',
    ]);
    $person_a->save();

    $person_b = Contact::create([
      'bundle' => 'person',
      'name' => 'Person B',
    ]);
    $person_b->save();

    // Create a relationship where contact_b is a person.
    $relationship = Relationship::create([
      'bundle' => 'test_contact_b_in_use',
      'contacts' => [
        ['target_id' => $person_a->id()],
        ['target_id' => $person_b->id()],
      ],
    ]);
    $relationship->save();

    // Reload the relationship type.
    $relationship_type = $this->entityTypeManager->getStorage('crm_relationship_type')->load('test_contact_b_in_use');

    $this->form->setEntity($relationship_type);

    // Try to remove 'person' from contact_type_b which is in use.
    $form_state = new FormState();
    $form_state->setValues([
      'label' => 'Test Contact B In Use',
      'id' => 'test_contact_b_in_use',
      'asymmetric' => TRUE,
      'label_a' => 'Contact A',
      'contact_type_a' => ['person'],
      'label_b' => 'Contact B',
      'contact_type_b' => ['organization'],
    ]);

    $form = $this->form->buildForm([], $form_state);
    $this->form->validateForm($form, $form_state);

    $this->assertTrue($form_state->hasAnyErrors(), 'Removing contact type in use by contact_b should cause validation error.');

    $errors = $form_state->getErrors();
    $error_messages = array_map('strval', $errors);
    $this->assertStringContainsString('Cannot remove contact type', implode(' ', $error_messages));
    $this->assertStringContainsString('person', implode(' ', $error_messages));
    $this->assertStringContainsString('Contact B', implode(' ', $error_messages));
  }

  /**
   * Tests that adding new contact types always succeeds.
   */
  public function testAddingNewContactTypesSucceeds(): void {
    // Create a relationship type with single contact types.
    $relationship_type = RelationshipType::create([
      'id' => 'test_add_types',
      'label' => 'Test Add Types',
      'asymmetric' => TRUE,
      'label_a' => 'Contact A',
      'contact_type_a' => ['person'],
      'label_b' => 'Contact B',
      'contact_type_b' => ['organization'],
    ]);
    $relationship_type->save();

    // Create a relationship.
    $person = Contact::create([
      'bundle' => 'person',
      'name' => 'Test Person',
    ]);
    $person->save();

    $organization = Contact::create([
      'bundle' => 'organization',
      'name' => 'Test Organization',
    ]);
    $organization->save();

    $relationship = Relationship::create([
      'bundle' => 'test_add_types',
      'contacts' => [
        ['target_id' => $person->id()],
        ['target_id' => $organization->id()],
      ],
    ]);
    $relationship->save();

    // Reload the relationship type.
    $relationship_type = $this->entityTypeManager->getStorage('crm_relationship_type')->load('test_add_types');

    $this->form->setEntity($relationship_type);

    // Add more contact types (this should always succeed).
    $form_state = new FormState();
    $form_state->setValues([
      'label' => 'Test Add Types',
      'id' => 'test_add_types',
      'asymmetric' => TRUE,
      'label_a' => 'Contact A',
      'contact_type_a' => ['person', 'organization', 'household'],
      'label_b' => 'Contact B',
      'contact_type_b' => ['organization', 'person', 'household'],
    ]);

    $form = $this->form->buildForm([], $form_state);
    $this->form->validateForm($form, $form_state);

    $this->assertFalse($form_state->hasAnyErrors(), 'Adding new contact types should not cause validation errors.');
  }

  /**
   * Tests validation with no existing relationships.
   */
  public function testRemovingContactTypeWithNoRelationshipsSucceeds(): void {
    // Create a relationship type but no relationships.
    $relationship_type = RelationshipType::create([
      'id' => 'test_no_relationships',
      'label' => 'Test No Relationships',
      'asymmetric' => TRUE,
      'label_a' => 'Contact A',
      'contact_type_a' => ['person', 'organization'],
      'label_b' => 'Contact B',
      'contact_type_b' => ['person', 'organization'],
    ]);
    $relationship_type->save();

    // Reload the relationship type.
    $relationship_type = $this->entityTypeManager->getStorage('crm_relationship_type')->load('test_no_relationships');

    $this->form->setEntity($relationship_type);

    // Remove all types except one (should succeed with no relationships).
    $form_state = new FormState();
    $form_state->setValues([
      'label' => 'Test No Relationships',
      'id' => 'test_no_relationships',
      'asymmetric' => TRUE,
      'label_a' => 'Contact A',
      'contact_type_a' => ['person'],
      'label_b' => 'Contact B',
      'contact_type_b' => ['organization'],
    ]);

    $form = $this->form->buildForm([], $form_state);
    $this->form->validateForm($form, $form_state);

    $this->assertFalse($form_state->hasAnyErrors(), 'Removing contact types with no existing relationships should succeed.');
  }

}
