<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Kernel\Form;

use Drupal\Core\Form\FormState;
use Drupal\crm\Entity\Contact;
use Drupal\crm\Entity\UserContactMapping;
use Drupal\crm\Form\UserContactMappingForm;
use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;
use Drupal\user\Entity\User;

/**
 * Kernel tests for the UserContactMappingForm.
 *
 * @group crm
 * @covers \Drupal\crm\Form\UserContactMappingForm
 */
class UserContactMappingFormTest extends EntityKernelTestBase {

  const SAVED_NEW = 1;
  const SAVED_UPDATED = 2;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'crm',
    'datetime',
    'inline_entity_form',
    'primary_entity_reference',
    'name',
    'telephone',
    'address',
  ];

  /**
   * The form under test.
   *
   * @var \Drupal\crm\Form\UserContactMappingForm
   */
  protected $form;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Test user.
   *
   * @var \Drupal\user\Entity\User
   */
  protected $testUser;

  /**
   * Test contact.
   *
   * @var \Drupal\crm\Entity\Contact
   */
  protected $testContact;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installSchema('user', ['users_data']);
    $this->installEntitySchema('crm_contact');
    $this->installEntitySchema('crm_contact_detail');
    $this->installEntitySchema('crm_user_contact_mapping');
    $this->installConfig(['crm', 'name']);

    $this->entityTypeManager = $this->container->get('entity_type.manager');

    $this->form = UserContactMappingForm::create($this->container);
    $this->form->setModuleHandler($this->container->get('module_handler'));
    $this->form->setEntityTypeManager($this->entityTypeManager);
    $this->form->setStringTranslation($this->container->get('string_translation'));
    $this->form->setMessenger($this->container->get('messenger'));

    // Create test user.
    $this->testUser = User::create([
      'name' => 'test_user',
      'mail' => 'test@example.com',
    ]);
    $this->testUser->save();

    // Create test contact.
    $this->testContact = Contact::create([
      'bundle' => 'person',
      'full_name' => [
        'given' => 'Test',
        'family' => 'User',
      ],
    ]);
    $this->testContact->save();
  }

  /**
   * Tests form creation and dependency injection.
   */
  public function testFormCreation(): void {
    $this->assertInstanceOf(UserContactMappingForm::class, $this->form);
  }

  /**
   * Tests building the form with a new user contact.
   */
  public function testBuildFormWithNewUserContact(): void {
    $user_contact = UserContactMapping::create([
      'user' => $this->testUser->id(),
      'crm_contact' => $this->testContact->id(),
    ]);

    $this->form->setEntity($user_contact);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    $this->assertIsArray($form);
    $this->assertArrayHasKey('user', $form);
    $this->assertArrayHasKey('crm_contact', $form);

    $this->assertArrayNotHasKey('#disabled', $form['user']['widget'] ?? []);
    $this->assertArrayNotHasKey('#disabled', $form['crm_contact']['widget'] ?? []);
  }

  /**
   * Tests building the form with an existing user contact.
   */
  public function testBuildFormWithExistingUserContact(): void {
    $user_contact = UserContactMapping::create([
      'user' => $this->testUser->id(),
      'crm_contact' => $this->testContact->id(),
    ]);
    $user_contact->save();

    $this->form->setEntity($user_contact);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    $this->assertIsArray($form);

    // For existing entities, user field should be disabled.
    $this->assertTrue($form['user']['widget']['#disabled']);
    $this->assertArrayNotHasKey('#disabled', $form['crm_contact']['widget'] ?? []);
  }

  /**
   * Tests saving a new user contact.
   */
  public function testSaveNewUserContact(): void {
    $user_contact = UserContactMapping::create([
      'user' => $this->testUser->id(),
      'crm_contact' => $this->testContact->id(),
    ]);

    $this->form->setEntity($user_contact);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);
    $result = $this->form->save($form, $form_state);

    $this->assertEquals(self::SAVED_NEW, $result);

    // Verify the entity was saved.
    $saved_entity = $this->entityTypeManager
      ->getStorage('crm_user_contact_mapping')
      ->load($user_contact->id());

    $this->assertNotNull($saved_entity);
    $this->assertEquals($this->testUser->id(), $saved_entity->getUserId());
    $this->assertEquals($this->testContact->id(), $saved_entity->getContactId());

    // Verify redirect was set.
    $redirect = $form_state->getRedirect();
    $this->assertEquals('entity.crm_user_contact_mapping.collection', $redirect->getRouteName());
  }

  /**
   * Tests updating an existing user contact.
   */
  public function testSaveExistingUserContact(): void {
    // Create and save initial user contact.
    $user_contact = UserContactMapping::create([
      'user' => $this->testUser->id(),
      'crm_contact' => $this->testContact->id(),
    ]);
    $user_contact->save();

    // Create another contact to update to.
    $another_contact = Contact::create([
      'bundle' => 'person',
      'full_name' => [
        'given' => 'Another',
        'family' => 'Contact',
      ],
    ]);
    $another_contact->save();

    // Update the user contact.
    $user_contact->setContactId($another_contact->id());
    $this->form->setEntity($user_contact);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);
    $result = $this->form->save($form, $form_state);

    $this->assertEquals(self::SAVED_UPDATED, $result);

    // Reload and verify the entity was updated.
    $this->entityTypeManager->getStorage('crm_user_contact_mapping')->resetCache();
    $updated_entity = $this->entityTypeManager
      ->getStorage('crm_user_contact_mapping')
      ->load($user_contact->id());

    $this->assertEquals($another_contact->id(), $updated_entity->getContactId());
    $this->assertEquals($this->testUser->id(), $updated_entity->getUserId());

    // Verify redirect was set.
    $redirect = $form_state->getRedirect();
    $this->assertEquals('entity.crm_user_contact_mapping.collection', $redirect->getRouteName());
  }

  /**
   * Tests form validation through entity constraints.
   */
  public function testFormValidationWithDuplicateMapping(): void {
    // Create an existing mapping.
    $existing_mapping = UserContactMapping::create([
      'user' => $this->testUser->id(),
      'crm_contact' => $this->testContact->id(),
    ]);
    $existing_mapping->save();

    // Try to create another mapping with the same user.
    $duplicate_mapping = UserContactMapping::create([
      'user' => $this->testUser->id(),
      'crm_contact' => $this->testContact->id(),
    ]);

    $this->form->setEntity($duplicate_mapping);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    // The validation should occur at the entity level through constraints.
    // We test that the form can be built without errors.
    $this->assertIsArray($form);
    $this->assertArrayHasKey('user', $form);
    $this->assertArrayHasKey('crm_contact', $form);
  }

  /**
   * Tests form behavior with partial entity data.
   */
  public function testFormWithPartialEntityData(): void {
    // Test with only user set.
    $user_contact_with_user = UserContactMapping::create([
      'user' => $this->testUser->id(),
    ]);

    $this->form->setEntity($user_contact_with_user);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    $this->assertIsArray($form);
    $this->assertArrayHasKey('user', $form);
    $this->assertArrayHasKey('crm_contact', $form);

    // Test with only contact set.
    $user_contact_with_contact = UserContactMapping::create([
      'crm_contact' => $this->testContact->id(),
    ]);

    $this->form->setEntity($user_contact_with_contact);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    $this->assertIsArray($form);
    $this->assertArrayHasKey('user', $form);
    $this->assertArrayHasKey('crm_contact', $form);
  }

}
