<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Kernel\Hook;

use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;

/**
 * Tests InstallOptionalConfigHooks integration.
 *
 * @group crm
 * @coversDefaultClass \Drupal\crm\Hook\InstallOptionalConfigHooks
 */
class InstallOptionalConfigHooksTest extends EntityKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'address',
    'crm',
    'inline_entity_form',
    'primary_entity_reference',
    'datetime',
    'name',
    'telephone',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('crm_contact');
    $this->installEntitySchema('crm_contact_detail');
    $this->installEntitySchema('crm_relationship');
    $this->installEntitySchema('crm_user_contact_mapping');

    $this->installConfig(['crm', 'name']);
  }

  /**
   * Tests that comment configs are installed when comment module is enabled.
   *
   * @covers ::modulesInstalled
   */
  public function testCommentConfigsInstalledAfterCommentModule(): void {
    // Verify comment configs do not exist before installing comment module.
    $config_factory = \Drupal::configFactory();
    $comment_type_config = $config_factory->get('comment.type.crm_contact');
    $this->assertTrue($comment_type_config->isNew(), 'Comment type config should not exist before comment module is installed.');

    $field_storage_config = $config_factory->get('field.storage.crm_contact.comment');
    $this->assertTrue($field_storage_config->isNew(), 'Comment field storage config should not exist before comment module is installed.');

    // Install the comment module (this also installs schemas automatically).
    $this->container->get('module_installer')->install(['comment']);

    // Clear config factory static cache to get fresh config.
    $config_factory = \Drupal::configFactory();

    // Verify comment type config was installed.
    $comment_type_config = $config_factory->get('comment.type.crm_contact');
    $this->assertFalse($comment_type_config->isNew(), 'Comment type config should exist after comment module is installed.');
    $this->assertEquals('crm_contact', $comment_type_config->get('id'));
    $this->assertEquals('Contact Comment', $comment_type_config->get('label'));

    // Verify field storage config was installed.
    $field_storage_config = $config_factory->get('field.storage.crm_contact.comment');
    $this->assertFalse($field_storage_config->isNew(), 'Comment field storage config should exist after comment module is installed.');
    $this->assertEquals('crm_contact.comment', $field_storage_config->get('id'));

    // Verify field instance configs were installed for each bundle.
    $bundles = ['person', 'organization', 'household'];
    foreach ($bundles as $bundle) {
      $field_config = $config_factory->get("field.field.crm_contact.{$bundle}.comment");
      $this->assertFalse($field_config->isNew(), "Comment field config for {$bundle} should exist after comment module is installed.");
      $this->assertEquals("crm_contact.{$bundle}.comment", $field_config->get('id'));
    }
  }

  /**
   * Tests that search configs are installed when search module is enabled.
   *
   * Note: The search.page.crm_search config is owned by the search module
   * (based on config name prefix) and only depends on crm, so it's installed
   * by Drupal's core config installer when search is enabled. The view modes
   * only depend on crm, so they're installed with CRM initially.
   *
   * @covers ::modulesInstalled
   */
  public function testSearchConfigsInstalledAfterSearchModule(): void {
    // Verify search page config does not exist before installing search module.
    // This config is owned by the search module (search.page.*) so it can only
    // be installed when search is enabled.
    $config_factory = \Drupal::configFactory();
    $search_page_config = $config_factory->get('search.page.crm_search');
    $this->assertTrue($search_page_config->isNew(), 'Search page config should not exist before search module is installed.');

    // View modes only depend on crm, so they should already exist.
    $search_index_view_mode = $config_factory->get('core.entity_view_mode.crm_contact.search_index');
    $this->assertFalse($search_index_view_mode->isNew(), 'Search index view mode should exist (only depends on crm).');

    // Install the search module.
    $this->container->get('module_installer')->install(['search']);

    // Clear config factory static cache to get fresh config.
    $config_factory = \Drupal::configFactory();

    // Verify search page config was installed.
    $search_page_config = $config_factory->get('search.page.crm_search');
    $this->assertFalse($search_page_config->isNew(), 'Search page config should exist after search module is installed.');
    $this->assertEquals('crm_search', $search_page_config->get('id'));
    $this->assertEquals('Contacts', $search_page_config->get('label'));
  }

  /**
   * Tests that unrelated modules do not trigger config installation.
   *
   * @covers ::modulesInstalled
   */
  public function testUnrelatedModuleDoesNotTriggerConfigInstall(): void {
    // Verify comment configs do not exist (they require comment module).
    $config_factory = \Drupal::configFactory();
    $comment_type_config = $config_factory->get('comment.type.crm_contact');
    $this->assertTrue($comment_type_config->isNew(), 'Comment type config should not exist.');

    $field_storage_config = $config_factory->get('field.storage.crm_contact.comment');
    $this->assertTrue($field_storage_config->isNew(), 'Comment field storage should not exist.');

    // Verify search page config does not exist (requires search module).
    $search_page_config = $config_factory->get('search.page.crm_search');
    $this->assertTrue($search_page_config->isNew(), 'Search page config should not exist.');

    // Install an unrelated module.
    $this->container->get('module_installer')->install(['views']);

    // Clear config factory static cache.
    $config_factory = \Drupal::configFactory();

    // Verify configs still do not exist.
    $comment_type_config = $config_factory->get('comment.type.crm_contact');
    $this->assertTrue($comment_type_config->isNew(), 'Comment type config should still not exist after installing views.');

    $field_storage_config = $config_factory->get('field.storage.crm_contact.comment');
    $this->assertTrue($field_storage_config->isNew(), 'Comment field storage should still not exist after installing views.');

    $search_page_config = $config_factory->get('search.page.crm_search');
    $this->assertTrue($search_page_config->isNew(), 'Search page config should still not exist after installing views.');
  }

}
