<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Kernel\Hook;

use Drupal\crm\Entity\Contact;
use Drupal\crm\Entity\ContactType;
use Drupal\crm\Entity\Relationship;
use Drupal\crm\Entity\RelationshipType;
use Drupal\KernelTests\KernelTestBase;

/**
 * Tests the relationship statistics hooks.
 *
 * @group crm
 * @coversDefaultClass \Drupal\crm\Hook\RelationshipHooks
 */
class RelationshipStatisticsHooksTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'address',
    'crm',
    'datetime',
    'field',
    'inline_entity_form',
    'primary_entity_reference',
    'name',
    'system',
    'telephone',
    'text',
    'user',
  ];

  /**
   * Relationship types for testing.
   *
   * @var array
   */
  protected array $relationshipTypes = [];

  /**
   * Contact type for testing.
   *
   * @var \Drupal\crm\Entity\ContactType
   */
  protected ContactType $contactType;

  /**
   * Contacts for testing.
   *
   * @var array
   */
  protected array $contacts = [];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('crm_contact');
    $this->installEntitySchema('crm_contact_type');
    $this->installEntitySchema('crm_relationship');
    $this->installEntitySchema('crm_relationship_type');
    $this->installSchema('system', ['sequences']);

    // Create a contact type.
    $this->contactType = ContactType::create([
      'id' => 'person',
      'label' => 'Person',
      'date' => [
        'start_date' => [
          'label' => 'Start date',
        ],
        'end_date' => [
          'label' => 'End date',
        ],
      ],
    ]);
    $this->contactType->save();

    // Create a symmetric relationship type (friends).
    $this->relationshipTypes['friends'] = RelationshipType::create([
      'id' => 'friends',
      'label' => 'Friends',
      'label_a' => 'Friend',
      'contact_type_a' => ['person'],
      'asymmetric' => FALSE,
    ]);
    $this->relationshipTypes['friends']->save();

    // Create an asymmetric relationship type (parent-child).
    $this->relationshipTypes['parent_child'] = RelationshipType::create([
      'id' => 'parent_child',
      'label' => 'Parent-Child',
      'label_a' => 'Parent',
      'label_b' => 'Child',
      'contact_type_a' => ['person'],
      'contact_type_b' => ['person'],
      'asymmetric' => TRUE,
    ]);
    $this->relationshipTypes['parent_child']->save();

    // Create test contacts.
    for ($i = 0; $i < 4; $i++) {
      $this->contacts[$i] = Contact::create([
        'bundle' => 'person',
        'name' => 'Contact ' . $i,
      ]);
      $this->contacts[$i]->save();
    }
  }

  /**
   * Tests that statistics are incremented when a relationship is inserted.
   *
   * @covers ::relationshipInsert
   */
  public function testRelationshipInsertIncrementsStatistics(): void {
    // Create a symmetric relationship.
    $relationship = Relationship::create([
      'bundle' => 'friends',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[1]->id()],
      'status' => TRUE,
    ]);
    $relationship->save();

    // Reload contacts to get updated statistics.
    $contact0 = Contact::load($this->contacts[0]->id());
    $contact1 = Contact::load($this->contacts[1]->id());

    // Both contacts should have 'friends' with count 1.
    $stats0 = $this->getStatisticsArray($contact0);
    $stats1 = $this->getStatisticsArray($contact1);

    $this->assertEquals(1, $stats0['friends'] ?? 0, 'Contact 0 should have 1 friends relationship.');
    $this->assertEquals(1, $stats1['friends'] ?? 0, 'Contact 1 should have 1 friends relationship.');
  }

  /**
   * Tests asymmetric relationship statistics.
   *
   * @covers ::relationshipInsert
   */
  public function testAsymmetricRelationshipStatistics(): void {
    // Create a parent-child relationship.
    $relationship = Relationship::create([
      'bundle' => 'parent_child',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[1]->id()],
      'status' => TRUE,
    ]);
    $relationship->save();

    // Reload contacts.
    $contact0 = Contact::load($this->contacts[0]->id());
    $contact1 = Contact::load($this->contacts[1]->id());

    $stats0 = $this->getStatisticsArray($contact0);
    $stats1 = $this->getStatisticsArray($contact1);

    // Contact 0 is in position A (parent).
    $this->assertEquals(1, $stats0['parent_child:a'] ?? 0, 'Contact 0 should be a parent.');
    $this->assertArrayNotHasKey('parent_child:b', $stats0, 'Contact 0 should not be a child.');

    // Contact 1 is in position B (child).
    $this->assertEquals(1, $stats1['parent_child:b'] ?? 0, 'Contact 1 should be a child.');
    $this->assertArrayNotHasKey('parent_child:a', $stats1, 'Contact 1 should not be a parent.');
  }

  /**
   * Tests that statistics are decremented when a relationship is deleted.
   *
   * @covers ::relationshipDelete
   */
  public function testRelationshipDeleteDecrementsStatistics(): void {
    // Create a relationship.
    $relationship = Relationship::create([
      'bundle' => 'friends',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[1]->id()],
      'status' => TRUE,
    ]);
    $relationship->save();

    // Verify statistics are set.
    $contact0 = Contact::load($this->contacts[0]->id());
    $stats0 = $this->getStatisticsArray($contact0);
    $this->assertEquals(1, $stats0['friends'] ?? 0);

    // Delete the relationship.
    $relationship->delete();

    // Reload contacts.
    $contact0 = Contact::load($this->contacts[0]->id());
    $contact1 = Contact::load($this->contacts[1]->id());

    $stats0 = $this->getStatisticsArray($contact0);
    $stats1 = $this->getStatisticsArray($contact1);

    // Statistics should be removed (count was 0).
    $this->assertArrayNotHasKey('friends', $stats0, 'Contact 0 should have no friends statistics after delete.');
    $this->assertArrayNotHasKey('friends', $stats1, 'Contact 1 should have no friends statistics after delete.');
  }

  /**
   * Tests that count reaches zero and entry is removed.
   *
   * @covers \Drupal\crm\Service\RelationshipStatisticsService::decrement
   */
  public function testZeroCountRemovesEntry(): void {
    // Create two relationships.
    $relationship1 = Relationship::create([
      'bundle' => 'friends',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[1]->id()],
      'status' => TRUE,
    ]);
    $relationship1->save();

    $relationship2 = Relationship::create([
      'bundle' => 'friends',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[2]->id()],
      'status' => TRUE,
    ]);
    $relationship2->save();

    // Verify count is 2.
    $contact0 = Contact::load($this->contacts[0]->id());
    $stats0 = $this->getStatisticsArray($contact0);
    $this->assertEquals(2, $stats0['friends'] ?? 0);

    // Delete one relationship.
    $relationship1->delete();

    // Verify count is 1.
    $contact0 = Contact::load($this->contacts[0]->id());
    $stats0 = $this->getStatisticsArray($contact0);
    $this->assertEquals(1, $stats0['friends'] ?? 0);

    // Delete second relationship.
    $relationship2->delete();

    // Verify entry is removed.
    $contact0 = Contact::load($this->contacts[0]->id());
    $stats0 = $this->getStatisticsArray($contact0);
    $this->assertArrayNotHasKey('friends', $stats0, 'Entry should be removed when count reaches 0.');
  }

  /**
   * Tests inactive relationships are not counted.
   *
   * @covers ::relationshipInsert
   */
  public function testInactiveRelationshipsNotCounted(): void {
    // Create an inactive relationship.
    $relationship = Relationship::create([
      'bundle' => 'friends',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[1]->id()],
      'status' => FALSE,
    ]);
    $relationship->save();

    // Reload contacts.
    $contact0 = Contact::load($this->contacts[0]->id());
    $contact1 = Contact::load($this->contacts[1]->id());

    $stats0 = $this->getStatisticsArray($contact0);
    $stats1 = $this->getStatisticsArray($contact1);

    // No statistics should be recorded.
    $this->assertEmpty($stats0, 'Contact 0 should have no statistics for inactive relationship.');
    $this->assertEmpty($stats1, 'Contact 1 should have no statistics for inactive relationship.');
  }

  /**
   * Tests status change from inactive to active.
   *
   * @covers ::relationshipUpdate
   */
  public function testStatusChangeToActiveIncrementsStatistics(): void {
    // Create an inactive relationship.
    $relationship = Relationship::create([
      'bundle' => 'friends',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[1]->id()],
      'status' => FALSE,
    ]);
    $relationship->save();

    // Activate the relationship.
    $relationship->set('status', TRUE);
    $relationship->save();

    // Reload contacts.
    $contact0 = Contact::load($this->contacts[0]->id());
    $contact1 = Contact::load($this->contacts[1]->id());

    $stats0 = $this->getStatisticsArray($contact0);
    $stats1 = $this->getStatisticsArray($contact1);

    $this->assertEquals(1, $stats0['friends'] ?? 0, 'Statistics should be incremented when relationship becomes active.');
    $this->assertEquals(1, $stats1['friends'] ?? 0, 'Statistics should be incremented when relationship becomes active.');
  }

  /**
   * Tests status change from active to inactive.
   *
   * @covers ::relationshipUpdate
   */
  public function testStatusChangeToInactiveDecrementsStatistics(): void {
    // Create an active relationship.
    $relationship = Relationship::create([
      'bundle' => 'friends',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[1]->id()],
      'status' => TRUE,
    ]);
    $relationship->save();

    // Verify statistics exist.
    $contact0 = Contact::load($this->contacts[0]->id());
    $stats0 = $this->getStatisticsArray($contact0);
    $this->assertEquals(1, $stats0['friends'] ?? 0);

    // Deactivate the relationship.
    $relationship->set('status', FALSE);
    $relationship->save();

    // Reload contacts.
    $contact0 = Contact::load($this->contacts[0]->id());
    $contact1 = Contact::load($this->contacts[1]->id());

    $stats0 = $this->getStatisticsArray($contact0);
    $stats1 = $this->getStatisticsArray($contact1);

    $this->assertArrayNotHasKey('friends', $stats0, 'Statistics should be removed when relationship becomes inactive.');
    $this->assertArrayNotHasKey('friends', $stats1, 'Statistics should be removed when relationship becomes inactive.');
  }

  /**
   * Tests that relationship operations do not create new contact revisions.
   *
   * Statistics updates should write directly to the database without saving
   * the contact entity, avoiding the creation of new revisions.
   *
   * @covers ::relationshipInsert
   * @covers ::relationshipUpdate
   * @covers ::relationshipDelete
   */
  public function testRelationshipOperationsDoNotCreateContactRevisions(): void {
    $contact_storage = \Drupal::entityTypeManager()->getStorage('crm_contact');

    // Get initial revision IDs.
    $contact0 = Contact::load($this->contacts[0]->id());
    $contact1 = Contact::load($this->contacts[1]->id());
    $initial_revision_0 = $contact0->getRevisionId();
    $initial_revision_1 = $contact1->getRevisionId();

    // Create a relationship (triggers insert hook).
    $relationship = Relationship::create([
      'bundle' => 'friends',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[1]->id()],
      'status' => TRUE,
    ]);
    $relationship->save();

    // Verify statistics were updated but revisions unchanged.
    $contact_storage->resetCache([$this->contacts[0]->id(), $this->contacts[1]->id()]);
    $contact0 = Contact::load($this->contacts[0]->id());
    $contact1 = Contact::load($this->contacts[1]->id());

    $this->assertEquals(1, $this->getStatisticsArray($contact0)['friends'] ?? 0);
    $this->assertEquals($initial_revision_0, $contact0->getRevisionId(), 'Insert: Contact 0 revision should not change.');
    $this->assertEquals($initial_revision_1, $contact1->getRevisionId(), 'Insert: Contact 1 revision should not change.');

    // Deactivate the relationship (triggers update hook).
    $relationship->set('status', FALSE);
    $relationship->save();

    $contact_storage->resetCache([$this->contacts[0]->id(), $this->contacts[1]->id()]);
    $contact0 = Contact::load($this->contacts[0]->id());
    $contact1 = Contact::load($this->contacts[1]->id());

    $this->assertArrayNotHasKey('friends', $this->getStatisticsArray($contact0));
    $this->assertEquals($initial_revision_0, $contact0->getRevisionId(), 'Update (deactivate): Contact 0 revision should not change.');
    $this->assertEquals($initial_revision_1, $contact1->getRevisionId(), 'Update (deactivate): Contact 1 revision should not change.');

    // Reactivate the relationship.
    $relationship->set('status', TRUE);
    $relationship->save();

    $contact_storage->resetCache([$this->contacts[0]->id(), $this->contacts[1]->id()]);
    $contact0 = Contact::load($this->contacts[0]->id());
    $contact1 = Contact::load($this->contacts[1]->id());

    $this->assertEquals($initial_revision_0, $contact0->getRevisionId(), 'Update (reactivate): Contact 0 revision should not change.');
    $this->assertEquals($initial_revision_1, $contact1->getRevisionId(), 'Update (reactivate): Contact 1 revision should not change.');

    // Delete the relationship (triggers delete hook).
    $relationship->delete();

    $contact_storage->resetCache([$this->contacts[0]->id(), $this->contacts[1]->id()]);
    $contact0 = Contact::load($this->contacts[0]->id());
    $contact1 = Contact::load($this->contacts[1]->id());

    $this->assertArrayNotHasKey('friends', $this->getStatisticsArray($contact0));
    $this->assertEquals($initial_revision_0, $contact0->getRevisionId(), 'Delete: Contact 0 revision should not change.');
    $this->assertEquals($initial_revision_1, $contact1->getRevisionId(), 'Delete: Contact 1 revision should not change.');
  }

  /**
   * Helper function to get statistics as an associative array.
   *
   * @param \Drupal\crm\Entity\Contact $contact
   *   The contact entity.
   *
   * @return array
   *   An associative array of type_key => count.
   */
  protected function getStatisticsArray(Contact $contact): array {
    $field = $contact->get('relationship_statistics');
    $values = $field->getValue();

    $result = [];
    foreach ($values as $item) {
      $result[$item['value']] = (int) $item['count'];
    }

    return $result;
  }

}
