<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Kernel;

use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;
use Drupal\crm\Entity\Contact;
use Drupal\crm\Entity\UserContactMapping;

/**
 * Tests crm user contact mapping.
 *
 * @group crm
 */
class HookUserDeleteTest extends EntityKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'address',
    'crm',
    'inline_entity_form',
    'primary_entity_reference',
    'datetime',
    'name',
    'telephone',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installSchema('user', ['users_data']);
    $this->installEntitySchema('crm_contact');
    $this->installEntitySchema('crm_contact_detail');
    $this->installEntitySchema('crm_user_contact_mapping');

    $this->installConfig(['crm', 'name']);

  }

  /**
   * Verify the mapping is deleted when a user is deleted.
   */
  public function testUserDeleteWithContactMapping(): void {
    // Create a user.
    $user_1 = $this->drupalCreateUser();
    $user_1->save();

    $contact_1 = Contact::create([
      'name' => 'Test Contact',
      'bundle' => 'person',
      'full_name' => [
        'given_name' => 'Test',
        'family_name' => 'UserContact 1',
      ],
    ]);
    $contact_1->save();

    $crm_user_1 = UserContactMapping::create([
      'user' => $user_1->id(),
      'crm_contact' => $contact_1->id(),
    ]);
    $crm_user_1->save();

    $user_2 = $this->drupalCreateUser();
    $user_2->save();

    $contact_2 = Contact::create([
      'name' => 'Test Contact 2',
      'bundle' => 'person',
      'full_name' => [
        'given_name' => 'Test',
        'family_name' => 'UserContact 2',
      ],
    ]);
    $contact_2->save();

    $crm_user_2 = UserContactMapping::create([
      'user' => $user_2->id(),
      'crm_contact' => $contact_2->id(),
    ]);
    $crm_user_2->save();

    $crm_users = UserContactMapping::loadMultiple();
    $this->assertCount(2, $crm_users, 'Two crm_user exists.');

    // Delete the first user.
    $user_1->delete();

    // Reset the user cache to ensure we are working with the latest data.
    $crm_users = UserContactMapping::loadMultiple();
    $this->assertCount(1, $crm_users, 'One crm_user entity remains after deleting the user.');
    $crm_user = reset($crm_users);
    $this->assertEquals($user_2->id(), $crm_user->getUserId(), 'The remaining crm_user entity is linked to the second user.');
    $this->assertEquals($contact_2->id(), $crm_user->getContactId(), 'The remaining crm_user entity is linked to the second contact.');
  }

}
