<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Kernel\Plugin\Validation\Constraint;

use Drupal\crm\Entity\Contact;
use Drupal\crm\Entity\Relationship;
use Drupal\crm\Entity\RelationshipType;
use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;

/**
 * Tests the RelationshipLimit constraint validator.
 *
 * @group crm
 */
class RelationshipLimitConstraintValidatorTest extends EntityKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'address',
    'crm',
    'inline_entity_form',
    'primary_entity_reference',
    'datetime',
    'name',
    'telephone',
  ];

  /**
   * Relationship type with limits.
   *
   * @var \Drupal\crm\Entity\RelationshipType
   */
  protected RelationshipType $limitedType;

  /**
   * Relationship type without limits.
   *
   * @var \Drupal\crm\Entity\RelationshipType
   */
  protected RelationshipType $unlimitedType;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('crm_contact');
    $this->installEntitySchema('crm_contact_detail');
    $this->installEntitySchema('crm_relationship');

    $this->installConfig(['crm', 'name']);

    // Create relationship type with limits.
    $this->limitedType = RelationshipType::create([
      'id' => 'test_parent_limited',
      'label' => 'Parent',
      'asymmetric' => TRUE,
      'label_a' => 'Parent',
      'contact_type_a' => ['person'],
      'label_b' => 'Child',
      'contact_type_b' => ['person'],
      'limit_a' => NULL,
      'limit_b' => 2,
      'limit_active_only' => FALSE,
    ]);
    $this->limitedType->save();

    // Create relationship type without limits.
    $this->unlimitedType = RelationshipType::create([
      'id' => 'test_friendship_unlimited',
      'label' => 'Friendship',
      'asymmetric' => FALSE,
      'label_a' => 'Friend',
      'contact_type_a' => ['person'],
      'label_b' => 'Friend',
      'contact_type_b' => ['person'],
      'limit_a' => NULL,
      'limit_b' => NULL,
      'limit_active_only' => FALSE,
    ]);
    $this->unlimitedType->save();
  }

  /**
   * Tests that relationships can be created when under the limit.
   */
  public function testRelationshipCreationUnderLimit(): void {
    $parent1 = $this->createContact('Parent 1');
    $parent2 = $this->createContact('Parent 2');
    $child = $this->createContact('Child');

    // First parent relationship should succeed.
    $relationship1 = Relationship::create([
      'bundle' => 'test_parent_limited',
      'contacts' => [
        ['target_id' => $parent1->id()],
        ['target_id' => $child->id()],
      ],
    ]);
    $violations = $relationship1->validate();
    $this->assertCount(0, $violations, 'First relationship has no limit violations.');
    $relationship1->save();

    // Second parent relationship should also succeed (limit is 2).
    $relationship2 = Relationship::create([
      'bundle' => 'test_parent_limited',
      'contacts' => [
        ['target_id' => $parent2->id()],
        ['target_id' => $child->id()],
      ],
    ]);
    $violations = $relationship2->validate();
    $this->assertCount(0, $violations, 'Second relationship has no limit violations.');
    $relationship2->save();
  }

  /**
   * Tests that validation fails when limit is exceeded.
   */
  public function testRelationshipCreationExceedsLimit(): void {
    $parent1 = $this->createContact('Parent 1');
    $parent2 = $this->createContact('Parent 2');
    $parent3 = $this->createContact('Parent 3');
    $child = $this->createContact('Child');

    // Create two parent relationships (at the limit).
    Relationship::create([
      'bundle' => 'test_parent_limited',
      'contacts' => [
        ['target_id' => $parent1->id()],
        ['target_id' => $child->id()],
      ],
    ])->save();

    Relationship::create([
      'bundle' => 'test_parent_limited',
      'contacts' => [
        ['target_id' => $parent2->id()],
        ['target_id' => $child->id()],
      ],
    ])->save();

    // Third parent relationship should fail.
    $relationship3 = Relationship::create([
      'bundle' => 'test_parent_limited',
      'contacts' => [
        ['target_id' => $parent3->id()],
        ['target_id' => $child->id()],
      ],
    ]);

    $violations = $relationship3->validate();
    $this->assertGreaterThan(0, $violations->count(), 'Third relationship exceeds limit.');

    $violation_messages = [];
    foreach ($violations as $violation) {
      $violation_messages[] = (string) $violation->getMessage();
    }

    $this->assertTrue(
      $this->containsLimitMessage($violation_messages, 'Child'),
      'Violation message mentions Child position limit.'
    );
  }

  /**
   * Tests that unlimited relationship type allows many relationships.
   */
  public function testUnlimitedRelationshipType(): void {
    $contacts = [];
    for ($i = 0; $i < 5; $i++) {
      $contacts[] = $this->createContact('Person ' . $i);
    }

    // Create multiple friendships with the first person.
    for ($i = 1; $i < 5; $i++) {
      $relationship = Relationship::create([
        'bundle' => 'test_friendship_unlimited',
        'contacts' => [
          ['target_id' => $contacts[0]->id()],
          ['target_id' => $contacts[$i]->id()],
        ],
      ]);
      $violations = $relationship->validate();
      $this->assertCount(0, $violations, "Friendship $i has no limit violations.");
      $relationship->save();
    }
  }

  /**
   * Tests active-only limit counting.
   */
  public function testActiveOnlyLimits(): void {
    // Create a type that only counts active relationships.
    $activeOnlyType = RelationshipType::create([
      'id' => 'test_supervisor_active_only',
      'label' => 'Supervisor',
      'asymmetric' => TRUE,
      'label_a' => 'Supervisor',
      'contact_type_a' => ['person'],
      'label_b' => 'Subordinate',
      'contact_type_b' => ['person'],
      'limit_a' => NULL,
      'limit_b' => 1,
      'limit_active_only' => TRUE,
    ]);
    $activeOnlyType->save();

    $supervisor1 = $this->createContact('Supervisor 1');
    $supervisor2 = $this->createContact('Supervisor 2');
    $subordinate = $this->createContact('Subordinate');

    // Create an inactive relationship.
    $relationship1 = Relationship::create([
      'bundle' => 'test_supervisor_active_only',
      'contacts' => [
        ['target_id' => $supervisor1->id()],
        ['target_id' => $subordinate->id()],
      ],
      'status' => FALSE,
    ]);
    $relationship1->save();

    // Create an active relationship - should succeed because the previous is
    // inactive.
    $relationship2 = Relationship::create([
      'bundle' => 'test_supervisor_active_only',
      'contacts' => [
        ['target_id' => $supervisor2->id()],
        ['target_id' => $subordinate->id()],
      ],
      'status' => TRUE,
    ]);
    $violations = $relationship2->validate();
    $this->assertCount(0, $violations, 'Active relationship should succeed when only inactive exists.');
  }

  /**
   * Tests that editing existing relationship doesn't count against itself.
   */
  public function testEditingExistingRelationship(): void {
    $parent = $this->createContact('Parent');
    $child = $this->createContact('Child');

    // Create a relationship at the limit.
    $limitedType = RelationshipType::create([
      'id' => 'test_single_parent',
      'label' => 'Single Parent',
      'asymmetric' => TRUE,
      'label_a' => 'Parent',
      'contact_type_a' => ['person'],
      'label_b' => 'Child',
      'contact_type_b' => ['person'],
      'limit_a' => NULL,
      'limit_b' => 1,
      'limit_active_only' => FALSE,
    ]);
    $limitedType->save();

    $relationship = Relationship::create([
      'bundle' => 'test_single_parent',
      'contacts' => [
        ['target_id' => $parent->id()],
        ['target_id' => $child->id()],
      ],
    ]);
    $relationship->save();

    // Edit the relationship (change a different field).
    $relationship->set('status', FALSE);
    $violations = $relationship->validate();
    $this->assertCount(0, $violations, 'Editing existing relationship should not trigger limit violation.');
  }

  /**
   * Tests that limit applies to correct side (asymmetric).
   */
  public function testAsymmetricLimits(): void {
    $parent1 = $this->createContact('Parent 1');
    $child1 = $this->createContact('Child 1');
    $child2 = $this->createContact('Child 2');
    $child3 = $this->createContact('Child 3');

    // Parent can have unlimited children (limit_a is NULL).
    Relationship::create([
      'bundle' => 'test_parent_limited',
      'contacts' => [
        ['target_id' => $parent1->id()],
        ['target_id' => $child1->id()],
      ],
    ])->save();

    Relationship::create([
      'bundle' => 'test_parent_limited',
      'contacts' => [
        ['target_id' => $parent1->id()],
        ['target_id' => $child2->id()],
      ],
    ])->save();

    $relationship3 = Relationship::create([
      'bundle' => 'test_parent_limited',
      'contacts' => [
        ['target_id' => $parent1->id()],
        ['target_id' => $child3->id()],
      ],
    ]);
    $violations = $relationship3->validate();
    $this->assertCount(0, $violations, 'Parent can have multiple children without limit.');
  }

  /**
   * Creates a person contact.
   *
   * @param string $name
   *   The contact name.
   *
   * @return \Drupal\crm\Entity\Contact
   *   The created contact.
   */
  protected function createContact(string $name): Contact {
    $contact = Contact::create([
      'bundle' => 'person',
      'name' => $name,
    ]);
    $contact->save();
    return $contact;
  }

  /**
   * Checks if any message contains a limit-related message for a given label.
   *
   * @param array $messages
   *   The violation messages.
   * @param string $label
   *   The label to check for.
   *
   * @return bool
   *   TRUE if a limit message is found.
   */
  protected function containsLimitMessage(array $messages, string $label): bool {
    foreach ($messages as $message) {
      if (str_contains($message, $label) && str_contains($message, 'maximum')) {
        return TRUE;
      }
    }
    return FALSE;
  }

}
