<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Kernel\Plugin\views;

use Drupal\crm\Entity\Contact;
use Drupal\crm\Entity\ContactType;
use Drupal\crm\Entity\Relationship;
use Drupal\crm\Entity\RelationshipType;
use Drupal\KernelTests\KernelTestBase;
use Drupal\views\Views;

/**
 * Tests the relationship statistics Views plugins.
 *
 * @group crm
 */
class RelationshipStatisticsViewsTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'address',
    'crm',
    'datetime',
    'field',
    'filter',
    'inline_entity_form',
    'primary_entity_reference',
    'name',
    'system',
    'telephone',
    'text',
    'user',
    'views',
  ];

  /**
   * Relationship types for testing.
   *
   * @var array
   */
  protected array $relationshipTypes = [];

  /**
   * Contact type for testing.
   *
   * @var \Drupal\crm\Entity\ContactType
   */
  protected ContactType $contactType;

  /**
   * Contacts for testing.
   *
   * @var array
   */
  protected array $contacts = [];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('crm_contact');
    $this->installEntitySchema('crm_contact_type');
    $this->installEntitySchema('crm_relationship');
    $this->installEntitySchema('crm_relationship_type');
    $this->installSchema('system', ['sequences']);

    // Create a contact type.
    $this->contactType = ContactType::create([
      'id' => 'person',
      'label' => 'Person',
      'date' => [
        'start_date' => [
          'label' => 'Start date',
        ],
        'end_date' => [
          'label' => 'End date',
        ],
      ],
    ]);
    $this->contactType->save();

    // Create a symmetric relationship type.
    $this->relationshipTypes['friends'] = RelationshipType::create([
      'id' => 'friends',
      'label' => 'Friends',
      'label_a' => 'Friend',
      'contact_type_a' => ['person'],
      'asymmetric' => FALSE,
    ]);
    $this->relationshipTypes['friends']->save();

    // Create an asymmetric relationship type.
    $this->relationshipTypes['parent_child'] = RelationshipType::create([
      'id' => 'parent_child',
      'label' => 'Parent-Child',
      'label_a' => 'Parent',
      'label_b' => 'Child',
      'contact_type_a' => ['person'],
      'contact_type_b' => ['person'],
      'asymmetric' => TRUE,
    ]);
    $this->relationshipTypes['parent_child']->save();

    // Create test contacts.
    for ($i = 0; $i < 4; $i++) {
      $this->contacts[$i] = Contact::create([
        'bundle' => 'person',
        'name' => 'Contact ' . $i,
      ]);
      $this->contacts[$i]->save();
    }

    // Create relationships.
    // Contact 0: 2 friends, 1 parent.
    Relationship::create([
      'bundle' => 'friends',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[1]->id()],
      'status' => TRUE,
    ])->save();

    Relationship::create([
      'bundle' => 'friends',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[2]->id()],
      'status' => TRUE,
    ])->save();

    Relationship::create([
      'bundle' => 'parent_child',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[3]->id()],
      'status' => TRUE,
    ])->save();
  }

  /**
   * Tests that views data is properly altered for relationship statistics.
   *
   * @covers \Drupal\crm\ContactViewsData::getViewsData
   */
  public function testViewsDataAlter(): void {
    $views_data = Views::viewsData()->getAll();

    // Check that the relationship statistics table data exists.
    $this->assertArrayHasKey('crm_contact__relationship_statistics', $views_data);

    $table_data = $views_data['crm_contact__relationship_statistics'];

    // Check the value field has the filter plugin.
    $this->assertArrayHasKey('relationship_statistics_value', $table_data);
    $this->assertArrayHasKey('filter', $table_data['relationship_statistics_value']);
    $this->assertEquals('crm_relationship_statistics_type', $table_data['relationship_statistics_value']['filter']['id']);

    // Check the count field has the field and sort plugins.
    $this->assertArrayHasKey('relationship_statistics_count', $table_data);
    $this->assertArrayHasKey('field', $table_data['relationship_statistics_count']);
    $this->assertArrayHasKey('sort', $table_data['relationship_statistics_count']);
    $this->assertEquals('crm_relationship_statistics', $table_data['relationship_statistics_count']['field']['id']);
    $this->assertEquals('crm_relationship_statistics_count', $table_data['relationship_statistics_count']['sort']['id']);
  }

  /**
   * Tests the RelationshipStatistics field plugin creation.
   *
   * @covers \Drupal\crm\Plugin\views\field\RelationshipStatistics
   */
  public function testFieldPluginCreation(): void {
    $plugin_manager = $this->container->get('plugin.manager.views.field');

    // Check the plugin can be created.
    $this->assertTrue($plugin_manager->hasDefinition('crm_relationship_statistics'));

    $plugin = $plugin_manager->createInstance('crm_relationship_statistics', [
      'field' => 'relationship_statistics_count',
      'table' => 'crm_contact__relationship_statistics',
    ]);

    $this->assertInstanceOf('Drupal\crm\Plugin\views\field\RelationshipStatistics', $plugin);
  }

  /**
   * Tests the RelationshipStatisticsType filter plugin creation.
   *
   * @covers \Drupal\crm\Plugin\views\filter\RelationshipStatisticsType
   */
  public function testFilterPluginCreation(): void {
    $plugin_manager = $this->container->get('plugin.manager.views.filter');

    // Check the plugin can be created.
    $this->assertTrue($plugin_manager->hasDefinition('crm_relationship_statistics_type'));

    $plugin = $plugin_manager->createInstance('crm_relationship_statistics_type', [
      'field' => 'relationship_statistics_value',
      'table' => 'crm_contact__relationship_statistics',
    ]);

    $this->assertInstanceOf('Drupal\crm\Plugin\views\filter\RelationshipStatisticsType', $plugin);
  }

  /**
   * Tests the RelationshipStatisticsType filter plugin options.
   *
   * @covers \Drupal\crm\Plugin\views\filter\RelationshipStatisticsType::getValueOptions
   */
  public function testFilterPluginOptions(): void {
    $plugin_manager = $this->container->get('plugin.manager.views.filter');

    $plugin = $plugin_manager->createInstance('crm_relationship_statistics_type', [
      'field' => 'relationship_statistics_value',
      'table' => 'crm_contact__relationship_statistics',
    ]);

    $options = $plugin->getValueOptions();

    // Should have 3 options: friends + parent_child:a + parent_child:b.
    $this->assertCount(3, $options);
    $this->assertArrayHasKey('friends', $options);
    $this->assertArrayHasKey('parent_child:a', $options);
    $this->assertArrayHasKey('parent_child:b', $options);
  }

  /**
   * Tests the RelationshipStatisticsCount sort plugin creation.
   *
   * @covers \Drupal\crm\Plugin\views\sort\RelationshipStatisticsCount
   */
  public function testSortPluginCreation(): void {
    $plugin_manager = $this->container->get('plugin.manager.views.sort');

    // Check the plugin can be created.
    $this->assertTrue($plugin_manager->hasDefinition('crm_relationship_statistics_count'));

    $plugin = $plugin_manager->createInstance('crm_relationship_statistics_count', [
      'field' => 'relationship_statistics_count',
      'table' => 'crm_contact__relationship_statistics',
    ]);

    $this->assertInstanceOf('Drupal\crm\Plugin\views\sort\RelationshipStatisticsCount', $plugin);
  }

  /**
   * Tests contact statistics are correct for test data.
   */
  public function testContactStatisticsSetup(): void {
    // Verify Contact 0 has expected statistics.
    $contact0 = Contact::load($this->contacts[0]->id());
    $stats = $this->getStatisticsArray($contact0);

    $this->assertEquals(2, $stats['friends'] ?? 0, 'Contact 0 should have 2 friends.');
    $this->assertEquals(1, $stats['parent_child:a'] ?? 0, 'Contact 0 should be a parent of 1 child.');

    // Verify Contact 3 is a child.
    $contact3 = Contact::load($this->contacts[3]->id());
    $stats = $this->getStatisticsArray($contact3);

    $this->assertEquals(1, $stats['parent_child:b'] ?? 0, 'Contact 3 should be a child.');
  }

  /**
   * Helper function to get statistics as an associative array.
   *
   * @param \Drupal\crm\Entity\Contact $contact
   *   The contact entity.
   *
   * @return array
   *   An associative array of type_key => count.
   */
  protected function getStatisticsArray(Contact $contact): array {
    $field = $contact->get('relationship_statistics');
    $values = $field->getValue();

    $result = [];
    foreach ($values as $item) {
      $result[$item['value']] = (int) $item['count'];
    }

    return $result;
  }

}
