<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Kernel\Service;

use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\crm\Entity\Contact;
use Drupal\crm\Entity\ContactType;
use Drupal\crm\Entity\Relationship;
use Drupal\crm\Entity\RelationshipType;
use Drupal\crm\RelationshipStatisticsServiceInterface;
use Drupal\KernelTests\KernelTestBase;

/**
 * Tests the RelationshipStatisticsService.
 *
 * @group crm
 * @coversDefaultClass \Drupal\crm\Service\RelationshipStatisticsService
 */
class RelationshipStatisticsServiceTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'address',
    'crm',
    'datetime',
    'field',
    'inline_entity_form',
    'primary_entity_reference',
    'name',
    'system',
    'telephone',
    'text',
    'user',
  ];

  /**
   * The relationship statistics service.
   *
   * @var \Drupal\crm\RelationshipStatisticsServiceInterface
   */
  protected RelationshipStatisticsServiceInterface $statisticsService;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $database;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Relationship types for testing.
   *
   * @var array
   */
  protected array $relationshipTypes = [];

  /**
   * Contact type for testing.
   *
   * @var \Drupal\crm\Entity\ContactType
   */
  protected ContactType $contactType;

  /**
   * Contacts for testing.
   *
   * @var array
   */
  protected array $contacts = [];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('crm_contact');
    $this->installEntitySchema('crm_contact_type');
    $this->installEntitySchema('crm_relationship');
    $this->installEntitySchema('crm_relationship_type');
    $this->installSchema('system', ['sequences']);

    $this->statisticsService = $this->container->get('crm.relationship_statistics');
    $this->database = $this->container->get('database');
    $this->entityTypeManager = $this->container->get('entity_type.manager');

    // Create a contact type.
    $this->contactType = ContactType::create([
      'id' => 'person',
      'label' => 'Person',
      'date' => [
        'start_date' => [
          'label' => 'Start date',
        ],
        'end_date' => [
          'label' => 'End date',
        ],
      ],
    ]);
    $this->contactType->save();

    // Create relationship types.
    $this->relationshipTypes['friends'] = RelationshipType::create([
      'id' => 'friends',
      'label' => 'Friends',
      'label_a' => 'Friend',
      'contact_type_a' => ['person'],
      'asymmetric' => FALSE,
    ]);
    $this->relationshipTypes['friends']->save();

    $this->relationshipTypes['parent_child'] = RelationshipType::create([
      'id' => 'parent_child',
      'label' => 'Parent-Child',
      'label_a' => 'Parent',
      'label_b' => 'Child',
      'contact_type_a' => ['person'],
      'contact_type_b' => ['person'],
      'asymmetric' => TRUE,
    ]);
    $this->relationshipTypes['parent_child']->save();

    // Create test contacts.
    for ($i = 0; $i < 4; $i++) {
      $this->contacts[$i] = Contact::create([
        'bundle' => 'person',
        'name' => 'Contact ' . $i,
      ]);
      $this->contacts[$i]->save();
    }
  }

  /**
   * Tests getTypeKey for symmetric relationships.
   *
   * @covers ::getTypeKey
   */
  public function testGetTypeKeySymmetric(): void {
    $relationship = Relationship::create([
      'bundle' => 'friends',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[1]->id()],
      'status' => TRUE,
    ]);

    $key_a = $this->statisticsService->getTypeKey($relationship, 'a');
    $key_b = $this->statisticsService->getTypeKey($relationship, 'b');

    // For symmetric relationships, both positions should have the same key.
    $this->assertEquals('friends', $key_a);
    $this->assertEquals('friends', $key_b);
  }

  /**
   * Tests getTypeKey for asymmetric relationships.
   *
   * @covers ::getTypeKey
   */
  public function testGetTypeKeyAsymmetric(): void {
    $relationship = Relationship::create([
      'bundle' => 'parent_child',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[1]->id()],
      'status' => TRUE,
    ]);

    $key_a = $this->statisticsService->getTypeKey($relationship, 'a');
    $key_b = $this->statisticsService->getTypeKey($relationship, 'b');

    // For asymmetric relationships, positions should have different keys.
    $this->assertEquals('parent_child:a', $key_a);
    $this->assertEquals('parent_child:b', $key_b);
  }

  /**
   * Tests recalculateForContact rebuilds statistics from scratch.
   *
   * @covers ::recalculateForContact
   */
  public function testRecalculateForContact(): void {
    $relationship1 = Relationship::create([
      'bundle' => 'friends',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[1]->id()],
      'status' => TRUE,
    ]);
    $relationship1->save();

    $relationship2 = Relationship::create([
      'bundle' => 'friends',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[2]->id()],
      'status' => TRUE,
    ]);
    $relationship2->save();

    $relationship3 = Relationship::create([
      'bundle' => 'parent_child',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[3]->id()],
      'status' => TRUE,
    ]);
    $relationship3->save();

    // Clear the statistics directly in the database to test recalculation.
    $contact_id = $this->contacts[0]->id();
    $this->database->delete('crm_contact__relationship_statistics')
      ->condition('entity_id', $contact_id)
      ->execute();

    // Reset entity cache and verify statistics are empty.
    $this->entityTypeManager->getStorage('crm_contact')->resetCache([$contact_id]);
    $contact0 = Contact::load($contact_id);
    $this->assertTrue($contact0->get('relationship_statistics')->isEmpty());

    // Recalculate.
    $this->statisticsService->recalculateForContact($contact0);

    // Reload and verify.
    $contact0 = Contact::load($this->contacts[0]->id());
    $stats = $this->getStatisticsArray($contact0);

    $this->assertEquals(2, $stats['friends'] ?? 0, 'Contact 0 should have 2 friends relationships.');
    $this->assertEquals(1, $stats['parent_child:a'] ?? 0, 'Contact 0 should have 1 parent relationship.');
  }

  /**
   * Tests recalculateAll processes all contacts.
   *
   * @covers ::recalculateAll
   */
  public function testRecalculateAll(): void {
    // Create relationships.
    $relationship1 = Relationship::create([
      'bundle' => 'friends',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[1]->id()],
      'status' => TRUE,
    ]);
    $relationship1->save();

    // Clear all statistics directly in the database.
    $contact_ids = array_map(fn($c) => $c->id(), $this->contacts);
    $this->database->delete('crm_contact__relationship_statistics')
      ->condition('entity_id', $contact_ids, 'IN')
      ->execute();

    // Reset entity cache.
    $this->entityTypeManager->getStorage('crm_contact')->resetCache($contact_ids);

    // Recalculate all.
    $processed = $this->statisticsService->recalculateAll(2);

    // Should have processed all 4 contacts.
    $this->assertEquals(4, $processed);

    // Verify contact 0 has statistics.
    $contact0 = Contact::load($this->contacts[0]->id());
    $stats = $this->getStatisticsArray($contact0);
    $this->assertEquals(1, $stats['friends'] ?? 0);

    // Verify contact 1 has statistics.
    $contact1 = Contact::load($this->contacts[1]->id());
    $stats = $this->getStatisticsArray($contact1);
    $this->assertEquals(1, $stats['friends'] ?? 0);
  }

  /**
   * Tests that statistics updates do not create new contact revisions.
   *
   * @covers ::increment
   * @covers ::decrement
   */
  public function testStatisticsUpdateDoesNotCreateRevision(): void {
    // Get initial revision ID for contact 0.
    $contact0 = Contact::load($this->contacts[0]->id());
    $initial_revision_id = $contact0->getRevisionId();

    // Create a relationship, which triggers statistics increment.
    $relationship = Relationship::create([
      'bundle' => 'friends',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[1]->id()],
      'status' => TRUE,
    ]);
    $relationship->save();

    // Reload contact and verify statistics were updated.
    $this->entityTypeManager->getStorage('crm_contact')->resetCache([$this->contacts[0]->id()]);
    $contact0 = Contact::load($this->contacts[0]->id());
    $stats = $this->getStatisticsArray($contact0);
    $this->assertEquals(1, $stats['friends'] ?? 0, 'Statistics should be updated.');

    // Verify revision ID has not changed.
    $this->assertEquals($initial_revision_id, $contact0->getRevisionId(), 'Contact revision should not change when statistics are updated.');

    // Delete the relationship, which triggers statistics decrement.
    $relationship->delete();

    // Reload and verify statistics were removed but revision unchanged.
    $this->entityTypeManager->getStorage('crm_contact')->resetCache([$this->contacts[0]->id()]);
    $contact0 = Contact::load($this->contacts[0]->id());
    $stats = $this->getStatisticsArray($contact0);
    $this->assertArrayNotHasKey('friends', $stats, 'Statistics should be removed after relationship deletion.');

    // Verify revision ID still has not changed.
    $this->assertEquals($initial_revision_id, $contact0->getRevisionId(), 'Contact revision should not change when statistics are decremented.');
  }

  /**
   * Tests that recalculateForContact does not create new revisions.
   *
   * @covers ::recalculateForContact
   */
  public function testRecalculateDoesNotCreateRevision(): void {
    // Create a relationship.
    $relationship = Relationship::create([
      'bundle' => 'friends',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[1]->id()],
      'status' => TRUE,
    ]);
    $relationship->save();

    // Get current revision ID.
    $this->entityTypeManager->getStorage('crm_contact')->resetCache([$this->contacts[0]->id()]);
    $contact0 = Contact::load($this->contacts[0]->id());
    $initial_revision_id = $contact0->getRevisionId();

    // Recalculate statistics.
    $this->statisticsService->recalculateForContact($contact0);

    // Reload and verify revision unchanged.
    $this->entityTypeManager->getStorage('crm_contact')->resetCache([$this->contacts[0]->id()]);
    $contact0 = Contact::load($this->contacts[0]->id());
    $this->assertEquals($initial_revision_id, $contact0->getRevisionId(), 'Contact revision should not change when statistics are recalculated.');
  }

  /**
   * Helper function to get statistics as an associative array.
   *
   * @param \Drupal\crm\Entity\Contact $contact
   *   The contact entity.
   *
   * @return array
   *   An associative array of type_key => count.
   */
  protected function getStatisticsArray(Contact $contact): array {
    $field = $contact->get('relationship_statistics');
    $values = $field->getValue();

    $result = [];
    foreach ($values as $item) {
      $result[$item['value']] = (int) $item['count'];
    }

    return $result;
  }

}
