<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Kernel;

use Drupal\crm\Entity\Contact;
use Drupal\crm\Entity\UserContactMapping;
use Drupal\crm\UserContactMappingListBuilder;
use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;

/**
 * Tests the UserContactMappingListBuilder.
 *
 * @group crm
 * @coversDefaultClass \Drupal\crm\UserContactMappingListBuilder
 */
class UserContactMappingListBuilderTest extends EntityKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'address',
    'crm',
    'inline_entity_form',
    'primary_entity_reference',
    'datetime',
    'name',
    'telephone',
  ];

  /**
   * The list builder.
   *
   * @var \Drupal\crm\UserContactMappingListBuilder
   */
  protected $listBuilder;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('crm_contact');
    $this->installEntitySchema('crm_contact_detail');
    $this->installEntitySchema('crm_user_contact_mapping');

    $this->installConfig(['crm', 'name']);

    $this->listBuilder = $this->entityTypeManager->getListBuilder('crm_user_contact_mapping');
  }

  /**
   * Tests that the list builder is the correct class.
   *
   * @covers ::createInstance
   */
  public function testListBuilderClass(): void {
    $this->assertInstanceOf(UserContactMappingListBuilder::class, $this->listBuilder);
  }

  /**
   * Tests that createInstance properly injects dependencies.
   *
   * @covers ::createInstance
   */
  public function testCreateInstance(): void {
    $entityType = $this->entityTypeManager->getDefinition('crm_user_contact_mapping');
    $listBuilder = UserContactMappingListBuilder::createInstance(
      \Drupal::getContainer(),
      $entityType
    );

    $this->assertInstanceOf(UserContactMappingListBuilder::class, $listBuilder);
  }

  /**
   * Tests render with no mappings.
   *
   * @covers ::render
   */
  public function testRenderWithNoMappings(): void {
    $build = $this->listBuilder->render();

    // Verify summary is present.
    $this->assertArrayHasKey('summary', $build);
    $this->assertArrayHasKey('#markup', $build['summary']);
    $this->assertStringContainsString('Total mappings: 0', (string) $build['summary']['#markup']);

    // Verify table is present.
    $this->assertArrayHasKey('table', $build);
  }

  /**
   * Tests render with mappings.
   *
   * @covers ::render
   */
  public function testRenderWithMappings(): void {
    // Create test data.
    $user = $this->createUser();
    $contact = $this->createContact();
    $this->createMapping($user, $contact);

    $build = $this->listBuilder->render();

    // Verify summary shows correct count.
    $this->assertArrayHasKey('summary', $build);
    $this->assertStringContainsString('Total mappings: 1', (string) $build['summary']['#markup']);

    // Verify table is present.
    $this->assertArrayHasKey('table', $build);
  }

  /**
   * Tests render with multiple mappings.
   *
   * @covers ::render
   */
  public function testRenderWithMultipleMappings(): void {
    // Create multiple test mappings.
    for ($i = 0; $i < 3; $i++) {
      $user = $this->createUser([], 'user_' . $i);
      $contact = $this->createContact('Person ' . $i);
      $this->createMapping($user, $contact);
    }

    $build = $this->listBuilder->render();

    // Verify summary shows correct count.
    $this->assertStringContainsString('Total mappings: 3', (string) $build['summary']['#markup']);
  }

  /**
   * Tests buildHeader returns expected columns.
   *
   * @covers ::buildHeader
   */
  public function testBuildHeader(): void {
    $header = $this->listBuilder->buildHeader();

    $this->assertArrayHasKey('id', $header);
    $this->assertArrayHasKey('user', $header);
    $this->assertArrayHasKey('crm_contact', $header);
    $this->assertArrayHasKey('operations', $header);

    $this->assertEquals('ID', (string) $header['id']);
    $this->assertEquals('User', (string) $header['user']);
    $this->assertEquals('Person', (string) $header['crm_contact']);
  }

  /**
   * Tests buildRow with a mapping entity.
   *
   * @covers ::buildRow
   */
  public function testBuildRow(): void {
    $user = $this->createUser([], 'test_user');
    $contact = $this->createContact('Test Person');
    $mapping = $this->createMapping($user, $contact);

    $row = $this->listBuilder->buildRow($mapping);

    $this->assertEquals($mapping->id(), $row['id']);
    $this->assertEquals('test_user', $row['user']);
    $this->assertEquals('Test Person', $row['crm_contact']);
  }

  /**
   * Tests buildRow with different entities.
   *
   * @covers ::buildRow
   */
  public function testBuildRowWithDifferentData(): void {
    $user = $this->createUser([], 'another_user');
    $contact = $this->createContact('Another Person');
    $mapping = $this->createMapping($user, $contact);

    $row = $this->listBuilder->buildRow($mapping);

    $this->assertEquals($mapping->id(), $row['id']);
    $this->assertEquals('another_user', $row['user']);
    $this->assertEquals('Another Person', $row['crm_contact']);
    $this->assertArrayHasKey('operations', $row);
  }

  /**
   * Creates a test contact entity.
   *
   * @param string $name
   *   The name for the contact.
   *
   * @return \Drupal\crm\Entity\Contact
   *   The created contact entity.
   */
  protected function createContact(string $name = 'Test Person'): Contact {
    $contact = Contact::create([
      'bundle' => 'person',
      'full_name' => ['given' => $name],
    ]);
    $contact->save();
    return $contact;
  }

  /**
   * Creates a user contact mapping.
   *
   * @param \Drupal\user\UserInterface $user
   *   The user entity.
   * @param \Drupal\crm\Entity\Contact $contact
   *   The contact entity.
   *
   * @return \Drupal\crm\Entity\UserContactMapping
   *   The created mapping entity.
   */
  protected function createMapping($user, Contact $contact): UserContactMapping {
    $mapping = UserContactMapping::create([
      'user' => $user->id(),
      'crm_contact' => $contact->id(),
    ]);
    $mapping->save();
    return $mapping;
  }

}
