<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Kernel;

use Drupal\crm\Entity\Contact;
use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;

/**
 * Tests the UserContactMappingStorageSchema.
 *
 * @group crm
 * @coversDefaultClass \Drupal\crm\UserContactMappingStorageSchema
 */
class UserContactMappingStorageSchemaTest extends EntityKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'address',
    'crm',
    'inline_entity_form',
    'primary_entity_reference',
    'datetime',
    'name',
    'telephone',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('crm_contact');
    $this->installEntitySchema('crm_contact_detail');
    $this->installEntitySchema('crm_user_contact_mapping');

    $this->installConfig(['crm', 'name']);
  }

  /**
   * Tests that the crm_user_contact_mapping table exists.
   *
   * @covers ::getEntitySchema
   */
  public function testTableExists(): void {
    $schema = \Drupal::database()->schema();
    $this->assertTrue($schema->tableExists('crm_user_contact_mapping'));
  }

  /**
   * Tests that the user field has a unique index.
   *
   * @covers ::getEntitySchema
   */
  public function testUserFieldHasUniqueIndex(): void {
    $schema = \Drupal::database()->schema();

    // Check that the unique index exists on the user field.
    $this->assertTrue(
      $schema->indexExists('crm_user_contact_mapping', 'crm_user_contact_mapping__user'),
      'Unique index exists on user field.'
    );
  }

  /**
   * Tests that the crm_contact field has a unique index.
   *
   * @covers ::getEntitySchema
   */
  public function testCrmContactFieldHasUniqueIndex(): void {
    $schema = \Drupal::database()->schema();

    // Check that the unique index exists on the crm_contact field.
    $this->assertTrue(
      $schema->indexExists('crm_user_contact_mapping', 'crm_user_contact_mapping__crm_contact'),
      'Unique index exists on crm_contact field.'
    );
  }

  /**
   * Tests that user field is not null in the schema.
   *
   * @covers ::getEntitySchema
   */
  public function testUserFieldNotNull(): void {
    // Get the storage schema from the entity type manager.
    $entityType = $this->entityTypeManager->getDefinition('crm_user_contact_mapping');
    $storage = $this->entityTypeManager->getStorage('crm_user_contact_mapping');

    // Get the schema handler.
    $schemaHandler = $storage->getEntityType()->getHandlerClass('storage_schema');
    $this->assertEquals('Drupal\crm\UserContactMappingStorageSchema', $schemaHandler);

    // Verify via database introspection that the columns exist.
    $schema = \Drupal::database()->schema();
    $this->assertTrue($schema->fieldExists('crm_user_contact_mapping', 'user'));
  }

  /**
   * Tests that crm_contact field is not null in the schema.
   *
   * @covers ::getEntitySchema
   */
  public function testCrmContactFieldNotNull(): void {
    $schema = \Drupal::database()->schema();
    $this->assertTrue($schema->fieldExists('crm_user_contact_mapping', 'crm_contact'));
  }

  /**
   * Tests that duplicate user mappings are prevented by unique constraint.
   *
   * @covers ::getEntitySchema
   */
  public function testDuplicateUserPrevented(): void {
    // Create a user and contact for testing.
    $user = $this->createUser();
    $contact = $this->createContact();

    // Create the first mapping.
    $mapping1 = $this->entityTypeManager->getStorage('crm_user_contact_mapping')->create([
      'user' => $user->id(),
      'crm_contact' => $contact->id(),
    ]);
    $mapping1->save();

    // Create a second contact.
    $contact2 = $this->createContact('Second Person');

    // Attempt to create a second mapping with the same user.
    $mapping2 = $this->entityTypeManager->getStorage('crm_user_contact_mapping')->create([
      'user' => $user->id(),
      'crm_contact' => $contact2->id(),
    ]);

    // This should throw a database exception due to unique constraint.
    $this->expectException(\Exception::class);
    $mapping2->save();
  }

  /**
   * Tests that duplicate contact mappings are prevented by unique constraint.
   *
   * @covers ::getEntitySchema
   */
  public function testDuplicateContactPrevented(): void {
    // Create users and contact for testing.
    $user1 = $this->createUser();
    $user2 = $this->createUser([], 'another_user');
    $contact = $this->createContact();

    // Create the first mapping.
    $mapping1 = $this->entityTypeManager->getStorage('crm_user_contact_mapping')->create([
      'user' => $user1->id(),
      'crm_contact' => $contact->id(),
    ]);
    $mapping1->save();

    // Attempt to create a second mapping with the same contact.
    $mapping2 = $this->entityTypeManager->getStorage('crm_user_contact_mapping')->create([
      'user' => $user2->id(),
      'crm_contact' => $contact->id(),
    ]);

    // This should throw a database exception due to unique constraint.
    $this->expectException(\Exception::class);
    $mapping2->save();
  }

  /**
   * Creates a test contact entity.
   *
   * @param string $name
   *   The name for the contact.
   *
   * @return \Drupal\crm\Entity\Contact
   *   The created contact entity.
   */
  protected function createContact(string $name = 'Test Person'): Contact {
    $contact = Contact::create([
      'bundle' => 'person',
      'full_name' => ['given' => $name],
    ]);
    $contact->save();
    return $contact;
  }

}
