<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Unit;

use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Routing\RedirectDestinationInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\crm\ContactListBuilder;
use Drupal\Tests\UnitTestCase;

/**
 * Tests ContactListBuilder functionality.
 *
 * @group crm
 * @coversDefaultClass \Drupal\crm\ContactListBuilder
 */
class ContactListBuilderTest extends UnitTestCase {

  /**
   * The list builder under test.
   *
   * @var \Drupal\crm\ContactListBuilder
   */
  protected $listBuilder;

  /**
   * Mock entity type.
   *
   * @var \Drupal\Core\Entity\EntityTypeInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityType;

  /**
   * Mock entity storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $storage;

  /**
   * Mock date formatter.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $dateFormatter;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->entityType = $this->createMock(EntityTypeInterface::class);
    $this->entityType->method('id')->willReturn('crm_contact');
    $this->entityType->method('getKey')->willReturn('id');

    $this->storage = $this->createMock(EntityStorageInterface::class);
    $this->dateFormatter = $this->createMock(DateFormatterInterface::class);

    // Set up the container with required services.
    $container = new ContainerBuilder();
    $moduleHandler = $this->createMock(ModuleHandlerInterface::class);
    $moduleHandler->method('invokeAll')->willReturn([]);
    $moduleHandler->method('alter')->willReturn(NULL);
    $container->set('module_handler', $moduleHandler);

    $redirectDestination = $this->createMock(RedirectDestinationInterface::class);
    $redirectDestination->method('getAsArray')->willReturn(['destination' => '/admin/content/crm/contact']);
    $container->set('redirect.destination', $redirectDestination);

    \Drupal::setContainer($container);

    $this->listBuilder = new ContactListBuilder(
      $this->entityType,
      $this->storage,
      $this->dateFormatter
    );

    // Set up string translation.
    $translation = $this->createMock(TranslationInterface::class);
    $translation->method('translateString')->willReturnCallback(
      function ($string) {
        return $string->getUntranslatedString();
      }
    );
    $this->listBuilder->setStringTranslation($translation);
  }

  /**
   * Tests buildHeader returns expected columns.
   *
   * @covers ::buildHeader
   */
  public function testBuildHeader(): void {
    $header = $this->listBuilder->buildHeader();

    $this->assertArrayHasKey('id', $header);
    $this->assertArrayHasKey('name', $header);
    $this->assertArrayHasKey('type', $header);
    $this->assertArrayHasKey('status', $header);
    $this->assertArrayHasKey('created', $header);
    $this->assertArrayHasKey('changed', $header);
    $this->assertArrayHasKey('operations', $header);

    $this->assertEquals('ID', (string) $header['id']);
    $this->assertEquals('Name', (string) $header['name']);
    $this->assertEquals('Type', (string) $header['type']);
    $this->assertEquals('Status', (string) $header['status']);
    $this->assertEquals('Created', (string) $header['created']);
    $this->assertEquals('Updated', (string) $header['changed']);
  }

}
