<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Unit;

use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Routing\RedirectDestinationInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\crm\ContactTypeListBuilder;
use Drupal\Tests\UnitTestCase;

// Define the constant if it doesn't exist (normally defined by system module).
if (!defined('RESPONSIVE_PRIORITY_MEDIUM')) {
  define('RESPONSIVE_PRIORITY_MEDIUM', 'priority-medium');
}

/**
 * Tests ContactTypeListBuilder functionality.
 *
 * @group crm
 * @coversDefaultClass \Drupal\crm\ContactTypeListBuilder
 */
class ContactTypeListBuilderTest extends UnitTestCase {

  /**
   * The list builder under test.
   *
   * @var \Drupal\crm\ContactTypeListBuilder
   */
  protected $listBuilder;

  /**
   * Mock entity type.
   *
   * @var \Drupal\Core\Entity\EntityTypeInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityType;

  /**
   * Mock entity storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $storage;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->entityType = $this->createMock(EntityTypeInterface::class);
    $this->entityType->method('id')->willReturn('crm_contact_type');
    $this->entityType->method('getKey')->willReturn('id');

    $this->storage = $this->createMock(EntityStorageInterface::class);

    // Set up the container with required services.
    $container = new ContainerBuilder();
    $moduleHandler = $this->createMock(ModuleHandlerInterface::class);
    $moduleHandler->method('invokeAll')->willReturn([]);
    $moduleHandler->method('alter')->willReturn(NULL);
    $container->set('module_handler', $moduleHandler);

    $redirectDestination = $this->createMock(RedirectDestinationInterface::class);
    $redirectDestination->method('getAsArray')->willReturn(['destination' => '/admin/structure/crm/contact-types']);
    $container->set('redirect.destination', $redirectDestination);

    \Drupal::setContainer($container);

    $this->listBuilder = new ContactTypeListBuilder(
      $this->entityType,
      $this->storage
    );

    // Set up string translation.
    $translation = $this->createMock(TranslationInterface::class);
    $translation->method('translateString')->willReturnCallback(
      function ($string) {
        return $string->getUntranslatedString();
      }
    );
    $this->listBuilder->setStringTranslation($translation);
  }

  /**
   * Tests buildHeader returns expected columns.
   *
   * @covers ::buildHeader
   */
  public function testBuildHeader(): void {
    $header = $this->listBuilder->buildHeader();

    $this->assertArrayHasKey('label', $header);
    $this->assertArrayHasKey('description', $header);
    $this->assertArrayHasKey('operations', $header);

    $this->assertEquals('Label', (string) $header['label']);
    // Description has data and class keys.
    $this->assertEquals('Description', (string) $header['description']['data']);
  }

}
