<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Unit\Controller;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\crm\Controller\CommentController;
use Drupal\crm\CrmContactInterface;
use Drupal\Tests\UnitTestCase;

/**
 * Unit tests for the CommentController class.
 *
 * @group crm
 * @coversDefaultClass \Drupal\crm\Controller\CommentController
 */
class CommentControllerTest extends UnitTestCase {

  /**
   * The controller under test.
   *
   * @var \Drupal\crm\Controller\CommentController
   */
  protected $controller;

  /**
   * The renderer service mock.
   *
   * @var \Drupal\Core\Render\RendererInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $renderer;

  /**
   * The module handler service mock.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $moduleHandler;

  /**
   * The current user mock.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $currentUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $container = new ContainerBuilder();

    $string_translation = $this->getStringTranslationStub();
    $container->set('string_translation', $string_translation);

    $this->renderer = $this->createMock(RendererInterface::class);
    $container->set('renderer', $this->renderer);

    $this->moduleHandler = $this->createMock(ModuleHandlerInterface::class);
    $container->set('module_handler', $this->moduleHandler);

    $this->currentUser = $this->createMock(AccountProxyInterface::class);
    $container->set('current_user', $this->currentUser);

    \Drupal::setContainer($container);

    $this->controller = new CommentController($this->renderer, $this->moduleHandler);
  }

  /**
   * Tests the title method returns correct translatable markup.
   *
   * @covers ::title
   */
  public function testTitle(): void {
    $contact = $this->createMock(CrmContactInterface::class);
    $contact->expects($this->once())
      ->method('label')
      ->willReturn('John Doe');

    $result = $this->controller->title($contact);

    $this->assertStringContainsString('Comments for', (string) $result);
    $this->assertStringContainsString('John Doe', (string) $result);
  }

  /**
   * Tests access returns forbidden when comment module is not enabled.
   *
   * @covers ::access
   */
  public function testAccessForbiddenWhenCommentModuleNotEnabled(): void {
    $contact = $this->createMock(CrmContactInterface::class);

    $this->moduleHandler->expects($this->once())
      ->method('moduleExists')
      ->with('comment')
      ->willReturn(FALSE);

    $result = $this->controller->access($contact);

    $this->assertInstanceOf(AccessResult::class, $result);
    $this->assertTrue($result->isForbidden());
  }

  /**
   * Tests access returns forbidden when contact has no comment field.
   *
   * @covers ::access
   */
  public function testAccessForbiddenWhenNoCommentField(): void {
    $contact = $this->createMock(CrmContactInterface::class);
    $contact->expects($this->once())
      ->method('hasField')
      ->with('comment')
      ->willReturn(FALSE);

    $this->moduleHandler->expects($this->once())
      ->method('moduleExists')
      ->with('comment')
      ->willReturn(TRUE);

    $result = $this->controller->access($contact);

    $this->assertInstanceOf(AccessResult::class, $result);
    $this->assertTrue($result->isForbidden());
  }

  /**
   * Tests access returns allowed when all conditions are met.
   *
   * @covers ::access
   */
  public function testAccessAllowedWhenAllConditionsMet(): void {
    $contact = $this->createMock(CrmContactInterface::class);
    $contact->expects($this->once())
      ->method('hasField')
      ->with('comment')
      ->willReturn(TRUE);
    $contact->expects($this->once())
      ->method('access')
      ->with('view')
      ->willReturn(TRUE);

    $this->moduleHandler->expects($this->once())
      ->method('moduleExists')
      ->with('comment')
      ->willReturn(TRUE);

    $this->currentUser->expects($this->once())
      ->method('hasPermission')
      ->with('access comments')
      ->willReturn(TRUE);

    $result = $this->controller->access($contact);

    $this->assertInstanceOf(AccessResult::class, $result);
    $this->assertTrue($result->isAllowed());
  }

  /**
   * Tests access returns neutral when user cannot view contact.
   *
   * @covers ::access
   */
  public function testAccessNeutralWhenCannotViewContact(): void {
    $contact = $this->createMock(CrmContactInterface::class);
    $contact->expects($this->once())
      ->method('hasField')
      ->with('comment')
      ->willReturn(TRUE);
    $contact->expects($this->once())
      ->method('access')
      ->with('view')
      ->willReturn(FALSE);

    $this->moduleHandler->expects($this->once())
      ->method('moduleExists')
      ->with('comment')
      ->willReturn(TRUE);

    $result = $this->controller->access($contact);

    $this->assertInstanceOf(AccessResult::class, $result);
    $this->assertTrue($result->isNeutral());
  }

  /**
   * Tests access returns neutral when user lacks comment permission.
   *
   * @covers ::access
   */
  public function testAccessNeutralWhenNoCommentPermission(): void {
    $contact = $this->createMock(CrmContactInterface::class);
    $contact->expects($this->once())
      ->method('hasField')
      ->with('comment')
      ->willReturn(TRUE);
    $contact->expects($this->once())
      ->method('access')
      ->with('view')
      ->willReturn(TRUE);

    $this->moduleHandler->expects($this->once())
      ->method('moduleExists')
      ->with('comment')
      ->willReturn(TRUE);

    $this->currentUser->expects($this->once())
      ->method('hasPermission')
      ->with('access comments')
      ->willReturn(FALSE);

    $result = $this->controller->access($contact);

    $this->assertInstanceOf(AccessResult::class, $result);
    $this->assertTrue($result->isNeutral());
  }

}
