<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Unit\Controller;

use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\crm\Controller\RelationshipController;
use Drupal\crm\CrmContactInterface;
use Drupal\crm\Relationship;
use Drupal\Tests\UnitTestCase;

/**
 * Unit tests for the RelationshipController class.
 *
 * @group crm
 * @coversDefaultClass \Drupal\crm\Controller\RelationshipController
 */
class RelationshipControllerTest extends UnitTestCase {

  /**
   * The controller under test.
   *
   * @var \Drupal\crm\Controller\RelationshipController
   */
  protected $controller;

  /**
   * The entity type manager mock.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * The relationship service mock.
   *
   * @var \Drupal\crm\Relationship|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $relationshipService;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $container = new ContainerBuilder();

    $string_translation = $this->getStringTranslationStub();
    $container->set('string_translation', $string_translation);

    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $container->set('entity_type.manager', $this->entityTypeManager);

    $this->relationshipService = $this->createMock(Relationship::class);
    $container->set('crm.relationship', $this->relationshipService);

    \Drupal::setContainer($container);

    $this->controller = new RelationshipController($this->entityTypeManager, $this->relationshipService);
  }

  /**
   * Tests addPage returns empty message when no eligible types.
   *
   * @covers ::addPage
   */
  public function testAddPageReturnsEmptyMessageWhenNoEligibleTypes(): void {
    $contact = $this->createMock(CrmContactInterface::class);

    $this->relationshipService->expects($this->once())
      ->method('getEligibleRelationshipTypesForContact')
      ->with($contact)
      ->willReturn([]);

    $build = $this->controller->addPage($contact);

    $this->assertIsArray($build);
    $this->assertArrayHasKey('#markup', $build);
    $this->assertStringContainsString('no relationship types available', (string) $build['#markup']);
  }

  /**
   * Tests addPage returns themed list when eligible types exist.
   *
   * @covers ::addPage
   */
  public function testAddPageReturnsThemedListWhenEligibleTypesExist(): void {
    $contact = $this->createMock(CrmContactInterface::class);
    $contact->expects($this->any())
      ->method('id')
      ->willReturn('1');

    // Create a mock relationship type.
    $relationshipType = $this->createMock('Drupal\crm\CrmRelationshipTypeInterface');
    $relationshipType->expects($this->any())
      ->method('id')
      ->willReturn('test_type');
    $relationshipType->expects($this->any())
      ->method('label')
      ->willReturn('Test Type');
    $relationshipType->expects($this->any())
      ->method('get')
      ->willReturnMap([
        ['asymmetric', FALSE],
        ['label_a', 'Label A'],
        ['label_b', 'Label B'],
      ]);
    $relationshipType->expects($this->any())
      ->method('getDescription')
      ->willReturn('Test description');

    $eligibleTypes = [
      'test_type' => [
        'type' => $relationshipType,
        'positions' => ['a'],
      ],
    ];

    $this->relationshipService->expects($this->once())
      ->method('getEligibleRelationshipTypesForContact')
      ->with($contact)
      ->willReturn($eligibleTypes);

    $build = $this->controller->addPage($contact);

    $this->assertIsArray($build);
    $this->assertArrayHasKey('#theme', $build);
    $this->assertEquals('entity_add_list', $build['#theme']);
    $this->assertArrayHasKey('#bundles', $build);
    $this->assertArrayHasKey('#cache', $build);
    $this->assertContains('config:crm_relationship_type_list', $build['#cache']['tags']);
  }

  /**
   * Tests addPage shows both positions for asymmetric relationships.
   *
   * @covers ::addPage
   */
  public function testAddPageShowsBothPositionsForAsymmetricRelationships(): void {
    $contact = $this->createMock(CrmContactInterface::class);
    $contact->expects($this->any())
      ->method('id')
      ->willReturn('1');

    // Create a mock asymmetric relationship type with both positions.
    $relationshipType = $this->createMock('Drupal\crm\CrmRelationshipTypeInterface');
    $relationshipType->expects($this->any())
      ->method('id')
      ->willReturn('asymmetric_type');
    $relationshipType->expects($this->any())
      ->method('label')
      ->willReturn('Asymmetric Type');
    $relationshipType->expects($this->any())
      ->method('get')
      ->willReturnMap([
        ['asymmetric', TRUE],
        ['label_a', 'Is Employer of'],
        ['label_b', 'Is Employee of'],
      ]);
    $relationshipType->expects($this->any())
      ->method('getDescription')
      ->willReturn('Asymmetric relationship');

    $eligibleTypes = [
      'asymmetric_type' => [
        'type' => $relationshipType,
        'positions' => ['a', 'b'],
      ],
    ];

    $this->relationshipService->expects($this->once())
      ->method('getEligibleRelationshipTypesForContact')
      ->with($contact)
      ->willReturn($eligibleTypes);

    $build = $this->controller->addPage($contact);

    $this->assertIsArray($build);
    $this->assertArrayHasKey('#bundles', $build);

    // For asymmetric relationships with both positions,
    // we expect two entries: one for each position.
    $this->assertArrayHasKey('asymmetric_type_a', $build['#bundles']);
    $this->assertArrayHasKey('asymmetric_type_b', $build['#bundles']);

    // Check that each entry has the inverted label.
    $this->assertEquals('Is Employee of', $build['#bundles']['asymmetric_type_a']['label']);
    $this->assertEquals('Is Employer of', $build['#bundles']['asymmetric_type_b']['label']);
  }

  /**
   * Tests addPage shows single entry for symmetric relationships.
   *
   * @covers ::addPage
   */
  public function testAddPageShowsSingleEntryForSymmetricRelationships(): void {
    $contact = $this->createMock(CrmContactInterface::class);
    $contact->expects($this->any())
      ->method('id')
      ->willReturn('1');

    // Create a mock symmetric relationship type.
    $relationshipType = $this->createMock('Drupal\crm\CrmRelationshipTypeInterface');
    $relationshipType->expects($this->any())
      ->method('id')
      ->willReturn('symmetric_type');
    $relationshipType->expects($this->any())
      ->method('label')
      ->willReturn('Friend Of');
    $relationshipType->expects($this->any())
      ->method('get')
      ->willReturnMap([
        ['asymmetric', FALSE],
        ['label_a', 'Friend Of'],
        ['label_b', 'Friend Of'],
      ]);
    $relationshipType->expects($this->any())
      ->method('getDescription')
      ->willReturn('Symmetric relationship');

    $eligibleTypes = [
      'symmetric_type' => [
        'type' => $relationshipType,
        'positions' => ['a', 'b'],
      ],
    ];

    $this->relationshipService->expects($this->once())
      ->method('getEligibleRelationshipTypesForContact')
      ->with($contact)
      ->willReturn($eligibleTypes);

    $build = $this->controller->addPage($contact);

    $this->assertIsArray($build);
    $this->assertArrayHasKey('#bundles', $build);

    // For symmetric relationships, we expect only one entry.
    $this->assertArrayHasKey('symmetric_type', $build['#bundles']);
    $this->assertArrayNotHasKey('symmetric_type_a', $build['#bundles']);
    $this->assertArrayNotHasKey('symmetric_type_b', $build['#bundles']);
  }

  /**
   * Tests addPage falls back to type label when position label is empty.
   *
   * @covers ::addPage
   */
  public function testAddPageFallsBackToTypeLabelWhenPositionLabelEmpty(): void {
    $contact = $this->createMock(CrmContactInterface::class);
    $contact->expects($this->any())
      ->method('id')
      ->willReturn('1');

    // Create a mock relationship type with an empty label_b.
    // When position is 'a', the code looks for label_b, which is empty,
    // so it should fall back to the main type label.
    $relationshipType = $this->createMock('Drupal\crm\CrmRelationshipTypeInterface');
    $relationshipType->expects($this->any())
      ->method('id')
      ->willReturn('empty_label_type');
    $relationshipType->expects($this->any())
      ->method('label')
      ->willReturn('Fallback Label');
    $relationshipType->expects($this->any())
      ->method('get')
      ->willReturnMap([
        ['asymmetric', FALSE],
        ['label_a', 'Has Label A'],
        ['label_b', ''],
      ]);
    $relationshipType->expects($this->any())
      ->method('getDescription')
      ->willReturn('Type with empty label_b');

    $eligibleTypes = [
      'empty_label_type' => [
        'type' => $relationshipType,
        'positions' => ['a'],
      ],
    ];

    $this->relationshipService->expects($this->once())
      ->method('getEligibleRelationshipTypesForContact')
      ->with($contact)
      ->willReturn($eligibleTypes);

    $build = $this->controller->addPage($contact);

    $this->assertIsArray($build);
    $this->assertArrayHasKey('#bundles', $build);
    $this->assertArrayHasKey('empty_label_type', $build['#bundles']);

    // When label_b is empty and position is 'a', it should fall back
    // to the main type label.
    $this->assertEquals('Fallback Label', $build['#bundles']['empty_label_type']['label']);
  }

}
