<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Unit\Entity;

use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\State\StateInterface;
use Drupal\crm\Entity\ContactType;
use Drupal\Tests\UnitTestCase;

/**
 * Unit tests for the ContactType::isLocked() method.
 *
 * @group crm
 * @coversDefaultClass \Drupal\crm\Entity\ContactType
 */
class ContactTypeIsLockedTest extends UnitTestCase {

  /**
   * The state service mock.
   *
   * @var \Drupal\Core\State\StateInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $state;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->state = $this->createMock(StateInterface::class);

    $container = new ContainerBuilder();
    $container->set('state', $this->state);
    \Drupal::setContainer($container);
  }

  /**
   * Tests that isLocked returns FALSE when no state is set.
   *
   * @covers ::isLocked
   */
  public function testIsLockedReturnsFalseByDefault(): void {
    $this->state->expects($this->once())
      ->method('get')
      ->with('crm.contact_type.locked')
      ->willReturn(NULL);

    $contact_type = new ContactType([
      'id' => 'test_type',
      'label' => 'Test Type',
    ], 'crm_contact_type');

    $this->assertFalse($contact_type->isLocked());
  }

  /**
   * Tests that isLocked returns the locking module when locked.
   *
   * @covers ::isLocked
   */
  public function testIsLockedReturnsModuleNameWhenLocked(): void {
    $this->state->expects($this->once())
      ->method('get')
      ->with('crm.contact_type.locked')
      ->willReturn(['test_type' => 'crm']);

    $contact_type = new ContactType([
      'id' => 'test_type',
      'label' => 'Test Type',
    ], 'crm_contact_type');

    $this->assertEquals('crm', $contact_type->isLocked());
  }

  /**
   * Tests that isLocked returns FALSE for unlocked type.
   *
   * @covers ::isLocked
   */
  public function testIsLockedReturnsFalseForUnlockedType(): void {
    $this->state->expects($this->once())
      ->method('get')
      ->with('crm.contact_type.locked')
      ->willReturn(['other_type' => 'some_module']);

    $contact_type = new ContactType([
      'id' => 'test_type',
      'label' => 'Test Type',
    ], 'crm_contact_type');

    $this->assertFalse($contact_type->isLocked());
  }

}
