<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Unit\Entity;

use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\crm\CrmContactInterface;
use Drupal\crm\Entity\UserContactMapping;
use Drupal\Tests\UnitTestCase;
use Drupal\user\UserInterface;

/**
 * Unit tests for the UserContactMapping entity.
 *
 * @group crm
 * @coversDefaultClass \Drupal\crm\Entity\UserContactMapping
 */
class UserContactMappingTest extends UnitTestCase {

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $container = new ContainerBuilder();
    \Drupal::setContainer($container);
  }

  /**
   * Tests label() with valid user and contact.
   *
   * @covers ::label
   */
  public function testLabelWithValidUserAndContact(): void {
    $mockUser = $this->createMock(UserInterface::class);
    $mockUser->method('getAccountName')->willReturn('test_user');

    $mockContact = $this->createMock(CrmContactInterface::class);
    $mockContact->method('label')->willReturn('Test Contact');

    $entity = $this->getMockBuilder(UserContactMapping::class)
      ->disableOriginalConstructor()
      ->onlyMethods(['getUser', 'getContact'])
      ->getMock();

    $entity->method('getUser')->willReturn($mockUser);
    $entity->method('getContact')->willReturn($mockContact);

    $this->assertEquals('test_user - Test Contact', $entity->label());
  }

  /**
   * Tests label() with null user.
   *
   * @covers ::label
   */
  public function testLabelWithNullUser(): void {
    $mockContact = $this->createMock(CrmContactInterface::class);
    $mockContact->method('label')->willReturn('Test Contact');

    $entity = $this->getMockBuilder(UserContactMapping::class)
      ->disableOriginalConstructor()
      ->onlyMethods(['getUser', 'getContact'])
      ->getMock();

    $entity->method('getUser')->willReturn(NULL);
    $entity->method('getContact')->willReturn($mockContact);

    $this->assertEquals('Unknown User - Test Contact', $entity->label());
  }

  /**
   * Tests label() with null contact.
   *
   * @covers ::label
   */
  public function testLabelWithNullContact(): void {
    $mockUser = $this->createMock(UserInterface::class);
    $mockUser->method('getAccountName')->willReturn('test_user');

    $entity = $this->getMockBuilder(UserContactMapping::class)
      ->disableOriginalConstructor()
      ->onlyMethods(['getUser', 'getContact'])
      ->getMock();

    $entity->method('getUser')->willReturn($mockUser);
    $entity->method('getContact')->willReturn(NULL);

    $this->assertEquals('test_user - Unknown Contact', $entity->label());
  }

  /**
   * Tests label() with both null user and contact.
   *
   * @covers ::label
   */
  public function testLabelWithBothNull(): void {
    $entity = $this->getMockBuilder(UserContactMapping::class)
      ->disableOriginalConstructor()
      ->onlyMethods(['getUser', 'getContact'])
      ->getMock();

    $entity->method('getUser')->willReturn(NULL);
    $entity->method('getContact')->willReturn(NULL);

    $this->assertEquals('Unknown User - Unknown Contact', $entity->label());
  }

}
