<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Unit\Form;

use Drupal\Tests\UnitTestCase;
use Drupal\crm\Form\ContactDetailTypeForm;
use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\crm\Entity\ContactDetailType;

/**
 * Contact detail type form test.
 *
 * @group crm
 * @coversDefaultClass \Drupal\crm\Form\ContactDetailTypeForm
 * @uses \Drupal\crm\Form\ContactDetailTypeForm
 */
class ContactDetailTypeFormTest extends UnitTestCase {

  const SAVED_NEW = 1;
  const SAVED_UPDATED = 2;

  /**
   * The form under test.
   *
   * @var \Drupal\crm\Form\ContactDetailTypeForm
   */
  protected $form;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $messenger;

  /**
   * The contact detail type entity.
   *
   * @var \Drupal\crm\Entity\ContactDetailType|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entity;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $container = new ContainerBuilder();

    $string_translation = $this->getStringTranslationStub();
    $container->set('string_translation', $string_translation);

    $this->messenger = $this->createMock(MessengerInterface::class);
    $container->set('messenger', $this->messenger);

    \Drupal::setContainer($container);

    $this->form = ContactDetailTypeForm::create($container);

    $this->entity = $this->createMock(ContactDetailType::class);
    $this->form->setEntity($this->entity);
  }

  /**
   * Tests the create method.
   *
   * @covers ::create
   */
  public function testCreate(): void {
    $container = \Drupal::getContainer();
    $form = ContactDetailTypeForm::create($container);

    $this->assertInstanceOf(ContactDetailTypeForm::class, $form);
  }

  /**
   * Test the form id.
   *
   * @covers ::getFormId
   */
  public function testGetFormId(): void {
    $entity_type = $this->createMock(EntityTypeInterface::class);
    $entity_type->expects($this->once())
      ->method('hasKey')
      ->with('bundle')
      ->willReturn(FALSE);

    $this->entity->expects($this->once())
      ->method('getEntityTypeId')
      ->willReturn('crm_contact_detail_type');
    $this->entity->expects($this->once())
      ->method('getEntityType')
      ->willReturn($entity_type);

    $this->form->setOperation('edit');

    $this->assertEquals('crm_contact_detail_type_edit_form', $this->form->getFormId());
  }

  /**
   * Test the form build with add operation.
   *
   * @covers ::form
   */
  public function testFormWithAddOperation(): void {
    $entity_type = $this->createMock(EntityTypeInterface::class);
    $entity_type->expects($this->once())
      ->method('getKey')
      ->with('id')
      ->willReturn('id');

    $this->entity->expects($this->once())
      ->method('getEntityType')
      ->willReturn($entity_type);
    $this->entity->expects($this->any())
      ->method('label')
      ->willReturn('');
    $this->entity->expects($this->any())
      ->method('id')
      ->willReturn('');
    $this->entity->expects($this->any())
      ->method('get')
      ->with('description')
      ->willReturn('');

    $this->form->setOperation('add');

    $form_state = $this->createMock(FormStateInterface::class);
    $form = $this->form->form([], $form_state);

    $this->assertIsArray($form);
    $this->assertArrayHasKey('label', $form);
    $this->assertArrayHasKey('id', $form);
    $this->assertArrayHasKey('description', $form);

    // Test label field.
    $this->assertEquals('textfield', $form['label']['#type']);
    $this->assertEquals('', $form['label']['#default_value']);
    $this->assertTrue($form['label']['#required']);
    $this->assertArrayHasKey('#description', $form['label']);

    // Test id field (machine name).
    $this->assertEquals('machine_name', $form['id']['#type']);
    $this->assertEquals('', $form['id']['#default_value']);
    $this->assertEquals(32, $form['id']['#maxlength']);
    $this->assertArrayHasKey('#machine_name', $form['id']);
    $this->assertEquals([ContactDetailType::class, 'load'], $form['id']['#machine_name']['exists']);
    $this->assertEquals(['label'], $form['id']['#machine_name']['source']);

    // Test description field.
    $this->assertEquals('textarea', $form['description']['#type']);
    $this->assertEquals('', $form['description']['#default_value']);
    $this->assertArrayHasKey('#description', $form['description']);

    // Should not have title for add operation.
    $this->assertArrayNotHasKey('#title', $form);
  }

  /**
   * Test the form build with edit operation.
   *
   * @covers ::form
   */
  public function testFormWithEditOperation(): void {
    $entity_type = $this->createMock(EntityTypeInterface::class);
    $entity_type->expects($this->once())
      ->method('getKey')
      ->with('id')
      ->willReturn('id');

    $this->entity->expects($this->once())
      ->method('getEntityType')
      ->willReturn($entity_type);
    $this->entity->expects($this->any())
      ->method('label')
      ->willReturn('Test Contact Detail Type');
    $this->entity->expects($this->any())
      ->method('id')
      ->willReturn('test_contact_detail_type');
    $this->entity->expects($this->any())
      ->method('get')
      ->with('description')
      ->willReturn('Test description');

    $this->form->setOperation('edit');

    $form_state = $this->createMock(FormStateInterface::class);
    $form = $this->form->form([], $form_state);

    $this->assertIsArray($form);
    $this->assertArrayHasKey('#title', $form);
    $expected_title = $this->getStringTranslationStub()->translate('Edit %label CRM contact detail type', ['%label' => 'Test Contact Detail Type']);
    $this->assertEquals($expected_title, $form['#title']);

    // Test that fields have proper default values.
    $this->assertEquals('Test Contact Detail Type', $form['label']['#default_value']);
    $this->assertEquals('test_contact_detail_type', $form['id']['#default_value']);
    $this->assertEquals('Test description', $form['description']['#default_value']);
  }

  /**
   * Test the actions method.
   *
   * @covers ::actions
   */
  public function testActions(): void {
    $form_state = $this->createMock(FormStateInterface::class);

    // We need to set up the entity properly to call actions.
    $this->entity->expects($this->any())
      ->method('isNew')
      ->willReturn(FALSE);

    // Use Reflection to access protected method.
    $reflection = new \ReflectionClass(ContactDetailTypeForm::class);
    $method = $reflection->getMethod('actions');
    $method->setAccessible(TRUE);

    $actions = $method->invoke($this->form, [], $form_state);

    $this->assertIsArray($actions);
    $this->assertArrayHasKey('submit', $actions);
    $this->assertArrayHasKey('delete', $actions);

    $expected_submit = $this->getStringTranslationStub()->translate('Save CRM contact detail type');
    $expected_delete = $this->getStringTranslationStub()->translate('Delete CRM contact detail type');

    $this->assertEquals($expected_submit, $actions['submit']['#value']);
    $this->assertEquals($expected_delete, $actions['delete']['#value']);
  }

  /**
   * Test the save method with new entity.
   *
   * @covers ::save
   */
  public function testSaveNewEntity(): void {
    $form_state = $this->createMock(FormStateInterface::class);

    $this->entity->expects($this->any())
      ->method('label')
      ->willReturn('Test Contact Detail Type');
    $this->entity->expects($this->once())
      ->method('save')
      ->willReturn(self::SAVED_NEW);

    $url = $this->createMock('Drupal\Core\Url');
    $this->entity->expects($this->once())
      ->method('toUrl')
      ->with('collection')
      ->willReturn($url);

    $form_state->expects($this->once())
      ->method('setRedirectUrl')
      ->with($url);

    $this->messenger->expects($this->once())
      ->method('addStatus')
      ->with($this->getStringTranslationStub()->translate('The CRM contact detail type %label has been added.', ['%label' => 'Test Contact Detail Type']));

    $result = $this->form->save([], $form_state);
    $this->assertEquals(self::SAVED_NEW, $result);
  }

  /**
   * Test the save method with existing entity.
   *
   * @covers ::save
   */
  public function testSaveExistingEntity(): void {
    $form_state = $this->createMock(FormStateInterface::class);

    $this->entity->expects($this->any())
      ->method('label')
      ->willReturn('Updated Contact Detail Type');
    $this->entity->expects($this->once())
      ->method('save')
      ->willReturn(self::SAVED_UPDATED);

    $url = $this->createMock('Drupal\Core\Url');
    $this->entity->expects($this->once())
      ->method('toUrl')
      ->with('collection')
      ->willReturn($url);

    $form_state->expects($this->once())
      ->method('setRedirectUrl')
      ->with($url);

    $this->messenger->expects($this->once())
      ->method('addStatus')
      ->with($this->getStringTranslationStub()->translate('The CRM contact detail type %label has been updated.', ['%label' => 'Updated Contact Detail Type']));

    $result = $this->form->save([], $form_state);
    $this->assertEquals(self::SAVED_UPDATED, $result);
  }

}
