<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Unit\Form;

use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\crm\Form\UserContactMappingSettingsForm;
use Drupal\Tests\UnitTestCase;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Messenger\MessengerInterface;

/**
 * Unit tests for the UserContactMappingSettingsForm class.
 *
 * @group crm
 * @coversDefaultClass \Drupal\crm\Form\UserContactMappingSettingsForm
 * @uses \Drupal\crm\Form\UserContactMappingSettingsForm
 */
class UserContactMappingSettingsFormTest extends UnitTestCase {

  /**
   * The form under test.
   *
   * @var \Drupal\crm\Form\UserContactMappingSettingsForm
   */
  protected $form;

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $configFactory;

  /**
   * The config object.
   *
   * @var \Drupal\Core\Config\Config|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $config;

  /**
   * The typed data manager service.
   *
   * @var \Drupal\Core\Config\TypedConfigManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $typedData;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $messenger;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $container = new ContainerBuilder();

    $string_translation = $this->getStringTranslationStub();
    $container->set('string_translation', $string_translation);

    $this->typedData = $this->createMock(TypedConfigManagerInterface::class);
    $container->set('config.typed', $this->typedData);

    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $container->set('config.factory', $this->configFactory);

    $this->messenger = $this->createMock(MessengerInterface::class);
    $container->set('messenger', $this->messenger);

    \Drupal::setContainer($container);

    $this->config = $this->createMock(Config::class);

    $this->form = UserContactMappingSettingsForm::create($container);
  }

  /**
   * Tests the create method.
   *
   * @covers ::create
   */
  public function testCreate(): void {
    $container = \Drupal::getContainer();
    $form = UserContactMappingSettingsForm::create($container);

    $this->assertInstanceOf(UserContactMappingSettingsForm::class, $form);
  }

  /**
   * Tests the getFormId method.
   *
   * @covers ::getFormId
   */
  public function testGetFormId(): void {
    $this->assertEquals('crm_user_contact_mapping_settings', $this->form->getFormId());
  }

  /**
   * Tests the getEditableConfigNames method.
   *
   * @covers ::getEditableConfigNames
   */
  public function testGetEditableConfigNames(): void {

    // Use Reflection to get the private method.
    $reflection = new \ReflectionClass(UserContactMappingSettingsForm::class);
    $method = $reflection->getMethod('getEditableConfigNames');
    $method->setAccessible(TRUE);
    $result = $method->invoke($this->form);

    $expected = ['crm.user_contact_mapping.settings'];
    $this->assertEquals($expected, $result);
  }

  /**
   * Tests the buildForm method with default values.
   *
   * @covers ::buildForm
   */
  public function testBuildFormWithDefaultValues(): void {
    $this->configFactory->expects($this->any())
      ->method('get')
      ->with('crm.user_contact_mapping.settings')
      ->willReturn($this->config);

    $this->configFactory->expects($this->any())
      ->method('getEditable')
      ->with('crm.user_contact_mapping.settings')
      ->willReturn($this->config);

    $this->config->expects($this->exactly(3))
      ->method('get')
      ->willReturnMap([
        ['display_name', FALSE],
        ['auto_create_crm_user_contact_mapping', FALSE],
        ['auto_create_lookup_contact', FALSE],
      ]);

    $form_state = $this->createMock(FormStateInterface::class);
    $form = $this->form->buildForm([], $form_state);

    $this->assertIsArray($form);

    // Test display_name checkbox.
    $this->assertArrayHasKey('display_name', $form);
    $this->assertEquals('checkbox', $form['display_name']['#type']);
    $this->assertInstanceOf(TranslatableMarkup::class, $form['display_name']['#title']);
    $this->assertEquals('Display name', (string) $form['display_name']['#title']);
    $this->assertFalse($form['display_name']['#default_value']);
    $this->assertInstanceOf(TranslatableMarkup::class, $form['display_name']['#description']);
    $this->assertEquals('Override the User name with the CRM name.', (string) $form['display_name']['#description']);

    // Test create_event details section.
    $this->assertArrayHasKey('create_event', $form);
    $this->assertEquals('details', $form['create_event']['#type']);
    $this->assertInstanceOf(TranslatableMarkup::class, $form['create_event']['#title']);
    $this->assertEquals('Create event', (string) $form['create_event']['#title']);
    $this->assertTrue($form['create_event']['#open']);

    // Test auto_create_crm_user_contact_mapping checkbox.
    $this->assertArrayHasKey('auto_create_crm_user_contact_mapping', $form['create_event']);
    $this->assertEquals('checkbox', $form['create_event']['auto_create_crm_user_contact_mapping']['#type']);
    $this->assertInstanceOf(TranslatableMarkup::class, $form['create_event']['auto_create_crm_user_contact_mapping']['#title']);
    $this->assertEquals('Create CRM user contact mapping automatically', (string) $form['create_event']['auto_create_crm_user_contact_mapping']['#title']);
    $this->assertFalse($form['create_event']['auto_create_crm_user_contact_mapping']['#default_value']);
    $this->assertInstanceOf(TranslatableMarkup::class, $form['create_event']['auto_create_crm_user_contact_mapping']['#description']);
    $this->assertEquals('Automatically create a contact mapping when a user is created.', (string) $form['create_event']['auto_create_crm_user_contact_mapping']['#description']);

    // Test auto_create_lookup_contact checkbox.
    $this->assertArrayHasKey('auto_create_lookup_contact', $form['create_event']);
    $this->assertEquals('checkbox', $form['create_event']['auto_create_lookup_contact']['#type']);
    $this->assertInstanceOf(TranslatableMarkup::class, $form['create_event']['auto_create_lookup_contact']['#title']);
    $this->assertEquals('Lookup contact automatically', (string) $form['create_event']['auto_create_lookup_contact']['#title']);
    $this->assertFalse($form['create_event']['auto_create_lookup_contact']['#default_value']);
    $this->assertInstanceOf(TranslatableMarkup::class, $form['create_event']['auto_create_lookup_contact']['#description']);
    $this->assertEquals('Automatically create a lookup contact when a user is created.', (string) $form['create_event']['auto_create_lookup_contact']['#description']);

    // Test conditional states for auto_create_lookup_contact.
    $this->assertArrayHasKey('#states', $form['create_event']['auto_create_lookup_contact']);
    $expected_states = [
      'enabled' => [
        ':input[name="auto_create_crm_user_contact_mapping"]' => ['checked' => TRUE],
      ],
    ];
    $this->assertEquals($expected_states, $form['create_event']['auto_create_lookup_contact']['#states']);
  }

  /**
   * Tests the buildForm method with custom values.
   *
   * @covers ::buildForm
   */
  public function testBuildFormWithCustomValues(): void {
    $this->configFactory->expects($this->any())
      ->method('get')
      ->with('crm.user_contact_mapping.settings')
      ->willReturn($this->config);

    $this->configFactory->expects($this->any())
      ->method('getEditable')
      ->with('crm.user_contact_mapping.settings')
      ->willReturn($this->config);

    $this->config->expects($this->exactly(3))
      ->method('get')
      ->willReturnMap([
        ['display_name', TRUE],
        ['auto_create_crm_user_contact_mapping', TRUE],
        ['auto_create_lookup_contact', TRUE],
      ]);

    $form_state = $this->createMock(FormStateInterface::class);
    $form = $this->form->buildForm([], $form_state);

    $this->assertTrue($form['display_name']['#default_value']);
    $this->assertTrue($form['create_event']['auto_create_crm_user_contact_mapping']['#default_value']);
    $this->assertTrue($form['create_event']['auto_create_lookup_contact']['#default_value']);
  }

  /**
   * Tests the submitForm method.
   *
   * @covers ::submitForm
   */
  public function testSubmitForm(): void {
    $this->configFactory->expects($this->any())
      ->method('getEditable')
      ->with('crm.user_contact_mapping.settings')
      ->willReturn($this->config);

    $form_state = $this->createMock(FormStateInterface::class);
    $form_state->expects($this->exactly(3))
      ->method('getValue')
      ->willReturnMap([
        ['display_name', TRUE],
        ['auto_create_crm_user_contact_mapping', TRUE],
        ['auto_create_lookup_contact', FALSE],
      ]);

    // Use individual expectations instead of withConsecutive.
    $this->config->expects($this->exactly(3))
      ->method('set')
      ->willReturnSelf();

    $this->config->expects($this->once())
      ->method('save')
      ->willReturn(TRUE);

    $form = [];
    $result = $this->form->submitForm($form, $form_state);
  }

  /**
   * Tests the submitForm method with all false values.
   *
   * @covers ::submitForm
   */
  public function testSubmitFormWithAllFalseValues(): void {
    $this->configFactory->expects($this->any())
      ->method('getEditable')
      ->with('crm.user_contact_mapping.settings')
      ->willReturn($this->config);

    $form_state = $this->createMock(FormStateInterface::class);
    $form_state->expects($this->exactly(3))
      ->method('getValue')
      ->willReturnMap([
        ['display_name', FALSE],
        ['auto_create_crm_user_contact_mapping', FALSE],
        ['auto_create_lookup_contact', FALSE],
      ]);

    $this->config->expects($this->exactly(3))
      ->method('set')
      ->willReturnSelf();

    $this->config->expects($this->once())
      ->method('save')
      ->willReturnSelf();

    $form = [];
    $result = $this->form->submitForm($form, $form_state);

    $this->assertNull($result);
  }

  /**
   * Tests the submitForm method with mixed values.
   *
   * @covers ::submitForm
   */
  public function testSubmitFormWithMixedValues(): void {
    $this->configFactory->expects($this->any())
      ->method('getEditable')
      ->with('crm.user_contact_mapping.settings')
      ->willReturn($this->config);

    $form_state = $this->createMock(FormStateInterface::class);
    $form_state->expects($this->exactly(3))
      ->method('getValue')
      ->willReturnMap([
        ['display_name', TRUE],
        ['auto_create_crm_user_contact_mapping', FALSE],
        ['auto_create_lookup_contact', TRUE],
      ]);

    $this->config->expects($this->exactly(3))
      ->method('set')
      ->willReturnSelf();

    $this->config->expects($this->once())
      ->method('save')
      ->willReturnSelf();

    $form = [];
    $result = $this->form->submitForm($form, $form_state);
  }

  /**
   * Tests that form structure includes all required elements.
   *
   * @covers ::buildForm
   */
  public function testFormStructure(): void {
    $this->configFactory->expects($this->any())
      ->method('get')
      ->with('crm.user_contact_mapping.settings')
      ->willReturn($this->config);

    $this->configFactory->expects($this->any())
      ->method('getEditable')
      ->with('crm.user_contact_mapping.settings')
      ->willReturn($this->config);

    $this->config->expects($this->any())
      ->method('get')
      ->willReturn(FALSE);

    $form_state = $this->createMock(FormStateInterface::class);
    $form = $this->form->buildForm([], $form_state);

    // Verify that parent form is called and includes submit actions.
    $this->assertArrayHasKey('actions', $form);
    $this->assertArrayHasKey('submit', $form['actions']);

    // Verify form structure hierarchy.
    $this->assertArrayHasKey('display_name', $form);
    $this->assertArrayHasKey('create_event', $form);
    $this->assertArrayHasKey('auto_create_crm_user_contact_mapping', $form['create_event']);
    $this->assertArrayHasKey('auto_create_lookup_contact', $form['create_event']);
  }

}
