<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Unit\Hook;

use Drupal\Core\Config\ConfigInstallerInterface;
use Drupal\Core\Config\FileStorage;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\crm\Hook\InstallOptionalConfigHooks;
use Drupal\Tests\UnitTestCase;

/**
 * Tests InstallOptionalConfigHooks functionality.
 *
 * @group crm
 * @coversDefaultClass \Drupal\crm\Hook\InstallOptionalConfigHooks
 */
class InstallOptionalConfigHooksTest extends UnitTestCase {

  /**
   * Mock config installer service.
   *
   * @var \Drupal\Core\Config\ConfigInstallerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $mockConfigInstaller;

  /**
   * Mock module extension list service.
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $mockModuleExtensionList;

  /**
   * The InstallOptionalConfigHooks instance under test.
   *
   * @var \Drupal\crm\Hook\InstallOptionalConfigHooks
   */
  protected $installOptionalConfigHooks;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create mock services.
    $this->mockConfigInstaller = $this->createMock(ConfigInstallerInterface::class);
    $this->mockModuleExtensionList = $this->createMock(ModuleExtensionList::class);

    // Set up module extension list to return a path.
    $this->mockModuleExtensionList
      ->method('getPath')
      ->with('crm')
      ->willReturn('modules/contrib/crm');

    // Create the hooks instance.
    $this->installOptionalConfigHooks = new InstallOptionalConfigHooks(
      $this->mockConfigInstaller,
      $this->mockModuleExtensionList
    );
  }

  /**
   * Tests modulesInstalled with comment module.
   *
   * @covers ::modulesInstalled
   */
  public function testModulesInstalledWithComment(): void {
    $this->mockConfigInstaller
      ->expects($this->once())
      ->method('installOptionalConfig')
      ->with(
        $this->isInstanceOf(FileStorage::class),
        $this->equalTo(['module' => 'comment'])
      );

    $this->installOptionalConfigHooks->modulesInstalled(['comment']);
  }

  /**
   * Tests modulesInstalled with search module.
   *
   * @covers ::modulesInstalled
   */
  public function testModulesInstalledWithSearch(): void {
    $this->mockConfigInstaller
      ->expects($this->once())
      ->method('installOptionalConfig')
      ->with(
        $this->isInstanceOf(FileStorage::class),
        $this->equalTo(['module' => 'search'])
      );

    $this->installOptionalConfigHooks->modulesInstalled(['search']);
  }

  /**
   * Tests modulesInstalled with an unrelated module.
   *
   * @covers ::modulesInstalled
   */
  public function testModulesInstalledWithUnrelatedModule(): void {
    $this->mockConfigInstaller
      ->expects($this->never())
      ->method('installOptionalConfig');

    $this->installOptionalConfigHooks->modulesInstalled(['views']);
  }

  /**
   * Tests modulesInstalled with multiple integration modules.
   *
   * @covers ::modulesInstalled
   */
  public function testModulesInstalledWithMultipleIntegrations(): void {
    $this->mockConfigInstaller
      ->expects($this->exactly(2))
      ->method('installOptionalConfig')
      ->with(
        $this->isInstanceOf(FileStorage::class),
        $this->logicalOr(
          $this->equalTo(['module' => 'comment']),
          $this->equalTo(['module' => 'search'])
        )
      );

    $this->installOptionalConfigHooks->modulesInstalled(['comment', 'search']);
  }

  /**
   * Tests modulesInstalled with mixed modules (integration and unrelated).
   *
   * @covers ::modulesInstalled
   */
  public function testModulesInstalledWithMixedModules(): void {
    $this->mockConfigInstaller
      ->expects($this->once())
      ->method('installOptionalConfig')
      ->with(
        $this->isInstanceOf(FileStorage::class),
        $this->equalTo(['module' => 'comment'])
      );

    $this->installOptionalConfigHooks->modulesInstalled(['views', 'comment', 'node']);
  }

  /**
   * Tests modulesInstalled with empty modules array.
   *
   * @covers ::modulesInstalled
   */
  public function testModulesInstalledWithEmptyArray(): void {
    $this->mockConfigInstaller
      ->expects($this->never())
      ->method('installOptionalConfig');

    $this->installOptionalConfigHooks->modulesInstalled([]);
  }

}
