<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Unit\Plugin\Field\FieldFormatter;

use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemList;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\crm\CrmRelationshipTypeInterface;
use Drupal\crm\Plugin\Field\FieldFormatter\RelationshipStatisticsFormatter;
use Drupal\Tests\UnitTestCase;

/**
 * Tests the RelationshipStatisticsFormatter.
 *
 * @group crm
 * @coversDefaultClass \Drupal\crm\Plugin\Field\FieldFormatter\RelationshipStatisticsFormatter
 */
class RelationshipStatisticsFormatterTest extends UnitTestCase {

  /**
   * The formatter under test.
   *
   * @var \Drupal\crm\Plugin\Field\FieldFormatter\RelationshipStatisticsFormatter
   */
  protected RelationshipStatisticsFormatter $formatter;

  /**
   * Mock entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * Mock relationship types.
   *
   * @var array
   */
  protected array $relationshipTypes = [];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Set up the container with string translation.
    $container = new ContainerBuilder();
    $translation = $this->createMock(TranslationInterface::class);
    $translation->method('translateString')
      ->willReturnCallback(function ($string) {
        return $string->getUntranslatedString();
      });
    $container->set('string_translation', $translation);
    \Drupal::setContainer($container);

    // Create mock relationship types.
    $friends_type = $this->createMock(CrmRelationshipTypeInterface::class);
    $friends_type->method('label')->willReturn('Friends');
    $friends_type->method('get')->willReturnCallback(function ($key) {
      return match ($key) {
        'label_a' => 'Friend',
        'label_a_plural' => 'Friends',
        'label_b' => 'Friend',
        'label_b_plural' => 'Friends',
        'asymmetric' => FALSE,
        default => NULL,
      };
    });

    $parent_child_type = $this->createMock(CrmRelationshipTypeInterface::class);
    $parent_child_type->method('label')->willReturn('Parent-Child');
    $parent_child_type->method('get')->willReturnCallback(function ($key) {
      return match ($key) {
        'label_a' => 'Parent',
        'label_a_plural' => 'Parents',
        'label_b' => 'Child',
        'label_b_plural' => 'Children',
        'asymmetric' => TRUE,
        default => NULL,
      };
    });

    $this->relationshipTypes = [
      'friends' => $friends_type,
      'parent_child' => $parent_child_type,
    ];

    // Create mock entity storage.
    $storage = $this->createMock(EntityStorageInterface::class);
    $storage->method('loadMultiple')
      ->willReturn($this->relationshipTypes);

    // Create mock entity type manager.
    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->entityTypeManager->method('getStorage')
      ->with('crm_relationship_type')
      ->willReturn($storage);

    // Create the formatter with default settings.
    $this->formatter = $this->createFormatterWithSettings([]);
  }

  /**
   * Creates a formatter instance with the specified settings.
   *
   * @param array $settings
   *   The settings to apply to the formatter.
   *
   * @return \Drupal\crm\Plugin\Field\FieldFormatter\RelationshipStatisticsFormatter
   *   The formatter instance.
   */
  protected function createFormatterWithSettings(array $settings): RelationshipStatisticsFormatter {
    $field_definition = $this->createMock(FieldDefinitionInterface::class);

    $default_settings = RelationshipStatisticsFormatter::defaultSettings();
    $merged_settings = array_merge($default_settings, $settings);

    return new RelationshipStatisticsFormatter(
      'crm_relationship_statistics_default',
      [],
      $field_definition,
      $merged_settings,
      'Test label',
      'default',
      [],
      $this->entityTypeManager,
    );
  }

  /**
   * Tests default settings.
   *
   * @covers ::defaultSettings
   */
  public function testDefaultSettings(): void {
    $settings = RelationshipStatisticsFormatter::defaultSettings();

    $this->assertArrayHasKey('format', $settings);
    $this->assertEquals('label_count', $settings['format']);

    $this->assertArrayHasKey('label_position', $settings);
    $this->assertEquals('opposite', $settings['label_position']);

    $this->assertArrayHasKey('sort_by', $settings);
    $this->assertEquals('none', $settings['sort_by']);

    $this->assertArrayHasKey('sort_order', $settings);
    $this->assertEquals('asc', $settings['sort_order']);
  }

  /**
   * Tests viewElements with symmetric relationship type.
   *
   * @covers ::viewElements
   */
  public function testViewElementsSymmetric(): void {
    $item = new \stdClass();
    $item->value = 'friends';
    $item->count = 5;

    $items = $this->createMock(FieldItemList::class);
    $items->method('getIterator')
      ->willReturn(new \ArrayIterator([$item]));

    $elements = $this->formatter->viewElements($items, 'en');

    $this->assertCount(1, $elements);
    $this->assertStringContainsString('Friends', (string) $elements[0]['#markup']);
    $this->assertStringContainsString('5', (string) $elements[0]['#markup']);
  }

  /**
   * Tests viewElements asymmetric relationship type using default settings.
   *
   * With the default label_position of 'opposite', position 'a' shows label_b
   * and position 'b' shows label_a.
   *
   * @covers ::viewElements
   */
  public function testViewElementsAsymmetric(): void {
    $item_a = new \stdClass();
    $item_a->value = 'parent_child:a';
    $item_a->count = 2;

    $item_b = new \stdClass();
    $item_b->value = 'parent_child:b';
    $item_b->count = 3;

    $items = $this->createMock(FieldItemList::class);
    $items->method('getIterator')
      ->willReturn(new \ArrayIterator([$item_a, $item_b]));

    $elements = $this->formatter->viewElements($items, 'en');

    $this->assertCount(2, $elements);
    // With 'opposite' label position (default):
    // Position 'a' shows label_b = "Child" (what the related contacts are).
    $this->assertStringContainsString('Child', (string) $elements[0]['#markup']);
    $this->assertStringContainsString('2', (string) $elements[0]['#markup']);
    // Position 'b' shows label_a = "Parent" (what the related contacts are).
    $this->assertStringContainsString('Parent', (string) $elements[1]['#markup']);
    $this->assertStringContainsString('3', (string) $elements[1]['#markup']);
  }

  /**
   * Tests viewElements with unknown relationship type.
   *
   * @covers ::viewElements
   */
  public function testViewElementsUnknownType(): void {
    $item = new \stdClass();
    $item->value = 'unknown_type';
    $item->count = 1;

    $items = $this->createMock(FieldItemList::class);
    $items->method('getIterator')
      ->willReturn(new \ArrayIterator([$item]));

    $elements = $this->formatter->viewElements($items, 'en');

    $this->assertCount(1, $elements);
    // Unknown type should fall back to the raw key.
    $this->assertStringContainsString('unknown_type', (string) $elements[0]['#markup']);
  }

  /**
   * Tests viewElements with 'opposite' label position for asymmetric types.
   *
   * @covers ::viewElements
   */
  public function testViewElementsOppositeLabelPosition(): void {
    $formatter = $this->createFormatterWithSettings([
      'label_position' => 'opposite',
    ]);

    $item_a = new \stdClass();
    $item_a->value = 'parent_child:a';
    $item_a->count = 2;

    $item_b = new \stdClass();
    $item_b->value = 'parent_child:b';
    $item_b->count = 3;

    $items = $this->createMock(FieldItemList::class);
    $items->method('getIterator')
      ->willReturn(new \ArrayIterator([$item_a, $item_b]));

    $elements = $formatter->viewElements($items, 'en');

    $this->assertCount(2, $elements);
    // Position 'a' shows label_b = "Child".
    $this->assertStringContainsString('Child', (string) $elements[0]['#markup']);
    // Position 'b' shows label_a = "Parent".
    $this->assertStringContainsString('Parent', (string) $elements[1]['#markup']);
  }

  /**
   * Tests viewElements with 'same' label position for asymmetric types.
   *
   * @covers ::viewElements
   */
  public function testViewElementsSameLabelPosition(): void {
    $formatter = $this->createFormatterWithSettings([
      'label_position' => 'same',
    ]);

    $item_a = new \stdClass();
    $item_a->value = 'parent_child:a';
    $item_a->count = 2;

    $item_b = new \stdClass();
    $item_b->value = 'parent_child:b';
    $item_b->count = 3;

    $items = $this->createMock(FieldItemList::class);
    $items->method('getIterator')
      ->willReturn(new \ArrayIterator([$item_a, $item_b]));

    $elements = $formatter->viewElements($items, 'en');

    $this->assertCount(2, $elements);
    // Position 'a' shows label_a = "Parent".
    $this->assertStringContainsString('Parent', (string) $elements[0]['#markup']);
    // Position 'b' shows label_b = "Child".
    $this->assertStringContainsString('Child', (string) $elements[1]['#markup']);
  }

  /**
   * Tests that symmetric relationships use main label regardless of setting.
   *
   * @covers ::viewElements
   */
  public function testSymmetricLabelPositionUnchanged(): void {
    // Test with 'opposite' setting.
    $formatter_opposite = $this->createFormatterWithSettings([
      'label_position' => 'opposite',
    ]);

    $item = new \stdClass();
    $item->value = 'friends';
    $item->count = 5;

    $items = $this->createMock(FieldItemList::class);
    $items->method('getIterator')
      ->willReturn(new \ArrayIterator([$item]));

    $elements = $formatter_opposite->viewElements($items, 'en');
    $this->assertStringContainsString('Friends', (string) $elements[0]['#markup']);

    // Test with 'same' setting.
    $formatter_same = $this->createFormatterWithSettings([
      'label_position' => 'same',
    ]);

    $items_same = $this->createMock(FieldItemList::class);
    $items_same->method('getIterator')
      ->willReturn(new \ArrayIterator([$item]));

    $elements_same = $formatter_same->viewElements($items_same, 'en');
    $this->assertStringContainsString('Friends', (string) $elements_same[0]['#markup']);
  }

  /**
   * Tests sorting by label in ascending order.
   *
   * @covers ::viewElements
   */
  public function testSortByLabelAscending(): void {
    $formatter = $this->createFormatterWithSettings([
      'label_position' => 'same',
      'sort_by' => 'label',
      'sort_order' => 'asc',
    ]);

    // Create items that will have labels: Parent, Child, Friends.
    $item_parent = new \stdClass();
    $item_parent->value = 'parent_child:a';
    $item_parent->count = 2;

    $item_child = new \stdClass();
    $item_child->value = 'parent_child:b';
    $item_child->count = 3;

    $item_friends = new \stdClass();
    $item_friends->value = 'friends';
    $item_friends->count = 5;

    $items = $this->createMock(FieldItemList::class);
    $items->method('getIterator')
      ->willReturn(new \ArrayIterator([$item_parent, $item_child, $item_friends]));

    $elements = $formatter->viewElements($items, 'en');

    $this->assertCount(3, $elements);
    // Sorted ascending: Child, Friends, Parent.
    $this->assertStringContainsString('Child', (string) $elements[0]['#markup']);
    $this->assertStringContainsString('Friends', (string) $elements[1]['#markup']);
    $this->assertStringContainsString('Parent', (string) $elements[2]['#markup']);
  }

  /**
   * Tests sorting by label in descending order.
   *
   * @covers ::viewElements
   */
  public function testSortByLabelDescending(): void {
    $formatter = $this->createFormatterWithSettings([
      'label_position' => 'same',
      'sort_by' => 'label',
      'sort_order' => 'desc',
    ]);

    $item_parent = new \stdClass();
    $item_parent->value = 'parent_child:a';
    $item_parent->count = 2;

    $item_child = new \stdClass();
    $item_child->value = 'parent_child:b';
    $item_child->count = 3;

    $item_friends = new \stdClass();
    $item_friends->value = 'friends';
    $item_friends->count = 5;

    $items = $this->createMock(FieldItemList::class);
    $items->method('getIterator')
      ->willReturn(new \ArrayIterator([$item_parent, $item_child, $item_friends]));

    $elements = $formatter->viewElements($items, 'en');

    $this->assertCount(3, $elements);
    // Sorted descending: Parent, Friends, Child.
    $this->assertStringContainsString('Parent', (string) $elements[0]['#markup']);
    $this->assertStringContainsString('Friends', (string) $elements[1]['#markup']);
    $this->assertStringContainsString('Child', (string) $elements[2]['#markup']);
  }

  /**
   * Tests sorting by count in ascending order.
   *
   * @covers ::viewElements
   */
  public function testSortByCountAscending(): void {
    $formatter = $this->createFormatterWithSettings([
      'sort_by' => 'count',
      'sort_order' => 'asc',
    ]);

    $item_high = new \stdClass();
    $item_high->value = 'friends';
    $item_high->count = 10;

    $item_low = new \stdClass();
    $item_low->value = 'parent_child:a';
    $item_low->count = 1;

    $item_mid = new \stdClass();
    $item_mid->value = 'parent_child:b';
    $item_mid->count = 5;

    $items = $this->createMock(FieldItemList::class);
    $items->method('getIterator')
      ->willReturn(new \ArrayIterator([$item_high, $item_low, $item_mid]));

    $elements = $formatter->viewElements($items, 'en');

    $this->assertCount(3, $elements);
    // Sorted by count ascending: 1, 5, 10.
    $this->assertStringContainsString('1', (string) $elements[0]['#markup']);
    $this->assertStringContainsString('5', (string) $elements[1]['#markup']);
    $this->assertStringContainsString('10', (string) $elements[2]['#markup']);
  }

  /**
   * Tests sorting by count in descending order.
   *
   * @covers ::viewElements
   */
  public function testSortByCountDescending(): void {
    $formatter = $this->createFormatterWithSettings([
      'sort_by' => 'count',
      'sort_order' => 'desc',
    ]);

    $item_high = new \stdClass();
    $item_high->value = 'friends';
    $item_high->count = 10;

    $item_low = new \stdClass();
    $item_low->value = 'parent_child:a';
    $item_low->count = 1;

    $item_mid = new \stdClass();
    $item_mid->value = 'parent_child:b';
    $item_mid->count = 5;

    $items = $this->createMock(FieldItemList::class);
    $items->method('getIterator')
      ->willReturn(new \ArrayIterator([$item_high, $item_low, $item_mid]));

    $elements = $formatter->viewElements($items, 'en');

    $this->assertCount(3, $elements);
    // Sorted by count descending: 10, 5, 1.
    $this->assertStringContainsString('10', (string) $elements[0]['#markup']);
    $this->assertStringContainsString('5', (string) $elements[1]['#markup']);
    $this->assertStringContainsString('1', (string) $elements[2]['#markup']);
  }

  /**
   * Tests that no sorting maintains original order.
   *
   * @covers ::viewElements
   */
  public function testNoSorting(): void {
    $formatter = $this->createFormatterWithSettings([
      'sort_by' => 'none',
    ]);

    // Create items in a specific order.
    $item1 = new \stdClass();
    $item1->value = 'parent_child:b';
    $item1->count = 5;

    $item2 = new \stdClass();
    $item2->value = 'friends';
    $item2->count = 1;

    $item3 = new \stdClass();
    $item3->value = 'parent_child:a';
    $item3->count = 10;

    $items = $this->createMock(FieldItemList::class);
    $items->method('getIterator')
      ->willReturn(new \ArrayIterator([$item1, $item2, $item3]));

    $elements = $formatter->viewElements($items, 'en');

    $this->assertCount(3, $elements);
    // Original order should be maintained: parent_child:b (5), friends (1),
    // parent_child:a (10).
    // With 'opposite' label position: Parent, Friends, Child.
    $this->assertStringContainsString('5', (string) $elements[0]['#markup']);
    $this->assertStringContainsString('1', (string) $elements[1]['#markup']);
    $this->assertStringContainsString('10', (string) $elements[2]['#markup']);
  }

  /**
   * Tests that singular label is used when count is 1.
   *
   * @covers ::viewElements
   */
  public function testViewElementsSingularLabel(): void {
    $item = new \stdClass();
    $item->value = 'parent_child:a';
    $item->count = 1;

    $items = $this->createMock(FieldItemList::class);
    $items->method('getIterator')
      ->willReturn(new \ArrayIterator([$item]));

    $elements = $this->formatter->viewElements($items, 'en');

    $this->assertCount(1, $elements);
    // With 'opposite' label position (default), position 'a' shows label_b.
    // Count is 1, so singular "Child" should be used.
    $this->assertStringContainsString('Child', (string) $elements[0]['#markup']);
    $this->assertStringNotContainsString('Children', (string) $elements[0]['#markup']);
  }

  /**
   * Tests that plural label is used when count is greater than 1.
   *
   * @covers ::viewElements
   */
  public function testViewElementsPluralLabel(): void {
    $item = new \stdClass();
    $item->value = 'parent_child:a';
    $item->count = 3;

    $items = $this->createMock(FieldItemList::class);
    $items->method('getIterator')
      ->willReturn(new \ArrayIterator([$item]));

    $elements = $this->formatter->viewElements($items, 'en');

    $this->assertCount(1, $elements);
    // With 'opposite', position 'a' shows label_b_plural.
    // Count is 3, so plural "Children" should be used.
    $this->assertStringContainsString('Children', (string) $elements[0]['#markup']);
  }

  /**
   * Tests that plural label is used when count is 0.
   *
   * @covers ::viewElements
   */
  public function testViewElementsZeroCountUsesPlural(): void {
    $item = new \stdClass();
    $item->value = 'parent_child:b';
    $item->count = 0;

    $items = $this->createMock(FieldItemList::class);
    $items->method('getIterator')
      ->willReturn(new \ArrayIterator([$item]));

    $elements = $this->formatter->viewElements($items, 'en');

    $this->assertCount(1, $elements);
    // With 'opposite', position 'b' shows label_a_plural.
    // Count is 0, so plural "Parents" should be used (0 is not 1).
    $this->assertStringContainsString('Parents', (string) $elements[0]['#markup']);
  }

}
