<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Unit\Plugin\Validation\Constraint;

use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\Query\QueryInterface;
use Drupal\crm\Plugin\Validation\Constraint\RelationshipLimitConstraint;
use Drupal\crm\Plugin\Validation\Constraint\RelationshipLimitConstraintValidator;
use Drupal\Tests\UnitTestCase;
use Symfony\Component\Validator\Context\ExecutionContextInterface;
use Symfony\Component\Validator\Violation\ConstraintViolationBuilderInterface;

/**
 * Tests the RelationshipLimitConstraintValidator.
 *
 * @group crm
 * @coversDefaultClass \Drupal\crm\Plugin\Validation\Constraint\RelationshipLimitConstraintValidator
 */
class RelationshipLimitConstraintValidatorTest extends UnitTestCase {

  /**
   * The validator under test.
   *
   * @var \Drupal\crm\Plugin\Validation\Constraint\RelationshipLimitConstraintValidator
   */
  protected $validator;

  /**
   * The constraint.
   *
   * @var \Drupal\crm\Plugin\Validation\Constraint\RelationshipLimitConstraint
   */
  protected $constraint;

  /**
   * The execution context.
   *
   * @var \Symfony\Component\Validator\Context\ExecutionContextInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $context;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->validator = new RelationshipLimitConstraintValidator($this->entityTypeManager);
    $this->constraint = new RelationshipLimitConstraint();
    $this->context = $this->createMock(ExecutionContextInterface::class);
    $this->validator->initialize($this->context);
  }

  /**
   * Tests validation passes when no limits are configured.
   *
   * @covers ::validate
   */
  public function testValidatePassesWithNoLimits(): void {
    $entity = $this->createRelationshipEntity(1, 2, NULL, NULL, FALSE, 'test_type');

    $this->context->expects($this->never())
      ->method('buildViolation');

    $this->validator->validate($entity, $this->constraint);
  }

  /**
   * Tests validation passes when under the limit.
   *
   * @covers ::validate
   */
  public function testValidatePassesWhenUnderLimit(): void {
    $entity = $this->createRelationshipEntity(1, 2, 5, 5, FALSE, 'test_type', NULL);

    // Return 0 relationships for contact A at delta 0.
    $this->setupQueryMock(0, 1, 0);

    $this->context->expects($this->never())
      ->method('buildViolation');

    $this->validator->validate($entity, $this->constraint);
  }

  /**
   * Tests validation fails when limit A is exceeded.
   *
   * @covers ::validate
   */
  public function testValidateFailsWhenLimit0Exceeded(): void {
    $entity = $this->createRelationshipEntity(1, 2, 2, NULL, FALSE, 'test_type', NULL);

    // Return 2 relationships for contact A (id=1) at delta 0.
    $this->setupQueryMock(2, 1, 0);

    $violationBuilder = $this->createMock(ConstraintViolationBuilderInterface::class);
    $violationBuilder->expects($this->once())
      ->method('atPath')
      ->with('contact_a')
      ->willReturnSelf();
    $violationBuilder->expects($this->exactly(4))
      ->method('setParameter')
      ->willReturnSelf();
    $violationBuilder->expects($this->once())
      ->method('addViolation');

    $this->context->expects($this->once())
      ->method('buildViolation')
      ->with($this->constraint->limitExceededMessageA)
      ->willReturn($violationBuilder);

    $this->validator->validate($entity, $this->constraint);
  }

  /**
   * Tests validation fails when limit B is exceeded.
   *
   * @covers ::validate
   */
  public function testValidateFailsWhenLimit1Exceeded(): void {
    $entity = $this->createRelationshipEntity(1, 2, NULL, 1, FALSE, 'test_type', NULL);

    // Return 1 relationship for contact B (id=2) at delta 1.
    $this->setupQueryMock(1, 2, 1);

    $violationBuilder = $this->createMock(ConstraintViolationBuilderInterface::class);
    $violationBuilder->expects($this->once())
      ->method('atPath')
      ->with('contact_b')
      ->willReturnSelf();
    $violationBuilder->expects($this->exactly(4))
      ->method('setParameter')
      ->willReturnSelf();
    $violationBuilder->expects($this->once())
      ->method('addViolation');

    $this->context->expects($this->once())
      ->method('buildViolation')
      ->with($this->constraint->limitExceededMessageB)
      ->willReturn($violationBuilder);

    $this->validator->validate($entity, $this->constraint);
  }

  /**
   * Tests validation when relationship type is null.
   *
   * @covers ::validate
   */
  public function testValidateWithNullRelationshipType(): void {
    $entity = $this->createRelationshipEntityWithNullType();

    $this->context->expects($this->never())
      ->method('buildViolation');

    $this->validator->validate($entity, $this->constraint);
  }

  /**
   * Sets up the query mock to return relationships with the specified count.
   *
   * @param int $count
   *   The number of relationships to return at the correct delta position.
   * @param int $contact_id
   *   The contact ID to match (defaults to 1 for contact A, 2 for contact B).
   * @param int $delta
   *   The delta position (0 for contact_a, 1 for contact_b).
   */
  protected function setupQueryMock(int $count, int $contact_id = 1, int $delta = 0): void {
    // Generate relationship IDs.
    $relationship_ids = [];
    for ($i = 1; $i <= $count; $i++) {
      $relationship_ids[$i] = $i;
    }

    $query = $this->createMock(QueryInterface::class);
    $query->expects($this->any())
      ->method('condition')
      ->willReturnSelf();
    $query->expects($this->any())
      ->method('accessCheck')
      ->willReturnSelf();
    $query->expects($this->any())
      ->method('execute')
      ->willReturn($relationship_ids);

    // Create mock relationships with contacts at correct positions.
    $relationships = [];
    foreach ($relationship_ids as $id) {
      $contacts_field = $this->createMock('Drupal\Core\Field\EntityReferenceFieldItemListInterface');
      $contacts_value = [
        ['target_id' => $delta === 0 ? (string) $contact_id : '999'],
        ['target_id' => $delta === 1 ? (string) $contact_id : '999'],
      ];
      $contacts_field->expects($this->any())
        ->method('getValue')
        ->willReturn($contacts_value);

      $relationship = $this->createMock('Drupal\crm\CrmRelationshipInterface');
      $relationship->expects($this->any())
        ->method('get')
        ->with('contacts')
        ->willReturn($contacts_field);

      $relationships[$id] = $relationship;
    }

    $storage = $this->createMock(EntityStorageInterface::class);
    $storage->expects($this->any())
      ->method('getQuery')
      ->willReturn($query);
    $storage->expects($this->any())
      ->method('loadMultiple')
      ->willReturn($relationships);

    $this->entityTypeManager->expects($this->any())
      ->method('getStorage')
      ->with('crm_relationship')
      ->willReturn($storage);
  }

  /**
   * Creates a mock relationship entity with the specified parameters.
   *
   * @param int $contact_a_id
   *   The ID of contact A.
   * @param int $contact_b_id
   *   The ID of contact B.
   * @param int|null $limit_a
   *   The limit for contact A position.
   * @param int|null $limit_b
   *   The limit for contact B position.
   * @param bool $limit_active_only
   *   Whether to only count active relationships.
   * @param string $relationship_type_id
   *   The relationship type ID.
   * @param int|string|null $entity_id
   *   The entity ID (null for new entities).
   *
   * @return \Drupal\crm\CrmRelationshipInterface|\PHPUnit\Framework\MockObject\MockObject
   *   The mock entity.
   */
  protected function createRelationshipEntity(
    int $contact_a_id,
    int $contact_b_id,
    ?int $limit_a,
    ?int $limit_b,
    bool $limit_active_only,
    string $relationship_type_id,
    int|string|null $entity_id = NULL,
  ) {
    // Create mock contacts.
    $contact_a = $this->createMock('Drupal\crm\Entity\Contact');
    $contact_a->expects($this->any())
      ->method('id')
      ->willReturn((string) $contact_a_id);

    $contact_b = $this->createMock('Drupal\crm\Entity\Contact');
    $contact_b->expects($this->any())
      ->method('id')
      ->willReturn((string) $contact_b_id);

    // Create mock contacts field.
    $contacts_field = $this->createMock('Drupal\Core\Field\EntityReferenceFieldItemListInterface');
    $contacts_field->expects($this->any())
      ->method('referencedEntities')
      ->willReturn([$contact_a, $contact_b]);

    // Create mock relationship type.
    $relationship_type = $this->createMock('Drupal\crm\CrmRelationshipTypeInterface');
    $relationship_type->expects($this->any())
      ->method('id')
      ->willReturn($relationship_type_id);
    $relationship_type->expects($this->any())
      ->method('label')
      ->willReturn('Test Type');
    $relationship_type->expects($this->any())
      ->method('getLimitA')
      ->willReturn($limit_a);
    $relationship_type->expects($this->any())
      ->method('getLimitB')
      ->willReturn($limit_b);
    $relationship_type->expects($this->any())
      ->method('isLimitActiveOnly')
      ->willReturn($limit_active_only);
    $relationship_type->expects($this->any())
      ->method('get')
      ->willReturnCallback(function ($key) {
        if ($key === 'label_a') {
          return 'Contact A Label';
        }
        if ($key === 'label_b') {
          return 'Contact B Label';
        }
        return NULL;
      });

    // Create mock bundle field.
    $bundle_field = new \stdClass();
    $bundle_field->entity = $relationship_type;

    // Create mock entity.
    $entity = $this->createMock('Drupal\crm\CrmRelationshipInterface');
    $entity->expects($this->any())
      ->method('id')
      ->willReturn($entity_id);
    $entity->expects($this->any())
      ->method('get')
      ->willReturnCallback(function ($field_name) use ($contacts_field, $bundle_field) {
        if ($field_name === 'contacts') {
          return $contacts_field;
        }
        if ($field_name === 'bundle') {
          return $bundle_field;
        }
        return NULL;
      });

    return $entity;
  }

  /**
   * Creates a mock relationship entity with null relationship type.
   *
   * @return \Drupal\crm\CrmRelationshipInterface|\PHPUnit\Framework\MockObject\MockObject
   *   The mock entity.
   */
  protected function createRelationshipEntityWithNullType() {
    // Create mock bundle field with null entity.
    $bundle_field = new \stdClass();
    $bundle_field->entity = NULL;

    // Create mock entity.
    $entity = $this->createMock('Drupal\crm\CrmRelationshipInterface');
    $entity->expects($this->any())
      ->method('get')
      ->willReturnCallback(function ($field_name) use ($bundle_field) {
        if ($field_name === 'bundle') {
          return $bundle_field;
        }
        return NULL;
      });

    return $entity;
  }

}
