<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Unit\Plugin\views\field;

use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\crm\CrmRelationshipTypeInterface;
use Drupal\crm\Plugin\views\field\RelationshipStatistics;
use Drupal\Tests\UnitTestCase;
use Drupal\views\Plugin\views\style\StylePluginBase;
use Drupal\views\ResultRow;
use Drupal\views\ViewExecutable;

/**
 * Tests the RelationshipStatistics views field plugin.
 *
 * @group crm
 * @coversDefaultClass \Drupal\crm\Plugin\views\field\RelationshipStatistics
 */
class RelationshipStatisticsTest extends UnitTestCase {

  /**
   * The field plugin under test.
   *
   * @var \Drupal\crm\Plugin\views\field\RelationshipStatistics
   */
  protected RelationshipStatistics $fieldPlugin;

  /**
   * Mock entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Set up the container with string translation.
    $container = new ContainerBuilder();
    $translation = $this->createMock(TranslationInterface::class);
    $translation->method('translateString')
      ->willReturnCallback(function ($string) {
        return $string->getUntranslatedString();
      });
    $container->set('string_translation', $translation);
    \Drupal::setContainer($container);

    // Create mock relationship types.
    $friends_type = $this->createMock(CrmRelationshipTypeInterface::class);
    $friends_type->method('label')->willReturn('Friends');
    $friends_type->method('get')->willReturnCallback(function ($key) {
      return match ($key) {
        'label_a' => 'Friend',
        'label_a_plural' => 'Friends',
        'label_b' => 'Friend',
        'label_b_plural' => 'Friends',
        'asymmetric' => FALSE,
        default => NULL,
      };
    });

    $parent_child_type = $this->createMock(CrmRelationshipTypeInterface::class);
    $parent_child_type->method('label')->willReturn('Parent-Child');
    $parent_child_type->method('get')->willReturnCallback(function ($key) {
      return match ($key) {
        'label_a' => 'Parent',
        'label_a_plural' => 'Parents',
        'label_b' => 'Child',
        'label_b_plural' => 'Children',
        'asymmetric' => TRUE,
        default => NULL,
      };
    });

    $relationshipTypes = [
      'friends' => $friends_type,
      'parent_child' => $parent_child_type,
    ];

    // Create mock entity storage.
    $storage = $this->createMock(EntityStorageInterface::class);
    $storage->method('loadMultiple')
      ->willReturn($relationshipTypes);

    // Create mock entity type manager.
    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->entityTypeManager->method('getStorage')
      ->with('crm_relationship_type')
      ->willReturn($storage);

    $container->set('entity_type.manager', $this->entityTypeManager);

    // Create the field plugin with minimal configuration.
    $configuration = [
      'field' => 'relationship_statistics_count',
      'table' => 'crm_contact__relationship_statistics',
    ];

    $this->fieldPlugin = RelationshipStatistics::create(
      $container,
      $configuration,
      'crm_relationship_statistics',
      ['id' => 'crm_relationship_statistics']
    );

    // Create mock style plugin.
    $style = $this->createMock(StylePluginBase::class);
    $style->method('tokenizeValue')->willReturnCallback(function ($value) {
      return $value;
    });

    // Create mock view and assign it to the plugin.
    $view = $this->createMock(ViewExecutable::class);
    $view->style_plugin = $style;
    $view->method('getStyle')->willReturn($style);

    $this->fieldPlugin->view = $view;
    $this->fieldPlugin->options = [
      'format' => 'label_count',
      'label_position' => 'opposite',
    ];
  }

  /**
   * Tests defineOptions returns expected defaults.
   *
   * @covers ::defineOptions
   */
  public function testDefineOptions(): void {
    $options = $this->invokeMethod($this->fieldPlugin, 'defineOptions');
    $this->assertArrayHasKey('format', $options);
    $this->assertEquals('label_count', $options['format']['default']);
    $this->assertArrayHasKey('label_position', $options);
    $this->assertEquals('opposite', $options['label_position']['default']);
  }

  /**
   * Tests render with symmetric relationship type and label_count format.
   *
   * @covers ::render
   */
  public function testRenderSymmetricLabelCount(): void {
    $this->fieldPlugin->options['format'] = 'label_count';

    $row = new ResultRow();
    $row->relationship_statistics_count = 5;
    $row->relationship_statistics_value = 'friends';

    $result = $this->fieldPlugin->render($row);

    $this->assertStringContainsString('Friends', (string) $result);
    $this->assertStringContainsString('5', (string) $result);
  }

  /**
   * Tests render with asymmetric relationship type position A.
   *
   * With default 'opposite' label position, position 'a' shows label_b.
   *
   * @covers ::render
   */
  public function testRenderAsymmetricPositionA(): void {
    $this->fieldPlugin->options['format'] = 'label_count';
    $this->fieldPlugin->options['label_position'] = 'opposite';

    $row = new ResultRow();
    $row->relationship_statistics_count = 2;
    $row->relationship_statistics_value = 'parent_child:a';

    $result = $this->fieldPlugin->render($row);

    // With 'opposite', position 'a' shows label_b = "Child".
    $this->assertStringContainsString('Child', (string) $result);
    $this->assertStringContainsString('2', (string) $result);
  }

  /**
   * Tests render with asymmetric relationship type position B.
   *
   * With default 'opposite' label position, position 'b' shows label_a.
   *
   * @covers ::render
   */
  public function testRenderAsymmetricPositionB(): void {
    $this->fieldPlugin->options['format'] = 'label_count';
    $this->fieldPlugin->options['label_position'] = 'opposite';

    $row = new ResultRow();
    $row->relationship_statistics_count = 3;
    $row->relationship_statistics_value = 'parent_child:b';

    $result = $this->fieldPlugin->render($row);

    // With 'opposite', position 'b' shows label_a = "Parent".
    $this->assertStringContainsString('Parent', (string) $result);
    $this->assertStringContainsString('3', (string) $result);
  }

  /**
   * Tests render with count_only format.
   *
   * @covers ::render
   */
  public function testRenderCountOnly(): void {
    $this->fieldPlugin->options['format'] = 'count_only';

    $row = new ResultRow();
    $row->relationship_statistics_count = 7;
    $row->relationship_statistics_value = 'friends';

    $result = $this->fieldPlugin->render($row);

    $this->assertEquals('7', (string) $result);
  }

  /**
   * Tests render with label_colon_count format.
   *
   * @covers ::render
   */
  public function testRenderLabelColonCount(): void {
    $this->fieldPlugin->options['format'] = 'label_colon_count';

    $row = new ResultRow();
    $row->relationship_statistics_count = 4;
    $row->relationship_statistics_value = 'friends';

    $result = $this->fieldPlugin->render($row);

    $this->assertStringContainsString('Friends', (string) $result);
    $this->assertStringContainsString('4', (string) $result);
    $this->assertStringContainsString(':', (string) $result);
  }

  /**
   * Tests render with count_label format.
   *
   * @covers ::render
   */
  public function testRenderCountLabel(): void {
    $this->fieldPlugin->options['format'] = 'count_label';

    $row = new ResultRow();
    $row->relationship_statistics_count = 6;
    $row->relationship_statistics_value = 'friends';

    $result = $this->fieldPlugin->render($row);

    $this->assertStringContainsString('Friends', (string) $result);
    $this->assertStringContainsString('6', (string) $result);
  }

  /**
   * Tests render with unknown relationship type.
   *
   * @covers ::render
   */
  public function testRenderUnknownType(): void {
    $this->fieldPlugin->options['format'] = 'label_count';

    $row = new ResultRow();
    $row->relationship_statistics_count = 1;
    $row->relationship_statistics_value = 'unknown_type';

    $result = $this->fieldPlugin->render($row);

    // Unknown type should fall back to the raw key.
    $this->assertStringContainsString('unknown_type', (string) $result);
    $this->assertStringContainsString('1', (string) $result);
  }

  /**
   * Tests render with empty values.
   *
   * @covers ::render
   */
  public function testRenderEmpty(): void {
    $this->fieldPlugin->options['format'] = 'label_count';

    $row = new ResultRow();
    $row->relationship_statistics_count = 0;
    $row->relationship_statistics_value = '';

    $result = $this->fieldPlugin->render($row);

    $this->assertEquals('', (string) $result);
  }

  /**
   * Tests render with label_only format.
   *
   * @covers ::render
   */
  public function testRenderLabelOnly(): void {
    $this->fieldPlugin->options['format'] = 'label_only';

    $row = new ResultRow();
    $row->relationship_statistics_count = 5;
    $row->relationship_statistics_value = 'friends';

    $result = $this->fieldPlugin->render($row);

    // Should only show the label, not the count.
    $this->assertEquals('Friends', (string) $result);
  }

  /**
   * Tests render with 'opposite' label position for asymmetric types.
   *
   * @covers ::render
   */
  public function testRenderOppositeLabelPosition(): void {
    $this->fieldPlugin->options['format'] = 'label_count';
    $this->fieldPlugin->options['label_position'] = 'opposite';

    // Position 'a' should show label_b.
    $row_a = new ResultRow();
    $row_a->relationship_statistics_count = 2;
    $row_a->relationship_statistics_value = 'parent_child:a';

    $result_a = $this->fieldPlugin->render($row_a);
    $this->assertStringContainsString('Child', (string) $result_a);

    // Position 'b' should show label_a.
    $row_b = new ResultRow();
    $row_b->relationship_statistics_count = 3;
    $row_b->relationship_statistics_value = 'parent_child:b';

    $result_b = $this->fieldPlugin->render($row_b);
    $this->assertStringContainsString('Parent', (string) $result_b);
  }

  /**
   * Tests render with 'same' label position for asymmetric types.
   *
   * @covers ::render
   */
  public function testRenderSameLabelPosition(): void {
    $this->fieldPlugin->options['format'] = 'label_count';
    $this->fieldPlugin->options['label_position'] = 'same';

    // Position 'a' should show label_a.
    $row_a = new ResultRow();
    $row_a->relationship_statistics_count = 2;
    $row_a->relationship_statistics_value = 'parent_child:a';

    $result_a = $this->fieldPlugin->render($row_a);
    $this->assertStringContainsString('Parent', (string) $result_a);

    // Position 'b' should show label_b.
    $row_b = new ResultRow();
    $row_b->relationship_statistics_count = 3;
    $row_b->relationship_statistics_value = 'parent_child:b';

    $result_b = $this->fieldPlugin->render($row_b);
    $this->assertStringContainsString('Child', (string) $result_b);
  }

  /**
   * Tests that singular label is used when count is 1.
   *
   * @covers ::render
   */
  public function testRenderSingularLabel(): void {
    $this->fieldPlugin->options['format'] = 'label_count';
    $this->fieldPlugin->options['label_position'] = 'opposite';

    $row = new ResultRow();
    $row->relationship_statistics_count = 1;
    $row->relationship_statistics_value = 'parent_child:a';

    $result = $this->fieldPlugin->render($row);

    // With 'opposite', position 'a' shows label_b.
    // Count is 1, so singular "Child" should be used.
    $this->assertStringContainsString('Child', (string) $result);
    $this->assertStringNotContainsString('Children', (string) $result);
  }

  /**
   * Tests that plural label is used when count is greater than 1.
   *
   * @covers ::render
   */
  public function testRenderPluralLabel(): void {
    $this->fieldPlugin->options['format'] = 'label_count';
    $this->fieldPlugin->options['label_position'] = 'opposite';

    $row = new ResultRow();
    $row->relationship_statistics_count = 3;
    $row->relationship_statistics_value = 'parent_child:a';

    $result = $this->fieldPlugin->render($row);

    // With 'opposite', position 'a' shows label_b_plural.
    // Count is 3, so plural "Children" should be used.
    $this->assertStringContainsString('Children', (string) $result);
  }

  /**
   * Tests that plural label is used when count is 0.
   *
   * @covers ::render
   */
  public function testRenderZeroCountUsesPlural(): void {
    $this->fieldPlugin->options['format'] = 'label_count';
    $this->fieldPlugin->options['label_position'] = 'opposite';

    // Need to use a type_key with a non-empty value to avoid early return.
    $row = new ResultRow();
    $row->relationship_statistics_count = 0;
    $row->relationship_statistics_value = 'parent_child:b';
    // Set a non-null value to prevent early empty return.
    $row->_field_data = [];

    // Override getValue to return non-null so we don't hit early return.
    $this->fieldPlugin->field_alias = 'test_field';
    $row->test_field = 0;

    $result = $this->fieldPlugin->render($row);

    // With 'opposite', position 'b' shows label_a_plural.
    // Count is 0, so plural "Parents" should be used (0 is not 1).
    $this->assertStringContainsString('Parents', (string) $result);
  }

  /**
   * Tests that label_only format uses singular label regardless of count.
   *
   * @covers ::render
   */
  public function testRenderLabelOnlyUsesSingular(): void {
    $this->fieldPlugin->options['format'] = 'label_only';
    $this->fieldPlugin->options['label_position'] = 'opposite';

    $row = new ResultRow();
    $row->relationship_statistics_count = 5;
    $row->relationship_statistics_value = 'parent_child:a';

    $result = $this->fieldPlugin->render($row);

    // With 'label_only', should always use singular label.
    // Position 'a' with 'opposite' shows label_b = "Child".
    $this->assertEquals('Child', (string) $result);
  }

  /**
   * Invokes a protected/private method on an object.
   *
   * @param object $object
   *   The object to invoke the method on.
   * @param string $methodName
   *   The method name to invoke.
   * @param array $parameters
   *   Parameters to pass to the method.
   *
   * @return mixed
   *   The result of the method call.
   */
  protected function invokeMethod($object, string $methodName, array $parameters = []) {
    $reflection = new \ReflectionClass(get_class($object));
    $method = $reflection->getMethod($methodName);
    $method->setAccessible(TRUE);
    return $method->invokeArgs($object, $parameters);
  }

}
